#*****************************************************************************
#*                            CmdWidget.tcl
#*
#* Author: Matthew Ballance
#* Desc:   Implements a command widget window...
#*
#*
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#*
#*
#*****************************************************************************
namespace eval CmdWidget {

    variable configspec {
            {-intr_handler ""}
            {-font         ""}
            {-log_fp       ""}
    }
}

#*********************************************************
#* CmdWidget
#*********************************************************
proc CmdWidget::CmdWidget {path args} {
    variable configspec

    set path [frame $path]

    rename $path ::$path:cmd
    proc ::$path {cmd args} "return \[eval CmdWidget::cmd $path \$cmd \$args\]"

    array set $path {_dummy _dummy}
    upvar #0 $path data

    foreach spec $configspec {
        set data([lindex $spec 0]) [lindex $spec 1]
    }
    eval CmdWidget::Configure $path $args

    set data(his_count) 0
    set data(cmd_running) 0

    set data(w:vscroll) [scrollbar $path.s -command "$path.t yview"]

    set data(w:text)    [text $path.t -relief sunken -bd 2 \
                         -yscrollcommand "$path.s set" \
                         -width 60 -height 13 -setgrid true \
                         -bg white]
    pack $data(w:vscroll) -side right -fill y
    pack $data(w:text)    -side right -expand yes -fill both

    bind $data(w:text) <Return> "CmdWidget::ReturnKey $path ; break"

    bind $data(w:text) <Delete> {
        catch {[%W tag remove sel sel.first promptEnd}
        if {[%W tag nextrange sel 1.0 end] == ""} {
            if {[%W compare insert < promptEnd]} {
                break
            }
        }
    }

    bind $data(w:text) <BackSpace> {
        catch {[%W tag remove sel sel.first promptEnd}
        if {[%W compare insert <= promptEnd]} {
            break
        }
        if {[%W tag nextrange sel 1.0 end] == ""} {
            if {[%W compare insert < promptEnd]} {
                break
            }
        }
    }

    bind $data(w:text) <Up> "CmdWidget::Up $path -1; break"
    bind $data(w:text) <Down> "CmdWidget::Up $path 1; break"
    bind $data(w:text) <Control-c> "CmdWidget::CtrlC $path; break"
    bind $data(w:text) <Control-C> "CmdWidget::CtrlC $path; break"
    bind $data(w:text) <Home> {%W mark set insert promptEnd ; break}

    $data(w:text) configure -foreground blue
    if {0} {
    if {$data(-font) != ""} {
        $data(w:text) tag configure t_bold -font $data(-font) \
                -foreground black
    }
    }
    $data(w:text) tag configure t_bold -foreground black

    $data(w:text) tag configure t_error -foreground    red
    $data(w:text) tag configure t_answer -foreground   black

    set data(cmd) ""

    CmdWidget::prompt $path 
    focus $data(w:text)

    return $path
}

#*********************************************************
#* cmd
#*********************************************************
proc CmdWidget::cmd {path cmd args} {
    upvar #0 $path data

    if {$cmd == "save"} {
        eval CmdWidget::save $path $args
    } elseif {$cmd == "puts"} {
        eval CmdWidget::c_puts $path $args
    } elseif {$cmd == "configure" || $cmd == "config"} {
        eval CmdWidget::Configure $path $args
    } elseif {$cmd == "cget"} {
        error "not implemented yet..."
    } else {
        error "no CmdWidget sub-cmd $cmd"
    }
}

#*********************************************************
#* Configure
#*********************************************************
proc CmdWidget::Configure {path args} {
    upvar #0 $path data

    while {[llength $args] > 0} {
        set arg [lindex $args 0]

        set val [array get data $arg]
        if {$val == ""} {
            error "no option \"$arg\""
        }

        if {[info commands CmdWidget::config$arg] != ""} {
            set data($arg) [CmdWidget::config$arg $path [lindex $args 1]]
        } else {
            set data($arg) [lindex $args 1]
        }

        set args [lrange $args 2 end]
    }
}

#*********************************************************
#* save
#*********************************************************
proc CmdWidget::save {w filename} {
    upvar #0 $w data

    set fp [open $filename "w"]

    puts $fp [$data(w:text) get 1.0 end]

    close $fp
}

#*********************************************************
#* puts
#*********************************************************
proc CmdWidget::c_puts {w str} {
    upvar #0 $w data

    if {[$data(w:text) compare insert != "insert linestart"]} {
        $data(w:text) insert insert \n
    }

    set str_out ""
    foreach tmp $str {
        set str_out "$str_out$tmp"
    }

    if {$data(cmd_running) != 1} {
        $data(w:text) tag remove t_bold {promptEnd linestart} promptEnd
        $data(w:text) delete {promptEnd linestart} insert
        set data(cmd) ""
    }

    $data(w:text) insert insert "# $str_out"

    if {$data(-log_fp) != ""} {
        puts $data(-log_fp) "# $str_out"
        flush $data(-log_fp)
    }

    if {$data(cmd_running) != 1} {
        $data(w:text) insert insert \n
        CmdWidget::prompt $w
    }

    $data(w:text) yview -pickplace insert
}

#*********************************************************
#* CtrlC Handler
#*********************************************************
proc CmdWidget::CtrlC {w} {
    upvar #0 $w data

    if {$data(cmd_running) == 1 && $data(-intr_handler) != ""} {
        $data(-intr_handler)
    } else {
        $data(w:text) insert insert \n
        CmdWidget::prompt $w
        set data(cmd) ""
        $data(w:text) yview -pickplace insert
    }
}

#*********************************************************
#* relev
#*********************************************************
proc CmdWidget::relev {w a} {
    upvar #0 $w data

    set res ""

    if {$a < 0} {
        set h [expr $data(his_count) - 1] 
        if { $h >= 1 } { 
            set res [history event $h] 
            set data(his_count) $h
        } elseif {[history nextid] > 1} {
            set res [history event 1]
            set data(his_count) 1
        }
    } else {
        set h [expr $data(his_count) + 1]
        if {$h < [history nextid]} {
            set res [history event $h]
            set data(his_count) $h
        } else {
            set data(his_count) [history nextid]
            set res ""
        }
    }

    return $res
}

#*********************************************************
#* Up
#*********************************************************
proc CmdWidget::Up {w dir} {
    upvar #0 $w data

    if {[$data(w:text) compare insert < promptEnd]} {
    } else {
        $data(w:text) delete promptEnd end
        $data(w:text) insert insert \
            [string trimleft [CmdWidget::relev $w $dir]] 
    }

    $data(w:text) yview -pickplace insert
}

#*********************************************************
#* prompt
#*********************************************************
proc CmdWidget::prompt {w} {
    upvar #0 $w data

    $data(w:text) mark set promptBegin {insert}
    set id [history nextid]
    $data(w:text) insert insert "ivi ($id) "

    if {$data(-log_fp) != ""} {
        puts -nonewline $data(-log_fp) "ivi ($id) "
        flush $data(-log_fp)
    }

    $data(w:text) mark set promptEnd {insert}
    $data(w:text) mark gravity promptEnd left
    $data(w:text) tag add t_bold {promptEnd linestart} promptEnd
}

#*********************************************************
#* prompt2
#*********************************************************
proc CmdWidget::prompt2 {w} {
    upvar #0 $w data
    $data(w:text) insert insert "=> "

    if {$data(-log_fp) != ""} {
        puts -nonewline $data(-log_fp) "=> "
        flush $data(-log_fp)
    }
    $data(w:text) mark set promptEnd {insert}
    $data(w:text) mark gravity promptEnd left
    $data(w:text) tag add t_bold {promptEnd linestart} promptEnd
}

#*********************************************************
#* ReturnKey
#*********************************************************
proc CmdWidget::ReturnKey {w} {
    upvar #0 $w data

    $data(w:text) mark set insert {end - 1c}
    set data(cmd) "$data(cmd) [$data(w:text) get promptEnd insert]"
    $data(w:text) insert insert \n

    if {$data(-log_fp) != ""} {
        puts -nonewline $data(-log_fp) "$data(cmd)\n"
        flush $data(-log_fp)
    }
 
    if { $data(cmd) != " "} {
        if [info complete $data(cmd)] {
            run_cmd $w $data(cmd)

            set data(cmd) ""
        } else {
            set data(cmd) "$data(cmd)\n"
            CmdWidget::prompt2 $w
        }
        set data(his_count) [history nextid]
    } else {
        set data(cmd) ""
        CmdWidget::prompt $w
    }

    $data(w:text) yview -pickplace insert
}

#*********************************************************
#* run_cmd
#*********************************************************
proc CmdWidget::run_cmd {w cmd} {
    upvar #0 $w data

    set data(cmd_running) 1

    if {[catch {history add "$cmd" exec} res]} {
        $data(w:text) insert insert "ERROR: "
        $data(w:text) tag add t_error {insert linestart} insert
        $data(w:text) insert insert "$res\n"
        if {$data(-log_fp) != ""} {
            puts $data(-log_fp) "ERROR: $res"
            flush $data(-log_fp)
        }
    } elseif {[string compare "" "$res"]} {
        $data(w:text) insert insert "$res"
        $data(w:text) tag add t_answer {insert linestart} insert
        $data(w:text) insert insert "\n"
        if {$data(-log_fp) != ""} {
            puts $data(-log_fp) "$res"
            flush $data(-log_fp)
        }
    }

    set data(cmd_running) 0

    set data(hist_count) [history nextid]
    if {[$data(w:text) compare insert != "insert linestart"]} {
        $data(w:text) insert insert \n
    }

    CmdWidget::prompt $w

    $data(w:text) yview -pickplace insert
}
 
#*********************************************************
#* insert_cmd
#*********************************************************
proc CmdWidget::insert_cmd {w cmd} {
    upvar #0 $w data

    c_puts $w $cmd

    ReturnKey $w 
}


