/*****************************************************************************
 *                             TclVpiAccess.cc
 *
 * Author: Matthew Ballance
 * Desc:   Provides TCL commands for accessing VPI features..
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *****************************************************************************/
#include "types.h"
#include "WidgetManager.h"
#include "DesignDB.h"
#include "ivi_String.h"
#include "TclBitVector.h"
#include "BitVector.h"
#include "ConfigDB.h"
#include <vpi_user.h>


/*****************************************************************
 * TclVpi_Inspect()
 *
 * Implements the 'inspect' command...
 *
 * inspect [options] <list_of_sigs>
 *****************************************************************/
static int TclVpi_Inspect(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        Char            **argv)
{
    DesignDB            *ddb;
    Vector<TreeNode>    *sigs;
    Char                *sigSpec;
    Tcl_Obj             *result;
    vpiHandle            vpiHndl;
    s_vpi_value          value_s;
    TreeNode            *sig;
    Uint32               len;
    const char          *radix_str = 0;
    BitVector           *bv;
    Int32                sig_start_idx=-1, i;
    Uint32               radix = 0;
    Tcl_Obj             *res_vect;
    
    ddb = (DesignDB *)WidgetMgr_GetObjHandle(interp, 
            "DesignDB", "@default@");

    if (!ddb) {
        Tcl_AppendResult(interp, "couldn't locate default DDB", 0);
        return TCL_ERROR;
    }

    if (!ddb->get_sim()) {
        Tcl_AppendResult(interp, "no vpi present", 0);
        return TCL_ERROR;
    }

    /**** Find the options... ****/
    for (i=1; i<argc; i++) {
        if (argv[i][0] == '-') {
            if (sig_start_idx > 0) {
                Tcl_AppendResult(interp, 
                        "option specified in wrong position", 0);
                return TCL_ERROR;
            }
            if (String::equal(argv[i], "-hex")) {
                radix = BitVector::Radix_Hex;
            } else if (String::equal(argv[i], "-dec")) {
                radix = BitVector::Radix_Dec;
            } else if (String::equal(argv[i], "-oct")) {
                radix = BitVector::Radix_Oct;
            } else if (String::equal(argv[i], "-bin")) {
                radix = BitVector::Radix_Bin;
            } else {
                Tcl_AppendResult(interp,
                        "unknown option ", argv[i], 0);
                return TCL_ERROR;
            }
        } else {
            sig_start_idx = i;
            break;
        }
    }

    res_vect = Tcl_NewListObj(0, 0);
    for (i=sig_start_idx; i<argc; i++) {

        /**** Figure out the options...     ****/
        sigSpec = argv[i];

        /**** Find the signals specified... ****/
        sigs = ddb->globElems(NULL, sigSpec, "@signals@", 0);

        if (!sigs->length()) {
            Tcl_AppendResult(interp, sigSpec, 
                    " doesn't match any signals", 0);
            return TCL_ERROR;
        }

        sig = sigs->idx(0);
        vpiHndl = (vpiHandle)sig->findField("vpiHandle")->getVoidVal();

        if (!vpiHndl) {
            Tcl_AppendResult(interp, "no vpiHndl field", 0);
            return TCL_ERROR;
        }

        value_s.format = vpiBinStrVal;
        ddb->get_sim()->vpi_get_value(vpiHndl, &value_s);

        /**** Find the default radix... ****/
        if (!radix) {
            len = strlen(value_s.value.str);
            if (len > 1) {
                radix_str = ConfigDB_GetCurrent("App.DefaultVectorDispRadix");
            } else {
                radix_str = ConfigDB_GetCurrent("App.DefaultScalarDispRadix");
            }

            if (String::equal(radix_str, "hex")) {
                radix = BitVector::Radix_Hex;
            } else if (String::equal(radix_str, "dec")) {
                radix = BitVector::Radix_Dec;
            } else if (String::equal(radix_str, "oct")) {
                radix = BitVector::Radix_Oct;
            } else if (String::equal(radix_str, "bin")) {
                radix = BitVector::Radix_Bin;
            } else {
                Tcl_AppendResult(interp,
                    "unknown option ", argv[i], 0);
                return TCL_ERROR;
            }
        }

        result = Tcl_NewBitVector_BinStr(value_s.value.str, len);

        /**** Now, figure out how to format the return value... ****/
        if (radix) {
            bv = (BitVector *)result->internalRep.otherValuePtr;
            bv->setRadix((bv->getRadix() & ~(BitVector::Radix_Mask))
                | radix | BitVector::Radix_Typed);
        }

        Tcl_ListObjAppendElement(interp, res_vect, result);
    }

    Tcl_SetObjResult(interp, res_vect);
    return TCL_OK;
}


/*****************************************************************
 * TclVpi_Init()
 *****************************************************************/ 
extern "C" int TclVpi_Init(Tcl_Interp *interp);
int TclVpi_Init(Tcl_Interp *interp)
{
    Tcl_CreateCommand(interp, "inspect", 
            (Tcl_CmdProc *)TclVpi_Inspect, 0, 0);
    Tcl_CreateCommand(interp, "ins", 
            (Tcl_CmdProc *)TclVpi_Inspect, 0, 0);

    return TCL_OK;
}


