/****************************************************************************
 *                                GCObj.h
 *
 * Author: Matthew Ballance
 * Desc:   Implements a GC object. Drawing on a pixmap is done through
 *         a GC object...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_GC_OBJ_H
#define INCLUDED_GC_OBJ_H
#include "types.h"
#include <string.h>
#include <tk.h>


struct StippleStruct {
    StippleStruct() : data(0), width(0), height(0), pixmap(0) { ; }
    void operator () (UChar *d, Uint32 w, Uint32 h) {
        if (data) {
            delete [] data;
        }
        Uint32 wp = (w%8)?(w/8+1):(w/8);
        data = new UChar[wp*h];
        memcpy(data, d, wp*h);
        width = w;
        height = h;
    }

    UChar       *data;
    Uint32       width;
    Uint32       height;
    Pixmap       pixmap;
};

class PsGC;
class PixmapObj;
class GCObj {

    friend class PsGC;

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * GCObj()
         ************************************************************/
        GCObj();

        /************************************************************
         * GCObj()
         ************************************************************/
        GCObj(PixmapObj *pixObj);

        /************************************************************
         * operator =
         ************************************************************/
        void operator = (GCObj &obj) {
            fprintf(stderr, "operator = called\n");
        }

        /************************************************************
         * line()
         ************************************************************/
        void line(Uint32 x1, Uint32 y1, Uint32 x2, Uint32 y2);

        /************************************************************
         * rect()
         ************************************************************/
        void rect(Uint32 x, Uint32 y, Uint32 width, Uint32 height);

        /************************************************************
         * fill_rect()
         ************************************************************/
        void fill_rect(Uint32 x, Uint32 y, Uint32 w, Uint32 h);

        /************************************************************
         * ComputeTextLayout()
         ************************************************************/
        Tk_TextLayout ComputeTextLayout(Char *str, Int32 len, 
                Tk_Justify justify, Int32 *width, Int32 *height);

        /************************************************************
         * DrawTextLayout()
         ************************************************************/
        void DrawTextLayout(Tk_TextLayout layout, Uint32 x, Uint32 y);

        /************************************************************
         * FreeTextLayout()
         ************************************************************/
        void FreeTextLayout(Tk_TextLayout layout);

        /************************************************************
         * getGC()
         *
         * Returns the natively-represented GC... 
         ************************************************************/
        virtual GC getGC();

        /************************************************************
         * invalidate()
         ************************************************************/
        virtual void invalidate();

        /************************************************************
         * invalidate()
         ************************************************************/
        static void invalidate(GCObj *objs, Uint32 num_objs);

        /************************************************************
         * getGCValues()
         ************************************************************/
        inline XGCValues *getGCValues() { return &d_GCValues; }

        /************************************************************
         * getValueMask()
         ************************************************************/
        inline Uint32 getValueMask() { return d_ValueMask; }

        /************************************************************
         * setLineStyle()
         ************************************************************/
        void setLineStyle();

        /************************************************************
         * setFgColor()
         ************************************************************/
        void setFgColor(Uint32 fg);

        /************************************************************
         * setBgColor()
         ************************************************************/
        void setBgColor(Uint32 bg);

        /************************************************************
         * setStipple()
         ************************************************************/
        void setStipple(UChar *data, Uint32 width, Uint32 height);

        /************************************************************
         * setFgColor()
         ************************************************************/
        void setFgColor(XColor *fg) { setFgColor(fg->pixel); }

        /************************************************************
         * setBgColor()
         ************************************************************/
        void setBgColor(XColor *bg) { setFgColor(bg->pixel); }

        /************************************************************
         * setBgColor()
         ************************************************************/
        void setBgColor(Tk_3DBorder bg) { 
            setBgColor(Tk_3DBorderColor(bg)->pixel); }

        /************************************************************
         * setFont()
         ************************************************************/
        void setFont(Tk_Font font);

        /************************************************************
         * getFont()
         ************************************************************/
        inline Tk_Font getFont() const { return d_font; }

        /************************************************************
         * setLineWidth()
         ************************************************************/
        void setLineWidth(Uint32 width);

        /************************************************************
         * setLineStyle()
         ************************************************************/
        void setLineStyle(Uint32 style);

        /************************************************************
         * setPixmap()
         ************************************************************/
        void setPixmap(PixmapObj *pixmap);

        /************************************************************
         * setPixmap()
         ************************************************************/
        static void setPixmap(GCObj *objs, Uint32 num_objs, PixmapObj *pm);

        /************************************************************
         * getPixmap()
         ************************************************************/
        PixmapObj *getPixmap() const { return pObj; }

        /************************************************************
         * isAttached()
         ************************************************************/
        inline Uint32 isAttached() const { return d_attached; }

        /************************************************************
         * setAttached()
         ************************************************************/
        inline void setAttached(Uint32 attached) { d_attached = attached; }

        /************************************************************
         * getStippleStruct()
         ************************************************************/
        inline StippleStruct &getStippleStruct() { return stipple; }

    /****************************************************************
     * Public Data
     ****************************************************************/
    public:

    /****************************************************************
     * Private Method
     ****************************************************************/
    private:

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        GC            d_GC;
        XGCValues     d_GCValues;
        Uint32        d_ValueMask;
        Tk_Font       d_font;

        PixmapObj    *pObj;
        Uint32        d_attached;
        StippleStruct stipple;
};

#endif /* INCLUDED_GC_OBJ_H */

