--------------------------------------------------------------------------------
-- Project:     MicroBlaze on TE-XC2Se
-- Name:        TBmemcon.vhd
-- Description: memory controller peripheral, testbench
--------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

use work.OpbPak.all;

entity TBmemcon is
end TBmemcon;

--------------------------------------------------------------------------------
architecture test of TBmemcon is
	component te_mem
		generic (
			C_OPB_AWIDTH : integer := 32;
			C_OPB_DWIDTH : integer := 32;
			C_BASEADDR   : std_logic_vector(0 to 31) := x"0080_0000";
			C_HIGHADDR   : std_logic_vector(0 to 31) := x"00FF_FFFF"
			);
		port (
			-- Global signals
			OPB_Clk      : in  std_logic;
			OPB_Rst      : in  std_logic;
			Interrupt    : out std_logic;
			-- OPB signals
			OPB_ABus     : in  std_logic_vector(0 to C_OPB_AWIDTH-1);
			OPB_BE       : in  std_logic_vector(0 to C_OPB_DWIDTH/8-1);
			OPB_DBus     : in  std_logic_vector(0 to C_OPB_DWIDTH-1);
			OPB_RNW      : in  std_logic;
			OPB_select   : in  std_logic;
			OPB_seqAddr  : in  std_logic;
			MEM_DBus     : out std_logic_vector(0 to C_OPB_DWIDTH-1);
			MEM_errAck   : out std_logic;
			MEM_retry    : out std_logic;
			MEM_toutSup  : out std_logic;
			MEM_xferAck  : out std_logic;
			-- external memory bus
			mem_a        : out   std_logic_vector(21 downto 1); -- mem: address bus
			--			mem_d        : inout std_logic_vector(15 downto 0); -- mem: data bus
			mem_d_i      : in    std_logic_vector(15 downto 0); -- mem: data bus
			mem_d_o      : out   std_logic_vector(15 downto 0); --      ...
			mem_d_t      : out   std_logic;                     --      ...
			mem_wen      : out   std_logic;                     -- mem: we#
			flash_cen    : out   std_logic;                     -- flash: ce#
			flash_oen    : out   std_logic;                     -- flash: oe#
			flash_rdy    : in    std_logic;                     -- flash: ready
			flash_8n     : out   std_logic;                     -- flash: byte#
			ram_csn      : out   std_logic;                     -- ram: cs#
			ram_oen      : out   std_logic;                     -- ram: oe#
			ram_blen     : out   std_logic;                     -- ram: ble#
			ram_bhen     : out   std_logic;                     -- ram: bhe#
			cpld_csn     : out   std_logic;                     -- cpld: cs#
			cpld_rwn     : out   std_logic);                    -- cpld: r/w#
	end component;
	--------------------------------------------------------
	signal OPB_Clk      : std_logic := '1';
	signal OPB_Rst      : std_logic;
	signal Interrupt    : std_logic;
	-- OPB signals
	signal OPB_ABus     : std_logic_vector(0 to 31);
	signal OPB_BE       : std_logic_vector(0 to 3);
	signal OPB_DBus     : std_logic_vector(0 to 31);
	signal OPB_RNW      : std_logic;
	signal OPB_select   : std_logic;
	signal OPB_seqAddr  : std_logic;
	signal Sln_DBus     : std_logic_vector(0 to 31);
	signal Sln_errAck   : std_logic;
	signal Sln_retry    : std_logic;
	signal Sln_toutSup  : std_logic;
	signal Sln_xferAck  : std_logic;
	-- external memory bus
	signal mem_a        : std_logic_vector(21 downto 1);
	signal mem_d        : std_logic_vector(15 downto 0);
	signal mem_d_i      : std_logic_vector(15 downto 0);
	signal mem_d_o      : std_logic_vector(15 downto 0);
	signal mem_d_t      : std_logic;                    
	signal mem_wen      : std_logic;                    
	signal flash_cen    : std_logic;                    
	signal flash_oen    : std_logic;                    
	signal flash_rdy    : std_logic;                    
	signal flash_8n     : std_logic;                    
	signal ram_csn      : std_logic;                    
	signal ram_oen      : std_logic;                    
	signal ram_blen     : std_logic;                    
	signal ram_bhen     : std_logic;                    
	signal cpld_csn     : std_logic;                    
	signal cpld_rwn     : std_logic;                   
	
	--
	constant U8         : std_logic_vector(0 to 7) := "--------";
	constant U16        : std_logic_vector(0 to 15):= "----------------";
	constant U24        : std_logic_vector(0 to 23):= "------------------------";
	signal   comment    : string(1 to 25);
	signal   sim        : std_logic:= '1';
begin
	OPB_Clk<= not(OPB_Clk) and sim after 500 ns / 48;
	OPB_Rst<= '1', '0' after 10 ns;
	
	--------------------------------------------------------
	process
	begin
		comment   <= "                         ";
		
		OPB_select<= '0';
		OPB_Abus  <= (others=> 'X');
		OPB_RNW   <= 'X';
		OPB_BE    <= (others=> 'X');
		OPB_DBus  <= (others=> 'X');
		--OPB_seqAddr,
		
		wait until rising_edge(OPB_Clk);
		
		--------------------------------
		-- check ram
		
		-- 32-bit write
		comment   <= "write ram 32-bit         ";
		OPB_write(x"0080_0000", x"01234567", "1111", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		-- 16-bit write
		comment   <= "write ram 16-bit         ";
		OPB_write(x"0080_0004", x"89AB89AB", "1100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_write(x"0080_0006", x"CDEFCDEF", "0011", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		-- 8-bit write
		comment   <= "write ram 8-bit          ";
		OPB_write(x"0080_0008", x"76767676", "1000", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_write(x"0080_0009", x"54545454", "0100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_write(x"0080_000A", x"32323232", "0010", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_write(x"0080_000B", x"10101010", "0001", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		-- 32-bit read
		comment   <= "read ram 32-bit          ";
		OPB_read(x"0080_0000", x"01234567", "1111", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0004", x"89ABCDEF", "1111", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0008", x"76543210", "1111", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		-- 16-bit read
		comment   <= "read ram 16-bit          ";
		OPB_read(x"0080_0000",       x"0123" & U16, "1100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0002", U16 & x"4567",       "0011", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		OPB_read(x"0080_0004",       x"89AB" & U16, "1100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0006", U16 & x"CDEF",       "0011", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		OPB_read(x"0080_0008",       x"7654" & U16, "1100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_000A", U16 & x"3210",       "0011", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		-- 8-bit read
		comment   <= "read ram 8-bit           ";
		OPB_read(x"0080_0000",       x"01" & U24, "1000", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0001", U8  & x"23" & U16, "0100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0002", U16 & x"45" & U8,  "0010", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0003", U24 & x"67",       "0001", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		OPB_read(x"0080_0004",       x"89" & U24, "1000", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0005", U8  & x"AB" & U16, "0100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0006", U16 & x"CD" & U8,  "0010", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0007", U24 & x"EF",       "0001", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		OPB_read(x"0080_0008",       x"76" & U24, "1000", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_0009", U8  & x"54" & U16, "0100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_000A", U16 & x"32" & U8,  "0010", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"0080_000B", U24 & x"10",       "0001", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		--------------------------------
		-- check flash
		
		-- 16-bit write
		comment   <= "write flash 16-bit       ";
		OPB_write(x"00C0_0004", x"89AB89AB", "1100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_write(x"00C0_0006", x"CDEFCDEF", "0011", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		-- 16-bit read
		comment   <= "read flash 16-bit        ";
		OPB_read(x"00C0_0004",       x"89AB" & U16, "1100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"00C0_0006", U16 & x"CDEF",       "0011", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		--------------------------------
		-- check push buttons, DIP switches, LEDs.
		-- 16-bit write
		comment   <= "write CPLD 16-bit        ";
		OPB_write(x"00E0_0000", x"89898989", "1100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_write(x"00E0_0002", x"ABABABAB", "0011", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_write(x"00E0_0004", x"CDCDCDCD", "1100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		-- 16-bit read
		comment   <= "read CPLD 16-bit         ";
		OPB_read(x"00E0_0000",       x"8900" & U16, "1100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"00E0_0002", U16 & x"AB00",       "0011", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		OPB_read(x"00E0_0004",       x"CD00" & U16, "1100", OPB_Clk, OPB_Rst, Interrupt, OPB_Abus, OPB_BE, OPB_DBus, OPB_RNW, OPB_select, OPB_seqAddr, Sln_DBus, Sln_errAck, Sln_retry, Sln_toutSup, Sln_xferAck);
		
		--------------------------------
		wait for 100 ns;

		report "*** successful end of simulation at " & time'image(now) & " ***";
		--assert false report "*** successful end of simulation at " & time'image(now) & " ***" severity failure;
		sim<= '0';
		wait;
	end process;
	
	--------------------------------------------------------
	-- instantiation of unit under test
	UUT: te_mem generic map (
	C_OPB_AWIDTH => 32,
	C_OPB_DWIDTH => 32,
	C_BASEADDR   => x"0080_0000",
	C_HIGHADDR   => x"00FF_FFFF")
	port map (
		-- Global signals
		OPB_Clk      => OPB_Clk,
		OPB_Rst      => OPB_Rst,
		Interrupt    => Interrupt,
		-- OPB signals
		OPB_ABus     => OPB_ABus,
		OPB_BE       => OPB_BE,
		OPB_DBus     => OPB_DBus,
		OPB_RNW      => OPB_RNW,
		OPB_select   => OPB_select,
		OPB_seqAddr  => OPB_seqAddr,
		MEM_DBus    => Sln_DBus,
		MEM_errAck  => Sln_errAck,
		MEM_retry   => Sln_retry,
		MEM_toutSup => Sln_toutSup,
		MEM_xferAck => Sln_xferAck,
		-- external memory bus
		mem_a       => mem_a,
		--		mem_d       => mem_d,
		mem_d_i     => mem_d_i,
		mem_d_o     => mem_d_o,
		mem_d_t     => mem_d_t,
		mem_wen     => mem_wen,
		flash_cen   => flash_cen,
		flash_oen   => flash_oen,
		flash_rdy   => flash_rdy,
		flash_8n    => flash_8n,
		ram_csn     => ram_csn,
		ram_oen     => ram_oen,
		ram_blen    => ram_blen,
		ram_bhen    => ram_bhen,
		cpld_csn    => cpld_csn,
		cpld_rwn    => cpld_rwn);
	
	
	--------------------------------------------------------
	-- bi-directional data bus
	mem_d<= mem_d_o when mem_d_t= '0'
	else    (others=> 'Z');
	mem_d_i<= mem_d;
	
	--------------------------------------------------------
	-- simple ram model
	ram_block: process(mem_a, mem_d, mem_wen, ram_csn, ram_oen, ram_blen, ram_bhen)
		subtype  wordT is std_logic_vector(15 downto 0);
		type     memT  is array(0 to 7) of wordT;
		variable ram:  memT;
		
		variable aa: integer;
		variable dd: std_logic_vector(15 downto 0);
	begin
		-- address and data setup
		aa:= to_integer(unsigned(mem_a(11 downto 1)));
		
		-- read from memory
		if ram_csn= '0' and ram_oen= '0' then mem_d<= ram(aa);
		else                                  mem_d<= (others=> 'Z'); end if;
		
		-- write to memory
		if ram_csn= '0' and mem_wen= '0' then
			if ram_blen= '0' then ram(aa)(7  downto 0):= mem_d(7  downto 0); end if;
			if ram_bhen= '0' then ram(aa)(15 downto 8):= mem_d(15 downto 8); end if;
		end if;
	end process;
	
	--------------------------------------------------------
	-- simple flash model
	flash_block: process(mem_a, mem_d, mem_wen, flash_cen, flash_oen)
		subtype  wordT  is std_logic_vector(15 downto 0);
		type     memT   is array(0 to 7) of wordT;
		variable flash: memT;
		
		variable aa: integer;
		variable dd: std_logic_vector(15 downto 0);
	begin
		-- address and data setup
		aa:= to_integer(unsigned(mem_a(11 downto 1)));
		
		-- read from memory
		if flash_cen= '0' and flash_oen= '0' then mem_d<= flash(aa);
		else                                      mem_d<= (others=> 'Z'); end if;
		
		-- write to memory
		if flash_cen= '0' and mem_wen= '0' then
			flash(aa):= mem_d;
		end if;
	end process;
	--------------------------------------------------------
	-- simple CPLD model
	cpld_block: process(mem_a, mem_d, cpld_csn, cpld_rwn)
		variable LEDs:     std_logic_vector(7 downto 0);
		variable buttons:  std_logic_vector(7 downto 0);
		variable switches: std_logic_vector(7 downto 0);
	begin
		mem_d<= (others=> 'Z');
		-- read from CPLD
		if cpld_csn= '0' and cpld_rwn= '1' then
			if mem_a(2 downto 1)= "00" then mem_d<= x"00" & buttons;  end if;
			if mem_a(2 downto 1)= "01" then mem_d<= x"00" & switches; end if;
			if mem_a(2 downto 1)= "10" then mem_d<= x"00" & LEDs;     end if;
		end if;
		-- write to CPLD
		if cpld_csn= '0' and cpld_rwn= '0' then
			if mem_a(2 downto 1)= "00" then buttons:=  mem_d(7 downto 0); end if;
			if mem_a(2 downto 1)= "01" then switches:= mem_d(7 downto 0); end if;
			if mem_a(2 downto 1)= "10" then LEDs:=     mem_d(7 downto 0); end if;
		end if;
	end process;
end test;
--------------------------------------------------------------------------------
-- end of file
