/******************************************************************************
    imgSeek ::  C++ database implementation
    ---------------------------------------
    begin                : Fri Jan 17 2003
    email                : nieder|at|mail.ru

    Copyright (C) 2003 Ricardo Niederberger Cabral
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* STL Includes */
#include <map>
#include <queue>
#include <list>
#include <fstream>
#include <iostream>
// NOTE: when running build-ext.sh (auto swig wrappers) this namespace line has to be commented
using namespace std;

/* imgSeek includes */
#include "haar.h"
/* Database */
#include "imgdb.h"
/* Python interface */
#include "imgdb_wrap.cxx"
/* C Includes */
#include <math.h>
#include <stdio.h>

#ifdef ImMagick
  /* ImageMagick includes */
  #include <Magick++.h>
  using namespace Magick;
#else
  /* QT Includes */
  #include <qimage.h>
#endif

void initImgBin() {
  int i,j;
  for (i = 0;i<128;i++) for (j=0;j<128;j++) imgBin[i*128+j] = min(max(i,j),5);
}

void initDbase() {
  /* should be called before adding images */
  printf("Image database initialized.\n");
  initImgBin();
}

void free_sigs() {
  for (sigIterator it = sigs.begin(); it != sigs.end(); it++) {
    free((*it).second->sig1);
    free((*it).second->sig2);
    free((*it).second->sig3);
    free((*it).second->avgl);
    delete (*it).second;
  }
}

void closeDbase() {
  /* should be called before exiting app */
  free_sigs();
  printf("Image database closed.\n");
}

int getImageWidth(long int id) {
  if (!sigs.count(id)) return 0;
  return sigs[id]->width;
}

int getImageHeight(long int id) {
  if (!sigs.count(id)) return 0;
  return sigs[id]->height;
}

int addImage(const long int id, char* filename,char* thname,int doThumb,int ignDim = 0) {
  /* id is a unique image identifier
     filename is the image location
     thname is the thumbnail location for this image
     doThumb should be set to 1 if you want to save the thumbnail on thname
     Images with a dimension smaller than ignDim are ignored
  */
  double* avgl = (double*)malloc(3*sizeof(double));
  int* sig1;
  int* sig2;
  int* sig3;
  int cn = 0;
  double* cdata1 = new_darray(16384);
  double* cdata2 = new_darray(16384);
  double* cdata3 = new_darray(16384);
  int i;

  sig1 = new_iarray(40);
  sig2 = new_iarray(40);
  sig3 = new_iarray(40);

  sigStruct* nsig = new sigStruct();
  nsig->sig1 = sig1;
  nsig->sig2 = sig2;
  nsig->sig3 = sig3;
  nsig->avgl = avgl;
  nsig->id = id;

  if (sigs.count(id)) {     
     printf("ID already in DB: %ld\n",id);   
    delete sigs[id];
    sigs.erase(id);
  }
#ifdef ImMagick
  //#TODO2: speed things up. Reading pixel by pixel is definetely not needed but imageMagick's writePixels didn't work. 
  //(there is some sort of bit packing problem on my part or theirs)
  Image image;
  try {
    image.read(filename);

    nsig->width = image.baseColumns();
    nsig->height = image.baseRows();
    if ((ignDim) && ((nsig->width <= ignDim) || (nsig->height <= ignDim))) return 2;

    if (doThumb) {
      Image im2(image);
      im2.scale("128x128");
      im2.write(thname);
    }
    image.sample("128x128!");   // force 128x128 dim
    unsigned char* rchan = (unsigned char*) malloc(128*128*sizeof(unsigned char));
    unsigned char* gchan = (unsigned char*) malloc(128*128*sizeof(unsigned char));
    unsigned char* bchan = (unsigned char*) malloc(128*128*sizeof(unsigned char));
    Pixels view(image);         // using the Pixels class because Image->getConstPixels didn't work. It would simply segfault on idx = 2112 !!!
    PixelPacket *pixel_cache = view.get(0,0,128,128);

    for (int idx = 0;idx<16384;idx++) {
        rchan[idx] = pixel_cache->red;
        gchan[idx] = pixel_cache->green;
        bchan[idx] = pixel_cache->blue;
        pixel_cache++;
    }
    transformChar(rchan,gchan,bchan,cdata1,cdata2,cdata3);

    free(rchan);
    free(bchan);
    free(gchan);
  }
  catch( Exception &error_ )
    {
      cout << "While adding image, caught exception: " << error_.what() << endl;
      return 0;
    }

#else // use QT
  QImage image = QImage();
  if (!image.load(filename)) {
    return 0;
  }
  if ( (ignDim) && ((image.width()<= ignDim) || (image.height()<= ignDim)) ) return 2;      
  nsig->width = image.width();
  nsig->height = image.height();
  if (doThumb) {
    image.smoothScale(128,128,QImage::ScaleMin).save(thname,"PNG");
  }
  //#TODO2: same problem. It's really stupid to read pixel by pixel. Perhaps place RGB->YIQ on this loop?
  image = image.scale(128,128);
  for(int j = 0; j<128; j++) {
    for( i = 0; i<128; i++) {
      cdata1[cn] = qRed(image.pixel(j,i));
      cdata2[cn] = qGreen(image.pixel(j,i));
      cdata3[cn] = qBlue(image.pixel(j,i));
      cn++;
    }
  }
  transform(cdata1,cdata2,cdata3);
#endif
  sigs[id] = nsig;

  calcHaar(cdata1,cdata2,cdata3,sig1,sig2,sig3,avgl);
  for (i = 0;i<40;i++) {         // populate buckets
    if (sig1[i]>0) imgbuckets[0][0][sig1[i]].push_back(id);
    if (sig1[i]<0) imgbuckets[0][1][-sig1[i]].push_back(id);

    if (sig2[i]>0) imgbuckets[1][0][sig2[i]].push_back(id);
    if (sig2[i]<0) imgbuckets[1][1][-sig2[i]].push_back(id);

    if (sig3[i]>0) imgbuckets[2][0][sig3[i]].push_back(id);
    if (sig3[i]<0) imgbuckets[2][1][-sig3[i]].push_back(id);   
  }

  free(cdata1);
  free(cdata2);
  free(cdata3);

  return 1;
}

int loaddb(char* filename) {
  std::ifstream f(filename, ios::binary);
  if (!f.is_open()) return 0;
  int sz,coef,c,k;
  long int id;

  // read buckets
  for ( c = 0;c<3;c++)  for (int pn=0;pn<2;pn++)
    for (int i = 0;i<16384;i++) {
      f.read ((char*)&(sz), sizeof(int) );
      for ( k = 0;k<sz;k++) {
        f.read ((char*)&(id), sizeof(long int) );
        imgbuckets[c][pn][i].push_back(id); 
      }
    }
  // read sigs
  f.read ((char*)&(sz), sizeof(int) );
  for ( k = 0;k<sz;k++) {
    f.read ((char*)&(id), sizeof(long int) );
    sigs[id] = new sigStruct();
    sigs[id]->id = id;
    sigs[id]->sig1 = new_iarray(40);
    sigs[id]->sig2 = new_iarray(40);
    sigs[id]->sig3 = new_iarray(40);
    sigs[id]->avgl = (double*)malloc(3*sizeof(double));
    // sig
    for ( c = 0;c<40;c++) {
      f.read ((char*)&(coef), sizeof( int) );   
      sigs[id]->sig1[c] = coef;
      f.read ((char*)&(coef), sizeof( int) );   
      sigs[id]->sig2[c] = coef;
      f.read ((char*)&(coef), sizeof( int) );   
      sigs[id]->sig3[c] = coef;    
    }
    // avgl
    for ( c = 0;c<3;c++) {
      f.read ((char*)&(sigs[id]->avgl[c]), sizeof( double) );      
    }
    // dims
    f.read ((char*)&(sigs[id]->width), sizeof( int) );
    f.read ((char*)&(sigs[id]->height), sizeof( int) );
  }
  f.close();
  return 1;
}

int savedb(char* filename) {
/*
  Serialization order:
  for each color {0,1,2}:
      for {positive,negative}:
          for each 128x128 coefficient {0-16384}:
              [int] bucket size (size of list of ids)
              for each id:
                  [long int] image id
  [int] number of images (signatures)
  for each image:
      [long id] image id
      for each sig coef {0-39}:  (the 40 greatest coefs)
          for each color {0,1,2}:
              [int] coef index (signed)
      for each color {0,1,2}:
          [double] average luminance
      [int] image width
      [int] image height

*/
  std::ofstream f(filename, ios::binary);
  if (!f.is_open()) return 0;
  int sz,c;
  long int id;
  // save buckets
  for ( c = 0;c<3;c++)  for (int pn=0;pn<2;pn++)
    for (int i = 0;i<16384;i++) {
      sz = imgbuckets[c][pn][i].size();
      f.write((char*)&(sz), sizeof(int) );
      for (long_listIterator it = imgbuckets[c][pn][i].begin(); it != imgbuckets[c][pn][i].end(); it++) {
        f.write ((char*)&((*it)), sizeof(long int) );
      }
    }
  // save sigs
  sz = sigs.size();
  f.write ((char*)&(sz), sizeof(int) );
  for (sigIterator it = sigs.begin(); it != sigs.end(); it++) {
    id = (*it).first;
    f.write ((char*)&(id), sizeof(long int));
    // sigs 
    for ( c = 0;c<40;c++) {
      f.write ((char*)&((*it).second->sig1[c]), sizeof( int));
      f.write ((char*)&((*it).second->sig2[c]), sizeof( int));
      f.write ((char*)&((*it).second->sig3[c]), sizeof( int));
    }
    // avgl
    for ( c = 0;c<3;c++) 
      f.write ((char*)&((*it).second->avgl[c]), sizeof(double));
    // dimensions
    f.write ((char*)&((*it).second->width), sizeof(int));
    f.write ((char*)&((*it).second->height), sizeof(int));
  }
  f.close();
  return 1;
}

/* sig1,2,3 are int arrays of lenght 40 
   avgl is the average luminance
   numres is the max number of results
   scanned tells which set of weights to use
*/
void queryImgData(int* sig1,int* sig2,int* sig3,double * avgl,int numres,int scanned) {
  int idx,c;
  bool pn;
  int * sig[3] = {sig1,sig2,sig3};
  int nres = numres+1;
  for (sigIterator sit = sigs.begin(); sit!=sigs.end(); sit++) { //#TODO3: do I really need to score every single sig on db?
    (*sit).second->score = 0;
    for (c = 0; c<3; c++) {
      (*sit).second->score += weights[scanned][0][c]*fabs((*sit).second->avgl[c]-avgl[c]);
    }
  }
  for (int b = 0;b<40;b++) {      // for every coef on a sig
    for ( c = 0;c<3;c++) {
      pn = 0;
      if (sig[c][b]>0) {
        pn = 0;
        idx = sig[c][b];
      } else {
        pn = 1;
        idx = -sig[c][b];
      }
      // update the score of every image which has this coef
      for (long_listIterator uit = imgbuckets[c][pn][idx].begin(); uit != imgbuckets[c][pn][idx].end(); uit++) {
        sigs[(*uit)]->score -= weights[scanned][imgBin[idx]][c];
      }
    }
  }
  while(!pqResults.empty()) pqResults.pop(); // make sure pqResults is empty. TODO: any faster way to empty it ? didn't find any on STL refs.
  int cnt = 0;
  for (sigIterator it = sigs.begin(); it != sigs.end(); it++) {
    cnt++;
    pqResults.push(*(*it).second);
    if (cnt>nres) {
      pqResults.pop();
    }     
  }
}

/* sig1,2,3 are int arrays of lenght 40 
   avgl is the average luminance
   thresd is the limit similarity threshold. Only images with score > thresd will be a result
   scanned tells which set of weights to use
   sigs is the source to query on (map of signatures)
   every search result is removed from sigs. (right now this functn is only used by clusterSim)
*/
long_list queryImgDataForThres(sigMap* tsigs, int* sig1,int* sig2,int* sig3,double * avgl,float thresd,int scanned) {
  int idx,c;
  bool pn;
  long_list res;
  int * sig[3] = {sig1,sig2,sig3};

  for (sigIterator sit = (*tsigs).begin(); sit!=(*tsigs).end(); sit++) { // TODO: do I really need to score every single sig on db?
    (*sit).second->score = 0;
    for ( c = 0;c<3;c++)
      (*sit).second->score += weights[scanned][0][c]*fabs((*sit).second->avgl[c]-avgl[c]);
  }
  for (int b = 0; b<40; b++) {      // for every coef on a sig
    for ( c = 0;c<3;c++) {
      pn = 0;
      if (sig[c][b]>0) {
        pn = 0;
        idx = sig[c][b];
      } else {
        pn = 1;
        idx = -sig[c][b];
      }
      // update the score of every image which has this coef
      for (long_listIterator uit = imgbuckets[c][pn][idx].begin(); uit != imgbuckets[c][pn][idx].end(); uit++) {
        if ((*tsigs).count((*uit))) (*tsigs)[(*uit)]->score -= weights[scanned][imgBin[idx]][c]; // this is an ugly line 
      }
    }
  }
  for (sigIterator it = (*tsigs).begin(); it!=(*tsigs).end(); it++) {
    if ((*it).second->score < thresd) {
      res.push_back((*it).second->id);
      (*tsigs).erase((*it).second->id);
    }
  }
  return res;
}

long_list queryImgDataForThresFast(sigMap* tsigs, double * avgl,float thresd,int scanned) {  
  // will only look for average luminance
  long_list res;
  for (sigIterator sit = (*tsigs).begin(); sit!=(*tsigs).end(); sit++) { 
    (*sit).second->score =0;
    for (int c = 0;c<3;c++)  (*sit).second->score += weights[scanned][0][c]*fabs((*sit).second->avgl[c]-avgl[c]);
    //cout << (*sit).second->score << " " << thresd << endl;
    if ((*sit).second->score < thresd) {
      res.push_back((*sit).second->id);
      (*tsigs).erase((*sit).second->id);
    }
  }
  return res;
}

// cluster by similarity. Returns list of list of long ints (img ids)
long_list_2 clusterSim(float thresd,int fast = 0) {
  long_list_2 res;              // will hold a list of lists. ie. a list of clusters
  sigMap wSigs(sigs);           // temporary map of sigs, as soon as an image becomes part of a cluster, it's removed from this map

  for (sigIterator sit = wSigs.begin(); sit!=wSigs.end(); sit++) { // for every img on db
    long_list res2;
    if (fast)
      res2 = queryImgDataForThresFast(&wSigs,(*sit).second->avgl,thresd,1);
    else
      res2 = queryImgDataForThres(&wSigs,(*sit).second->sig1,(*sit).second->sig2,(*sit).second->sig3,(*sit).second->avgl,thresd,1);
    long int hid = (*sit).second->id;
    wSigs.erase(hid);
    if (res2.size()<=1) continue;
    res2.push_front(hid);
    res.push_back(res2);
  }
  return res;
}

////////////////////////////////////////////////////////////////////////////////
// Python Wrappers/Helpers:
// TODO: learn how to properly wrap STL lists and such using SWIG. These helpers functs should work meanwhile...
//////

int getLongListSize(long_list& li) {
  return li.size();
}

long int popLongList(long_list& li) {
  long int a = li.front();
  li.pop_front();
  return a;
}

int getLongList2Size(long_list_2& li) {
  return li.size();
}

long_list popLong2List(long_list_2& li) {
  long_list a = li.front();
  li.pop_front();
  return a;
}

int getNumResults() {
  /*get the number of results the last query yielded
   */
  return pqResults.size();
}

long int getResultID() {
  /*get the id of the current query result, removing it from the result list
    (you should always call getResultID *before* getResultScore)
  */
  curResult = pqResults.top();
  pqResults.pop();
  return curResult.id;
}

double getResultScore() {
  /*get the score for the current query result
   */
  return curResult.score;
}

void queryImgID(long int id,int numres) {
  /*query for images similar to the one that has this id
    numres is the maximum number of results
  */
  while(!pqResults.empty()) pqResults.pop();
  if (!sigs.count(id)) {
    printf("ID not found.\n");
    return;
  }
  queryImgData(sigs[id]->sig1,sigs[id]->sig2,sigs[id]->sig3,sigs[id]->avgl, numres, 0);
}

int  queryImgFile(char* filename,int numres,int scanned) {
  /*query for images similar to the one on filename
    numres is the maximum number of results
    scanned should be 1 if this image is a drawing (TODO: change this name, as it's obviously misleading)
  */
  while(!pqResults.empty()) pqResults.pop();
  double* avgl = (double*)malloc(3*sizeof(double));
  int* sig1;
  int* sig2;
  int* sig3;
  int cn = 0;
  double * cdata1 = new_darray(16384);
  double * cdata2 = new_darray(16384);
  double * cdata3 = new_darray(16384);

  sig1 = new_iarray(40);
  sig2 = new_iarray(40);
  sig3 = new_iarray(40);

#ifdef ImMagick
  Image image;
  try {
    image.read(filename);
    image.sample("128x128!");
    unsigned char* rchan= (unsigned char*) malloc(128*128*sizeof(unsigned char));
    unsigned char* gchan= (unsigned char*) malloc(128*128*sizeof(unsigned char));
    unsigned char* bchan= (unsigned char*) malloc(128*128*sizeof(unsigned char));
    Pixels view(image);
    PixelPacket *pixel_cache = view.get(0,0,128,128);
    int idx = 0;
    for(int i = 0;i<128;i++) for(int j=0;j<128;j++) {
        rchan[idx] = pixel_cache->red;
        gchan[idx] = pixel_cache->green;
        bchan[idx] = pixel_cache->blue;
        pixel_cache++;
        idx++;
    }
    transformChar(rchan,gchan,bchan,cdata1,cdata2,cdata3);

    free(rchan);
    free(bchan);
    free(gchan);
  }
  catch( Exception &error_ ) {
    cout << "While reading image, caught exception: " << error_.what() << endl;
    return 0;
  }

#else // QT
  QImage image = QImage();
  if (!image.load(filename)) return 0;
  if ((image.width() != 128) || (image.height() != 128)) image = image.scale(128,128);

  for(int i = 0;i<128;i++) for(int j = 0;j<128;j++) {
      cdata1[cn] = qRed(image.pixel(j,i));
      cdata2[cn] = qGreen(image.pixel(j,i));
      cdata3[cn] = qBlue(image.pixel(j,i));
      cn++;
  }
  transform(cdata1,cdata2,cdata3);
#endif

  calcHaar(cdata1,cdata2,cdata3,sig1,sig2,sig3,avgl);
  queryImgData(sig1,sig2,sig3,avgl, numres, scanned);

  free(cdata1);
  free(cdata2);
  free(cdata3);

  free(avgl);

  free(sig1);
  free(sig2);
  free(sig3);

  return 1;
}

void removeID(long int id) { 
  /*remove image with this id from dbase
   */
  if (!sigs.count(id)) {         // don't remove something which isn't even on db.
      cout << "Attempt to remove invalid id: " << id << endl;
      return;
  }
  free(sigs[id]->sig1);
  free(sigs[id]->sig2);
  free(sigs[id]->sig3);
  free(sigs[id]->avgl);
  delete sigs[id];
  sigs.erase(id); 
  // remove id from each bucket it could be in
  for (int c = 0;c<3;c++)  for (int pn=0;pn<2;pn++)
    for (int i = 0;i<16384;i++) {
      imgbuckets[c][pn][i].remove(id);     
    }
}

double calcAvglDiff(long int id1,long int id2) {
  /* return the average luminance difference*/
  if (!sigs.count(id1)) return 0;
  if (!sigs.count(id2)) return 0;
  return fabs(sigs[id1]->avgl[0]-sigs[id2]->avgl[0])+fabs(sigs[id1]->avgl[1]-sigs[id2]->avgl[1])+fabs(sigs[id1]->avgl[2]-sigs[id2]->avgl[2]); 
}

double calcDiff(long int id1,long int id2) {
  /* use it to tell the content-based difference between two images
   */
  double diff = calcAvglDiff(id1,id2);
  int * sig1[3] = {sigs[id1]->sig1,sigs[id1]->sig2,sigs[id1]->sig3};
  int * sig2[3] = {sigs[id2]->sig1,sigs[id2]->sig2,sigs[id2]->sig3};
  for (int b = 0;b<40;b++) for (int c=0;c<3;c++) for (int b2 = 0;b2<40;b2++) {
    if (sig2[c][b2] == sig1[c][b]) 
      diff -= weights[0][imgBin[abs(sig1[c][b])]][c];
    }  
  return diff;
}

int resetdb(void) {
  /* call it to reset all buckets and signatures 
   */
  for (int c = 0;c<3;c++)  for (int pn=0;pn<2;pn++) for (int i=0;i<16384;i++) imgbuckets[c][pn][i].clear();
  // delete sigs
  free_sigs();
  sigs.clear();
  return 1;
}

////////////////////////////////////////////////////////////////////////////////
// ImageMagick Misc Functions
//
//////

// this function is just a way to tell Python code using this module whether or not ImageMagick was used when compiling 
int hasImageMagick(void) {
#ifdef ImMagick
  return 1;
#else
  return 0;
#endif 
}

#ifdef ImMagick
int convert(char* f1,char* f2) {
  // load f1 and save f2
  Image image;
  try {
    image.read(f1);
    image.write(f2);
  }
  catch( Exception &error_ )
    {
      cout << "While converting, caught exception: " << error_.what() << endl;
      return 0;
    }
  return 1;
}
int magickThumb(char* f1,char* f2) {
  // write f2 as a thumb for f1
  try {
    Image im2(f1);
    im2.scale("128x128");
    im2.write(f2);
  }
  catch( Exception &error_ )
    {
      cout << "While making thumbnail, caught exception: " << error_.what() << endl;
      return 0;
    }
  return 1;
}
#else
int convert(char* f1,char* f2) {
  return 0;
}
int magickThumb(char* f1,char* f2) {
  return 0;
}
#endif
