/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%                      PPPP   IIIII  X   X  EEEEE  L                          %
%                      P   P    I     X X   E      L                          %
%                      PPPP     I      X    EEE    L                          %
%                      P        I     X X   E      L                          %
%                      P      IIIII  X   X  EEEEE  LLLLL                      %
%                                                                             %
%            IIIII  TTTTT EEEEE  RRRR    AAA   TTTTT   OOO   RRRR             %
%              I      T   E      R   R  A   A    T    O   O  R   R            %
%              I      T   EEE    RRRR   AAAAA    T    O   O  RRRR             %
%              I      T   E      R R    A   A    T    O   O  R R              %
%            IIIII    T   EEEEE  R  R   A   A    T     OOO   R  R             %
%                                                                             %
%                                                                             %
%                   ImageMagick Image Pixel Iterator Methods                  %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                                March 2003                                   %
%                                                                             %
%                                                                             %
%  Copyright 1999-2004 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/cache_view.h"
#include "magick/error.h"
#include "magick/memory_.h"
#include "magick/string_.h"
#include "wand/magick_wand.h"
#include "wand/pixel_iterator.h"
#include "wand/pixel_wand.h"

/*
  Typedef declarations.
*/
struct _PixelIterator
{
  ExceptionInfo
    exception;

  ViewInfo
    *view;

  RectangleInfo
    region;

  long
    y;

  PixelWand
    **pixel_wand;

  unsigned long
    signature;
};

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D e s t r o y P i x e l I t e r a t o r                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyPixelIterator() deallocates resources associated with a PixelIterator.
%
%  The format of the DestroyPixelIterator method is:
%
%      MagickBooleanType DestroyPixelIterator(PixelIterator *iterator)
%
%  A description of each parameter follows:
%
%    o iterator: The pixel iterator.
%
%
*/
WandExport void DestroyPixelIterator(PixelIterator *iterator)
{
  (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(iterator != (const PixelIterator *) NULL);
  assert(iterator->signature == MagickSignature);
  CloseCacheView(iterator->view);
  DestroyPixelWands(iterator->pixel_wand,iterator->region.width);
  DestroyExceptionInfo(&iterator->exception);
  iterator=(PixelIterator *) RelinquishMagickMemory(iterator);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   N e w P i x e l I t e r a t o r                                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  NewPixelIterator() returns a new pixel iterator.
%
%  The format of the NewPixelIterator method is:
%
%      PixelIterator NewPixelIterator(MagickWand *wand)
%
%  A description of each parameter follows:
%
%    o wand: The magick wand.
%
*/
WandExport PixelIterator *NewPixelIterator(MagickWand *wand)
{
  Image
    *image;

  PixelIterator
    *iterator;

  assert(wand != (MagickWand *) NULL);
  image=GetImageFromMagickWand(wand);
  if (image == (Image *) NULL)
    return((PixelIterator *) NULL);
  iterator=(PixelIterator *) AcquireMagickMemory(sizeof(*iterator));
  if (iterator == (PixelIterator *) NULL)
    ThrowMagickFatalException(ResourceLimitFatalError,"MemoryAllocationFailed",
      strerror(errno));
  (void) ResetMagickMemory(iterator,0,sizeof(*iterator));
  GetExceptionInfo(&iterator->exception);
  iterator->view=OpenCacheView(image);
  SetGeometry(image,&iterator->region);
  iterator->pixel_wand=NewPixelWands(iterator->region.width);
  iterator->y=(-1);
  iterator->signature=MagickSignature;
  return(iterator);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   N e w P i x e l R e g i o n I t e r a t o r                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  NewPixelRegionIterator() returns a new pixel iterator.
%
%  The format of the NewPixelRegionIterator method is:
%
%      PixelIterator NewPixelRegionIterator(MagickWand *wand,const long x,
%        const long y,const unsigned long columns,const unsigned long rows,
%        const MagickBooleanType modify)
%
%  A description of each parameter follows:
%
%    o wand: The magick wand.
%
%    o x,y,columns,rows:  These values define the perimeter of a region of
%      pixels.
%
*/
WandExport PixelIterator *NewPixelRegionIterator(MagickWand *wand,const long x,
  const long y,const unsigned long columns,const unsigned long rows)
{
  Image
    *image;

  PixelIterator
    *iterator;

  assert(wand != (MagickWand *) NULL);
  image=GetImageFromMagickWand(wand);
  if (image == (Image *) NULL)
    return((PixelIterator *) NULL);
  iterator=(PixelIterator *) AcquireMagickMemory(sizeof(*iterator));
  if (iterator == (PixelIterator *) NULL)
    ThrowMagickFatalException(ResourceLimitFatalError,"MemoryAllocationFailed",
      strerror(errno));
  (void) ResetMagickMemory(iterator,0,sizeof(*iterator));
  GetExceptionInfo(&iterator->exception);
  iterator->view=OpenCacheView(image);
  SetGeometry(image,&iterator->region);
  iterator->region.x=x;
  iterator->region.y=y;
  iterator->region.width=columns;
  iterator->region.height=rows;
  iterator->pixel_wand=NewPixelWands(iterator->region.width);
  iterator->y=(-1);
  iterator->signature=MagickSignature;
  return(iterator);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P i x e l I t e r a t o r G e t E x c e p t i o n                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  PixelIteratorGetException() returns the severity, reason, and description of
%  any error that occurs when using other methods in this API.
%
%  The format of the PixelIteratorGetException method is:
%
%      char *PixelIteratorGetException(const Pixeliterator *iterator,
%        ExceptionType *severity)
%
%  A description of each parameter follows:
%
%    o iterator: The pixel iterator.
%
%    o severity: The severity of the error is returned here.
%
*/
WandExport char *PixelIteratorGetException(const PixelIterator *iterator,
  ExceptionType *severity)
{
  char
    *description;

  (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(iterator != (const PixelIterator *) NULL);
  assert(iterator->signature == MagickSignature);
  (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(severity != (ExceptionType *) NULL);
  *severity=iterator->exception.severity;
  description=(char *) AcquireMagickMemory(2*MaxTextExtent);
  if (description == (char *) NULL)
    ThrowMagickFatalException(ResourceLimitFatalError,"MemoryAllocationFailed",
      strerror(errno));
  *description='\0';
  if (iterator->exception.reason != (char *) NULL)
    (void) CopyMagickString(description,GetLocaleExceptionMessage(
      iterator->exception.severity,iterator->exception.reason),MaxTextExtent);
  if (iterator->exception.description != (char *) NULL)
    {
      (void) ConcatenateMagickString(description," (",MaxTextExtent);
      (void) ConcatenateMagickString(description,GetLocaleExceptionMessage(
        iterator->exception.severity,iterator->exception.description),
        MaxTextExtent);
      (void) ConcatenateMagickString(description,")",MaxTextExtent);
    }
  return(description);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P i x e l G e t N e x t R o w                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  PixelGetNextRow() returns the next row from the pixel iterator.
%
%  The format of the PixelGetNextRow method is:
%
%      PixelWand *PixelGetNextRow(PixelIterator *iterator)
%
%  A description of each parameter follows:
%
%    o iterator: The pixel iterator.
%
*/
WandExport PixelWand **PixelGetNextRow(PixelIterator *iterator)
{
  IndexPacket
    *indexes;

  register const PixelPacket
    *p;

  register long
    x;

  assert(iterator != (PixelIterator *) NULL);
  (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(iterator->signature == MagickSignature);
  iterator->y++;
  if (PixelSetIteratorRow(iterator,iterator->y) == MagickFalse)
    return((PixelWand **) NULL);
  p=AcquireCacheView(iterator->view,iterator->region.x,iterator->region.y+
    iterator->y,iterator->region.width,1,&iterator->exception);
  if (p == (const PixelPacket *) NULL)
    return((PixelWand **) NULL);
  indexes=GetCacheViewIndexes(iterator->view);
  for (x=0; x < (long) iterator->region.width; x++)
  {
    PixelSetQuantumColor(iterator->pixel_wand[x],p);
    if (iterator->view->image->colorspace == CMYKColorspace)
      PixelSetBlackQuantum(iterator->pixel_wand[x],indexes[x]);
    else
      if (iterator->view->image->storage_class == PseudoClass)
        PixelSetIndex(iterator->pixel_wand[x],indexes[x]);
    p++;
  }
  return(iterator->pixel_wand);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P i x e l R e s e t I t e r a t o r                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  PixelResetIterator() resets the pixel iterator.  Use it in conjunction
%  with PixelGetNextPixel() to iterate over all the pixels in a pixel
%  container.
%
%  The format of the PixelResetIterator method is:
%
%      void PixelResetIterator(PixelIterator *iterator)
%
%  A description of each parameter follows:
%
%    o iterator: The pixel iterator.
%
*/
WandExport void PixelResetIterator(PixelIterator *iterator)
{
  assert(iterator != (PixelIterator *) NULL);
  (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(iterator->signature == MagickSignature);
  iterator->y=(-1);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P i x e l S e t I t e r a t o r R o w                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  PixelSetIteratorRow() set the pixel iterator row.
%
%  The format of the PixelSetIteratorRow method is:
%
%      MagickBooleanType PixelSetIteratorRow(PixelIterator *iterator,
%        const long row)
%
%  A description of each parameter follows:
%
%    o iterator: The pixel iterator.
%
%
*/
WandExport MagickBooleanType PixelSetIteratorRow(PixelIterator *iterator,
  const long row)
{
  (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(iterator != (const PixelIterator *) NULL);
  assert(iterator->signature == MagickSignature);
  if ((row < iterator->region.y) || (row >= (long) iterator->region.height))
    return(MagickFalse);
  iterator->y=row;
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P i x e l S y n c I t e r a t o r                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  PixelSyncIterator() syncs the pixel iterator.
%
%  The format of the PixelSyncIterator method is:
%
%      MagickBooleanType PixelSyncIterator(PixelIterator *iterator)
%
%  A description of each parameter follows:
%
%    o iterator: The pixel iterator.
%
%
*/
WandExport MagickBooleanType PixelSyncIterator(PixelIterator *iterator)
{
  IndexPacket
    *indexes;

  register long
    x;

  register PixelPacket
    *p;

  (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(iterator != (const PixelIterator *) NULL);
  assert(iterator->signature == MagickSignature);
  p=GetCacheView(iterator->view,iterator->region.x,iterator->region.y+
    iterator->y,iterator->region.width,1);
  if (p == (PixelPacket *) NULL)
    {
      InheritException(&iterator->exception,&iterator->view->image->exception);
      return(MagickFalse);
    }
  indexes=GetCacheViewIndexes(iterator->view);
  for (x=0; x < (long) iterator->region.width; x++)
  {
    PixelGetQuantumColor(iterator->pixel_wand[x],p);
    if (iterator->view->image->colorspace == CMYKColorspace)
      indexes[x]=PixelGetBlackQuantum(iterator->pixel_wand[x]);
    else
      if (iterator->view->image->storage_class == PseudoClass)
        indexes[x]=PixelGetIndex(iterator->pixel_wand[x]);
    p++;
  }
  if (SyncCacheView(iterator->view) == MagickFalse)
    {
      InheritException(&iterator->exception,&iterator->view->image->exception);
      return(MagickFalse);
    }
  return(MagickTrue);
}
