/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: LocalFileMng.cpp,v 1.41 2004/09/06 09:40:37 comix Exp $
 *
 */

#include "LocalFileMng.h"

#include "xml/tinyxml.h"
#include "../config.h"
#include <cstdlib>
#include <dirent.h>
#include <sys/stat.h>

LocalFileMng::LocalFileMng() : Object( "LocalFileMng" )
{
//	infoLog("INIT");
}



LocalFileMng::~LocalFileMng(){
//	infoLog("DESTROY");
}



void LocalFileMng::fileCopy( const string& sOrigFilename, const string& sDestFilename )
{
	infoLog( "[fileCopy] " + sOrigFilename + " --> " + sDestFilename );

	if ( sOrigFilename == sDestFilename ) {
		return;
	}

	FILE *inputFile = fopen( sOrigFilename.c_str(), "rb" );
	if (inputFile == NULL) {
		errorLog( "[fileCopy] Error opening " + sOrigFilename );
		return;
	}

	FILE *outputFile = fopen( sDestFilename.c_str(), "wb" );
	if (outputFile == NULL) {
		errorLog( "[fileCopy] Error opening " + sDestFilename );
		return;
	}

	const int bufferSize = 512;
	char buffer[ bufferSize ];
	while( feof( inputFile ) == 0 ) {
		infoLog( "copio..." );
		size_t read = fread( buffer, sizeof(char), bufferSize, inputFile );
		fwrite( buffer, sizeof(char), read, outputFile );
	}

	fclose( inputFile );
	fclose( outputFile );
	infoLog( "[fileCopy] fine della copia" );
}




vector<string> LocalFileMng::listUserDrumkits() {
	vector<string> list;

	string directory = (Preferences::getInstance())->getUserHome() + "/.hydrogen/data";
	DIR* dirp = opendir( directory.c_str() );
	if( dirp == NULL ) {
		errorLog( "Directory " + string(directory) + " not found" );
	}
	else{
		while( true ) {
			dirent *dirInfo = readdir( dirp );
			if( dirInfo == NULL ){
				break;
			}
			if ( ( dirInfo->d_type != DT_DIR ) && ( dirInfo->d_type != DT_UNKNOWN ) ) {
				warningLog( "listUserDrumkits: file " + directory + "/" + string(dirInfo->d_name) + " is not a directory" );
				continue;
			}

			string filename( dirInfo->d_name );
			if( ( filename == "." ) || ( filename == ".." ) ){
				continue;
			}
			list.push_back( filename );
		}
		closedir( dirp );
	}
	return list;

}




vector<string> LocalFileMng::listSystemDrumkits() {
	vector<string> list;

	string directory = string( DATA_DIR ) + "/drumkits";
	DIR* dirp = opendir( directory.c_str() );
	if( dirp == NULL ) {
		errorLog( "Directory " + string(directory) + " not found" );
	}
	else{
		while( true ) {
			dirent *dirInfo = readdir( dirp );
			if( dirInfo == NULL ){
				break;
			}
			if ( ( dirInfo->d_type != DT_DIR ) && ( dirInfo->d_type != DT_UNKNOWN ) ) {
				warningLog( "listSystemDrumkits: file " + directory + "/" + string(dirInfo->d_name) + " is not a directory" );
				continue;
			}
			string filename( dirInfo->d_name );
			if( ( filename == "." ) || ( filename == ".." ) ){
				continue;
			}
			list.push_back( filename );
		}
		closedir( dirp );
	}
	return list;

}




string LocalFileMng::getDrumkitDirectory( string drumkitName ) {
	// search in system drumkit
	vector<string> systemDrumkits = listSystemDrumkits();
	for ( uint i = 0; i < systemDrumkits.size(); i++ ) {
		if ( systemDrumkits[ i ] == drumkitName ) {
			string path = string( DATA_DIR ) + "/drumkits/";
			return path;
		}
	}

	// search in user drumkit
	vector<string> userDrumkits = listUserDrumkits();
	for ( uint i = 0; i < userDrumkits.size(); i++ ) {
		if ( userDrumkits[ i ] == drumkitName ) {
			string path = (Preferences::getInstance())->getUserHome() + "/.hydrogen/data/";
			return path;
		}
	}

	errorLog( "[getDrumkitDirectory] drumkit \"" + drumkitName + "\" not found" );
	return "";	// FIXME
}



/// Restituisce un oggetto DrumkitInfo.
/// Gli strumenti non hanno dei veri propri sample,
/// viene utilizzato solo il campo filename.
DrumkitInfo* LocalFileMng::loadDrumkit( string directory )
{
	infoLog( "loadDrumkit " + directory );

	// che if the drumkit.xml file exists
	string drumkitInfoFile = directory + "/drumkit.xml";
	std::ifstream verify( drumkitInfoFile.c_str() , std::ios::in | std::ios::binary );
	if (verify == NULL){
		errorLog( "[loadDrumkit] Load Instrument: Data file " + drumkitInfoFile + " not found." );
		return NULL;
	}

	TiXmlDocument doc( drumkitInfoFile.c_str() );
	doc.LoadFile();

	// root element
	TiXmlNode* drumkitNode;	// root element
	if ( !( drumkitNode = doc.FirstChild( "drumkit_info" ) ) ) {
		errorLog( "[loadDrumkit] Error reading drumkit: drumkit_info node not found" );
		return NULL;
	}

	// Name
	string name = readXmlString( this, drumkitNode, "name", "");
	if (name == "") {
		errorLog( "[loadDrumkit] Error reading drumkit: name node not found" );
		return NULL;
	}

	string author = readXmlString( this, drumkitNode, "author", "undefined author", true );
	string info = readXmlString( this, drumkitNode, "info", "defaultInfo", true );

	DrumkitInfo *drumkitInfo = new DrumkitInfo();
	drumkitInfo->setName( name );
	drumkitInfo->setAuthor( author );
	drumkitInfo->setInfo( info );

	InstrumentList *instrumentList = new InstrumentList();

	TiXmlNode* instrumentListNode;
	if ( (instrumentListNode = drumkitNode->FirstChild( "instrumentList" ) ) ) {
		// INSTRUMENT NODE
		int instrumentList_count = 0;
		TiXmlNode* instrumentNode = 0;
		for( instrumentNode = instrumentListNode->FirstChild("instrument"); instrumentNode; instrumentNode = instrumentNode->NextSibling("instrument")) {
			instrumentList_count++;

			string id = readXmlString( this, instrumentNode, "id", "" );
			string name = readXmlString( this, instrumentNode, "name", "" );
			float volume = readXmlFloat( this, instrumentNode, "volume", 1.0f );
			bool isMuted = readXmlBool( this, instrumentNode, "isMuted", false );
			float pan_L = readXmlFloat( this, instrumentNode, "pan_L", 1.0f );
			float pan_R = readXmlFloat( this, instrumentNode, "pan_R", 1.0f );

			Instrument *pInstrument = new Instrument( id, name, "not used", volume );

			// exclude-notes vector
			TiXmlNode* excludeNode = instrumentNode->FirstChild( "exclude" );
			if ( excludeNode ) {
				TiXmlNode* idNode = 0;
				for( idNode = excludeNode->FirstChild( "id"); idNode; idNode = idNode->NextSibling( "id" )) {
					int id = atoi( idNode->FirstChild()->Value() );
					pInstrument->m_excludeVectId.push_back( id );
				}
			}
			else {
//				warningLog( "Error reading drumkit: exclude node not found" );
			}

			// back compatibility code
			TiXmlNode* filenameNode = instrumentNode->FirstChild( "filename" );
			if ( filenameNode ) {
				//warningLog( "Using back compatibility code. filename node found" );
				string sFilename = LocalFileMng::readXmlString( this, instrumentNode, "filename", "" );
				Sample *pSample = new Sample(0, sFilename );
				//Sample *pSample = Sample::load( directory + "/" + sFilename );
				//pSample->setFilename( sFilename );
				InstrumentLayer *pLayer = new InstrumentLayer( pSample );
				pInstrument->setLayer( pLayer, 0 );
			}
			//~ back compatibility code
			else {
				uint nLayer = 0;
				for( TiXmlNode* layerNode = instrumentNode->FirstChild( "layer" ); layerNode; layerNode = layerNode->NextSibling( "layer" ) ) {
					if (nLayer >= MAX_LAYERS) {
						errorLog( "[loadDrumkit] nLayer > MAX_LAYERS" );
						continue;
					}
					string sFilename = LocalFileMng::readXmlString( this, layerNode, "filename", "" );
					float fMin = LocalFileMng::readXmlFloat( this, layerNode, "min", 0.0 );
					float fMax = LocalFileMng::readXmlFloat( this, layerNode, "max", 1.0 );
					float fGain = LocalFileMng::readXmlFloat( this, layerNode, "gain", 1.0 );
					float fPitch = LocalFileMng::readXmlFloat( this, layerNode, "pitch", 0.0, true );

					Sample *pSample = new Sample(0, sFilename );
					//Sample *pSample = Sample::load( directory + "/" + sFilename );
					//pSample->setFilename( sFilename );
					InstrumentLayer *pLayer = new InstrumentLayer( pSample );
					pLayer->setStartVelocity( fMin );
					pLayer->setEndVelocity( fMax );
					pLayer->setGain( fGain );
					pLayer->setPitch( fPitch );
					pInstrument->setLayer( pLayer, nLayer );

					nLayer++;
				}
			}

			pInstrument->setMuted( isMuted );
			pInstrument->setPan_L( pan_L );
			pInstrument->setPan_R( pan_R );
			pInstrument->setDrumkitName( drumkitInfo->getName() );

			instrumentList->add( pInstrument );
		}
		// prepare the exclude vector
		for (uint nInstr = 0; nInstr < instrumentList->getSize(); nInstr++) {
			Instrument* pInstr = instrumentList->get( nInstr );
			for (uint i = 0; i < pInstr->m_excludeVectId.size(); i++) {
				int id = pInstr->m_excludeVectId[ i ];
				Instrument* pExcluded = instrumentList->get( id );
				pInstr->m_excludeVect.push_back( pExcluded );
			}
		}
	}
	else {
		warningLog( "[loadDrumkit] Error reading drumkit: instrumentList node not found" );
	}
	drumkitInfo->setInstrumentList( instrumentList );

	return drumkitInfo;
}



int LocalFileMng::saveDrumkit( DrumkitInfo *info )
{
	infoLog( "\n\n*****\n\n" );
	warningLog( "Saving drumkit " + info->getName() + " not implemented yet");
	info->dump();	// debug

	string sDrumkitDir = ( Preferences::getInstance() )->getUserHome() + string( "/.hydrogen/data/" ) + info->getName();

	// check if the directory exists
	DIR* dirp = opendir( sDrumkitDir.c_str() );
	if ( dirp == NULL ) {
		// create the drumkit directory
		mkdir( sDrumkitDir.c_str(), S_IRWXU );
	}
	else {
//		warningLog( "[saveDrumkit] Cleaning directory " + sDrumkitDir );
//		// clear all the old files in the directory
//		string clearCmd = "rm -f " + sDrumkitDir + "/*";
//		system( clearCmd.c_str() );
	}


	// create the drumkit.xml file
	string sDrumkitXmlFilename = sDrumkitDir + string( "/drumkit.xml" );

	TiXmlDocument doc( sDrumkitXmlFilename.c_str() );

	TiXmlElement rootNode( "drumkit_info" );

	writeXmlString( &rootNode, "name", info->getName() );	// name
	writeXmlString( &rootNode, "author", info->getAuthor() );	// author
	writeXmlString( &rootNode, "info", info->getInfo() );	// info

	TiXmlElement instrumentListNode( "instrumentList" );		// instrument list
	uint nInstrument = info->getInstrumentList()->getSize();
	// INSTRUMENT NODE
	for (uint i = 0; i < nInstrument; i++) {
		Instrument *instr = info->getInstrumentList()->get(i);

		for ( uint nLayer = 0; nLayer < MAX_LAYERS; nLayer++ ) {
			InstrumentLayer *pLayer = instr->getLayer( nLayer );
			if (pLayer) {
				Sample *pSample = pLayer->getSample();
				string sOrigFilename = pSample->getFilename();

				string sDestFilename = sOrigFilename;

				int nPos = sDestFilename.rfind( '/' );
				sDestFilename = sDestFilename.substr( nPos + 1, sDestFilename.size() - nPos - 1 );
				sDestFilename = sDrumkitDir + "/" + sDestFilename;

				fileCopy( sOrigFilename, sDestFilename );
				//string sCmd = "cp \"" + sOrigFilename + "\" \"" + sDestFilename + "\"";
				//infoLog( sCmd );
				//system( sCmd.c_str() );
			}
		}

		TiXmlElement instrumentNode("instrument");

		LocalFileMng::writeXmlString( &instrumentNode, "id", instr->getId() );
		LocalFileMng::writeXmlString( &instrumentNode, "name", instr->getName() );
		LocalFileMng::writeXmlString( &instrumentNode, "volume", toString( instr->getVolume() ) );
		LocalFileMng::writeXmlBool( &instrumentNode, "isMuted", instr->isMuted() );
		LocalFileMng::writeXmlString( &instrumentNode, "pan_L", toString( instr->getPan_L() ) );
		LocalFileMng::writeXmlString( &instrumentNode, "pan_R", toString( instr->getPan_R() ) );

		// exclude vector
		TiXmlElement excludeNode( "exclude" );
		if (instr->m_excludeVectId.size() != 0) {
			for (uint i = 0; i <instr->m_excludeVectId.size(); i++) {
				writeXmlString( &excludeNode, "id" , toString( instr->m_excludeVectId[ i ] ) );
			}
		}
		instrumentNode.InsertEndChild( excludeNode);

		for (uint nLayer = 0; nLayer < MAX_LAYERS; nLayer++) {
			InstrumentLayer *pLayer = instr->getLayer( nLayer );
			if (pLayer == NULL) continue;
			Sample *pSample = pLayer->getSample();

			string sFilename = pSample->getFilename();

			//if (instr->getDrumkitName() != "") {
				// se e' specificato un drumkit, considero solo il nome del file senza il path
				int nPos = sFilename.rfind("/");
				sFilename = sFilename.substr( nPos + 1, sFilename.length() );
			//}

			TiXmlElement layerNode( "layer" );
			LocalFileMng::writeXmlString( &layerNode, "filename", sFilename );
			LocalFileMng::writeXmlString( &layerNode, "min", toString( pLayer->getStartVelocity() ) );
			LocalFileMng::writeXmlString( &layerNode, "max", toString( pLayer->getEndVelocity() ) );
			LocalFileMng::writeXmlString( &layerNode, "gain", toString( pLayer->getGain() ) );
			LocalFileMng::writeXmlString( &layerNode, "pitch", toString( pLayer->getPitch() ) );

			instrumentNode.InsertEndChild( layerNode );
		}

		instrumentListNode.InsertEndChild(instrumentNode);
	}

	rootNode.InsertEndChild(instrumentListNode);

	doc.InsertEndChild( rootNode );
	doc.SaveFile();

	return 0; // ok
}



void LocalFileMng::installDrumkit( string filename ) {
	infoLog( "[installDrumkit] drumkit = " + filename );

	string dataDir = ( Preferences::getInstance() )->getUserHome() + string( "/.hydrogen/data/" );

	// unpack the drumkit
	string cmd = string( "cd " ) + dataDir + string( "; tar xzf \"" ) + filename + string( "\"" );
	infoLog( cmd );
	system( cmd.c_str() );
}



int LocalFileMng::uninstallDrumkit( string drumkitName ) {
	infoLog( "uninstall drumkit " + drumkitName );

	// verificare che non sia un drumkit di sistema

	return 0;	// OK
}



Song* LocalFileMng::loadSong(string filename) {

	Song *song = NULL;

	SongReader reader;
	song = reader.readSong(filename);

	return song;
}



void LocalFileMng::saveSong(Song *song, string filename) {
	SongWriter writer;

	writer.writeSong(song, filename);
}



string LocalFileMng::readXmlString( Object* obj, TiXmlNode* parent, string nodeName, string defaultValue, bool bCanBeEmpty )
{
	TiXmlNode* node;
	if ( ( node = parent->FirstChild( nodeName.c_str() ) ) ) {
		if ( node->FirstChild() ) {
			return node->FirstChild()->Value();
		}
		else {
			if (!bCanBeEmpty) {
				obj->warningLog( "[readXmlString] Using default value in " + nodeName );
			}
			return defaultValue;
		}
	}
	else {
		obj->warningLog( "[readXmlString] Error reading XML node: " + nodeName + " node not found" );
		return defaultValue;
	}
}



float LocalFileMng::readXmlFloat( Object* obj, TiXmlNode* parent, string nodeName, float defaultValue, bool bCanBeEmpty )
{
	TiXmlNode* node;
	if ( ( node = parent->FirstChild( nodeName.c_str() ) ) ) {
		if ( node->FirstChild() ) {
			//float res = atof( node->FirstChild()->Value() );
			float res = stringToFloat( node->FirstChild()->Value() );
			return res;
		}
		else {
			if (!bCanBeEmpty) {
				obj->warningLog( "[readXmlFloat] Using default value in " + nodeName );
			}
			return defaultValue;
		}
	}
	else {
		obj->warningLog( "[readXmlFloat] Error reading XML node: " + nodeName + " node not found" );
		return defaultValue;
	}
}



int LocalFileMng::readXmlInt( Object* obj, TiXmlNode* parent, string nodeName, int defaultValue, bool bCanBeEmpty )
{
	TiXmlNode* node;
	if ( ( node = parent->FirstChild( nodeName.c_str() ) ) ) {
		if ( node->FirstChild() ) {
			return atoi( node->FirstChild()->Value() );
		}
		else {
			if (!bCanBeEmpty) {
				obj->warningLog( "[readXmlInt] Using default value in " + nodeName );
			}
			return defaultValue;
		}
	}
	else {
		obj->warningLog( "[readXmlInt] Error reading XML node: " + nodeName + " node not found" );
		return defaultValue;
	}
}



bool LocalFileMng::readXmlBool( Object* obj, TiXmlNode* parent, string nodeName, bool defaultValue ) {
	TiXmlNode* node;
	if ( ( node = parent->FirstChild( nodeName.c_str() ) ) ) {
		if ( node->FirstChild() ) {
			if ( string( node->FirstChild()->Value() ) == "true" ) {
				return true;
			}
			else {
				return false;
			}
		}
		else {
			obj->warningLog( "[readXmlBool] Using default value in " + nodeName );
			return defaultValue;
		}
	}
	else {
		obj->warningLog( "[readXmlBool] Error reading XML node: " + nodeName + " node not found" );
		return defaultValue;
	}
}



void LocalFileMng::writeXmlString( TiXmlNode *parent, string name, string text ) {
	TiXmlElement versionNode( name );
	TiXmlText versionText( text );
	versionNode.InsertEndChild( versionText );
	parent->InsertEndChild( versionNode );
}



void LocalFileMng::writeXmlBool( TiXmlNode *parent, string name, bool value ) {
	if (value) {
		writeXmlString( parent, name, string("true") );
	}
	else {
		writeXmlString( parent, name, string("false") );
	}
}





//-----------------------------------------------------------------------------
//	Implementation of SongReader class
//-----------------------------------------------------------------------------


SongReader::SongReader() : Object( "SongReader" )
{
//	infoLog("init");
}





SongReader::~SongReader() {
//	infoLog("destroy");
}



/// Read a song.
/// return NULL = error reading song file
Song* SongReader::readSong(string filename) {
	infoLog("[readSong] " + filename);
	Song* song = NULL;

	std::ifstream verify(filename.c_str() , std::ios::in | std::ios::binary);	// song file exists??
	if (verify == NULL) {
		errorLog("[readSong] Song file " + filename + " not found.");
		return NULL;
	}


	TiXmlDocument doc(filename.c_str());
	doc.LoadFile();

	TiXmlNode* songNode;	// root element
	if ( !( songNode = doc.FirstChild("song") ) ) {
		errorLog("[readSong] Error reading song: song node not found");
		return NULL;
	}


	string sVersion = "";	// version
	sVersion = LocalFileMng::readXmlString( this, songNode, "version", sVersion );
	if ( sVersion != VERSION ) {
		warningLog( "[readSong] Trying to load a song created with a different version of hydrogen.");
		warningLog( "[readSong] Song [" + filename + "] saved with version " + sVersion );
	}

	float fBpm = LocalFileMng::readXmlFloat( this, songNode, "bpm", 120 );
	float fVolume = LocalFileMng::readXmlFloat( this, songNode, "volume", 0.5 );
	float fMetronomeVolume = LocalFileMng::readXmlFloat( this, songNode, "metronomeVolume", 0.5 );
	string sName = LocalFileMng::readXmlString( this, songNode, "name", "Untitled Song" );
	string sAuthor = LocalFileMng::readXmlString( this, songNode, "author", "Unknown Author" );
	string sNotes = LocalFileMng::readXmlString( this, songNode, "notes", "..." );
	bool bLoopEnabled = LocalFileMng::readXmlBool( this, songNode, "loopEnabled", false );

	Song::SongMode nMode = Song::PATTERN_MODE;	// Mode (song/pattern)
	string sMode = LocalFileMng::readXmlString( this, songNode, "mode", "pattern" );
	if ( sMode == "song" ) {
		nMode = Song::SONG_MODE;
	}

	bool bHumanizeTimeEnabled = LocalFileMng::readXmlBool( this, songNode, "humanizeTimeEnabled", false );
	float fHumanizeTimeValue = LocalFileMng::readXmlFloat( this, songNode, "humanize_time", 0.0 );
	bool bHumanizeVelocityEnabled = LocalFileMng::readXmlBool( this, songNode, "humanizeVelocityEnabled", false );
	float fHumanizeVelocityValue = LocalFileMng::readXmlFloat( this, songNode, "humanize_velocity", 0.0 );
	bool bSwingEnabled = LocalFileMng::readXmlBool( this, songNode, "swingEnabled", false );
	float fSwingFactor = LocalFileMng::readXmlFloat( this, songNode, "swing_factor", 0.0 );

	song = new Song( "id", sName, sAuthor, fBpm, fVolume );
	song->setMetronomeVolume( fMetronomeVolume );
	song->setNotes( sNotes );
	song->setLoopEnabled( bLoopEnabled );
	song->setMode( nMode );
	song->setHumanizeTimeEnabled( bHumanizeTimeEnabled );
	song->setHumanizeTimeValue( fHumanizeTimeValue );
	song->setHumanizeVelocityEnabled( bHumanizeVelocityEnabled );
	song->setHumanizeVelocityValue( fHumanizeVelocityValue );
	song->setSwingEnabled( bSwingEnabled );
	song->setSwingFactor( fSwingFactor );


	//  Instrument List
	LocalFileMng localFileMng;
	InstrumentList *instrumentList = new InstrumentList();

	TiXmlNode* instrumentListNode;
	if ( (instrumentListNode = songNode->FirstChild("instrumentList") ) ) {
		// INSTRUMENT NODE
		int instrumentList_count = 0;
		TiXmlNode* instrumentNode = 0;
		for( instrumentNode = instrumentListNode->FirstChild("instrument"); instrumentNode; instrumentNode = instrumentNode->NextSibling("instrument")) {
			instrumentList_count++;

			string sId = LocalFileMng::readXmlString( this, instrumentNode, "id", "" );			// instrument id
			string sDrumkit = LocalFileMng::readXmlString( this, instrumentNode, "drumkit", "" );	// drumkit
			string sName = LocalFileMng::readXmlString( this, instrumentNode, "name", "" );		// name
			float fVolume = LocalFileMng::readXmlFloat( this, instrumentNode, "volume", 1.0 );	// volume
			bool bIsMuted = LocalFileMng::readXmlBool( this, instrumentNode, "isMuted", false );	// is muted
			float fPan_L = LocalFileMng::readXmlFloat( this, instrumentNode, "pan_L", fPan_L );	// pan L
			float fPan_R = LocalFileMng::readXmlFloat( this, instrumentNode, "pan_R", fPan_R );	// pan R
			float fFX1Level = LocalFileMng::readXmlFloat( this, instrumentNode, "FX1Level", 0.0 );	// FX level
			float fFX2Level = LocalFileMng::readXmlFloat( this, instrumentNode, "FX2Level", 0.0 );	// FX level
			float fFX3Level = LocalFileMng::readXmlFloat( this, instrumentNode, "FX3Level", 0.0 );	// FX level
			float fFX4Level = LocalFileMng::readXmlFloat( this, instrumentNode, "FX4Level", 0.0 );	// FX level


			// create a new instrument
			Instrument *pInstrument = new Instrument( sId, sName, sAuthor, fVolume, bIsMuted, fPan_L, fPan_R, sDrumkit );
			pInstrument->setFXLevel( 0, fFX1Level );
			pInstrument->setFXLevel( 1, fFX2Level );
			pInstrument->setFXLevel( 2, fFX3Level );
			pInstrument->setFXLevel( 3, fFX4Level );

			string drumkitPath = "";
			if ( ( sDrumkit != "" ) && (sDrumkit != "-" ) ) {
				drumkitPath = localFileMng.getDrumkitDirectory( sDrumkit ) + sDrumkit + "/";
			}

			// back compatibility code ( song version <= 0.9.0 )
			TiXmlNode* filenameNode = instrumentNode->FirstChild( "filename" );
			if ( filenameNode ) {
				warningLog( "[readSong] Using back compatibility code. filename node found" );
				string sFilename = LocalFileMng::readXmlString( this, instrumentNode, "filename", "" );

				if (drumkitPath != "") {
					sFilename = drumkitPath + "/" + sFilename;
				}
				Sample *pSample = Sample::load( sFilename );
				if (pSample == NULL) {
					// nel passaggio tra 0.8.2 e 0.9.0 il drumkit di default e' cambiato.
					// Se fallisce provo a caricare il corrispettivo file in formato flac
					warningLog( "[readSong] Error loading sample: " + sFilename + " not found. Trying to load a flac..." );
					sFilename = sFilename.substr( 0, sFilename.length() - 4 );
					sFilename += ".flac";
					pSample = Sample::load( sFilename );
					if ( pSample ) {
						infoLog( "[readSong] Found FLAC file!" );
					}
				}
				if (pSample == NULL) {
					errorLog( "[readSong] Error loading sample: " + sFilename + " not found" );
					pInstrument->setMuted( true );
				}
				InstrumentLayer *pLayer = new InstrumentLayer( pSample );
				pInstrument->setLayer( pLayer, 0 );
			}
			//~ back compatibility code
			else {
				uint nLayer = 0;
				for( TiXmlNode* layerNode = instrumentNode->FirstChild( "layer" ); layerNode; layerNode = layerNode->NextSibling( "layer" ) ) {
					if (nLayer >= MAX_LAYERS) {
						errorLog( "[readSong] nLayer > MAX_LAYERS" );
						continue;
					}
					string sFilename = LocalFileMng::readXmlString( this, layerNode, "filename", "" );
					float fMin = LocalFileMng::readXmlFloat( this, layerNode, "min", 0.0 );
					float fMax = LocalFileMng::readXmlFloat( this, layerNode, "max", 1.0 );
					float fGain = LocalFileMng::readXmlFloat( this, layerNode, "gain", 1.0 );
					float fPitch = LocalFileMng::readXmlFloat( this, layerNode, "pitch", 0.0, true );

					if (drumkitPath != "") {
						sFilename = drumkitPath + "/" + sFilename;
					}
					Sample *pSample = Sample::load( sFilename );
					if (pSample == NULL) {
						errorLog( "[readSong] Error loading sample: " + sFilename + " not found" );
						pInstrument->setMuted( true );
					}
					InstrumentLayer *pLayer = new InstrumentLayer( pSample );
					pLayer->setStartVelocity( fMin );
					pLayer->setEndVelocity( fMax );
					pLayer->setGain( fGain );
					pLayer->setPitch( fPitch );

					pInstrument->setLayer( pLayer, nLayer );
					nLayer++;
				}
			}

			// exclude-notes vector
			TiXmlNode* excludeNode = instrumentNode->FirstChild( "exclude" );
			if ( excludeNode ) {
				TiXmlNode* idNode = 0;
				for( idNode = excludeNode->FirstChild( "id"); idNode; idNode = idNode->NextSibling( "id" )) {
					int id = atoi( idNode->FirstChild()->Value() );
					pInstrument->m_excludeVectId.push_back( id );
				}
			}
			else {
				warningLog( "[readSong] Error loading song: exclude node not found" );
			}

			instrumentList->add( pInstrument );
		}

		// prepare the exclude vector
		for (uint nInstr = 0; nInstr < instrumentList->getSize(); nInstr++) {
			Instrument* pInstr = instrumentList->get( nInstr );
			if (pInstr) {
				for (uint i = 0; i < pInstr->m_excludeVectId.size(); i++) {
					int id = pInstr->m_excludeVectId[ i ];
					Instrument* pExcluded = instrumentList->get( id );
					pInstr->m_excludeVect.push_back( pExcluded );
				}
			}
			else {
				errorLog( "[readSong] pInstr == NULL" );
			}
		}


		song->setInstrumentList( instrumentList );
		if ( instrumentList_count != MAX_INSTRUMENTS ) {
			warningLog( "[readSong] Instrument number != MAX_INSTRUMENTS. n=" + toString( instrumentList_count ) + ">" + toString(MAX_INSTRUMENTS) );

			// create the missing instruments with empty sample
			uint nInstrMissing = MAX_INSTRUMENTS - instrumentList_count;
			for (uint i = 0; i < nInstrMissing; i++) {
				int instrId = instrumentList_count + i;
				warningLog( "[readSong] Creating empty instrument");

				Instrument *pInstrument = new Instrument( "id", "", "", 0, 0.8 );

//				Instrument *instr = Instrument::load( string(DATA_DIR) + "/emptySample.wav" );
				char idStr[10];
				sprintf(idStr,"%d", instrId);
				pInstrument->setId( idStr );
				pInstrument->setVolume( 0.1 );
				pInstrument->setName( "Empty" );
				pInstrument->setMuted( false );
				pInstrument->setPan_L( 1.0 );
				pInstrument->setPan_R( 1.0 );

				string sFilename = string(DATA_DIR) + "/emptySample.wav";
				Sample *pSample = Sample::load( sFilename );
				if (pSample == NULL) {
					errorLog( "[readSong] Error loading sample: " + sFilename + " not found" );
				}
				InstrumentLayer *pLayer = new InstrumentLayer( pSample );
				pInstrument->setLayer( pLayer, 0 );

				instrumentList->add( pInstrument );
			}
		}
	}
	else {
		errorLog("[readSong] Error reading song: instrumentList node not found");
		delete song;
		return NULL;
	}





	// Pattern list
	TiXmlNode* patterns = songNode->FirstChild("patternList");

	PatternList *patternList = new PatternList();
	int pattern_count = 0;
	TiXmlNode* patternNode = 0;
	for (patternNode = patterns->FirstChild("pattern"); patternNode; patternNode = patternNode->NextSibling( "pattern" )) {
		pattern_count++;
		Pattern *pat = getPattern(patternNode, instrumentList);
		if (pat) {
			patternList->add(pat);
		}
		else {
			errorLog( "[readSong] Error loading pattern" );
			delete patternList;
			delete song;
			return NULL;
		}
	}
	song->setPatternList(patternList);


	// Pattern sequence
	TiXmlNode* patternSequenceNode = songNode->FirstChild("patternSequence");

	vector<PatternList*>* pPatternGroupVector = new vector<PatternList*>;

	// back-compatibility code..
	for (TiXmlNode* pPatternIDNode = patternSequenceNode->FirstChild("patternID"); pPatternIDNode; pPatternIDNode = pPatternIDNode->NextSibling("patternID")) {
		warningLog( "[readSong] Using old patternSequence code for back compatibility" );
		PatternList *patternSequence = new PatternList();
		string patId = pPatternIDNode->FirstChild()->Value();

		Pattern *pat = NULL;
		for (uint i = 0; i < patternList->getSize(); i++) {
			Pattern *tmp = patternList->get(i);
			if (tmp) {
				if (tmp->getName() == patId) {
					pat = tmp;
					break;
				}
			}
		}
		patternSequence->add( pat );

		pPatternGroupVector->push_back( patternSequence );
	}

	for (TiXmlNode* groupNode = patternSequenceNode->FirstChild("group"); groupNode; groupNode = groupNode->NextSibling("group")) {
		PatternList *patternSequence = new PatternList();
		for (TiXmlNode* patternId = groupNode->FirstChild("patternID"); patternId; patternId = patternId->NextSibling("patternID")) {
			string patId = patternId->FirstChild()->Value();

			Pattern *pat = NULL;
			for (uint i = 0; i < patternList->getSize(); i++) {
				Pattern *tmp = patternList->get(i);
				if (tmp) {
					if (tmp->getName() == patId) {
						pat = tmp;
						break;
					}
				}
			}
			patternSequence->add( pat );
		}
		pPatternGroupVector->push_back( patternSequence );
	}

	song->setPatternGroupVector( pPatternGroupVector );


	// LADSPA FX
	TiXmlNode* ladspaNode = songNode->FirstChild( "ladspa" );
	if (ladspaNode) {
		int nFX = 0;
		TiXmlNode* fxNode;
		for (fxNode = ladspaNode->FirstChild("fx"); fxNode; fxNode = fxNode->NextSibling("fx")) {
			string sName = LocalFileMng::readXmlString( this, fxNode, "name", "" );
			string sFilename = LocalFileMng::readXmlString( this, fxNode, "filename", "" );
			bool bEnabled = LocalFileMng::readXmlBool( this, fxNode, "enabled", false );
			float fVolume = LocalFileMng::readXmlFloat( this, fxNode, "volume", 1.0 );

			if (sName != "no plugin" ) {
				// FIXME: il caricamento va fatto fare all'engine, solo lui sa il samplerate esatto
				LadspaFX* pFX = LadspaFX::load( sFilename, sName, 44100 );
				song->setLadspaFX( nFX, pFX );
				if (pFX) {
					pFX->setEnabled( bEnabled );
					pFX->setVolume( fVolume );
					TiXmlNode* inputControlNode;
					for ( inputControlNode = fxNode->FirstChild("inputControlPort"); inputControlNode; inputControlNode = inputControlNode->NextSibling("inputControlPort")) {
						string sName = LocalFileMng::readXmlString( this, inputControlNode, "name", "" );
						float fValue = LocalFileMng::readXmlFloat( this, inputControlNode, "value", 0.0 );

						for (uint nPort = 0; nPort < pFX->inputControlPorts.size(); nPort++) {
							LadspaControlPort *port = pFX->inputControlPorts[ nPort ];
							if ( string(port->sName) == sName) {
								port->fControlValue = fValue;
							}
						}
					}

					TiXmlNode* outputControlNode;
					for ( outputControlNode = fxNode->FirstChild("outputControlPort"); outputControlNode; outputControlNode = outputControlNode->NextSibling("outputControlPort")) {
					}
				}
			}
			nFX++;
		}
	}
	else {
		warningLog( "[readSong()] ladspa node not found" );
	}


	song->setModified(false);
	song->setFilename(filename);

	return song;
}



Pattern* SongReader::getPattern(TiXmlNode* pattern, InstrumentList* instrList){
	Pattern *pat = NULL;

	string sName = "";	// name
	sName = LocalFileMng::readXmlString( this, pattern, "name", sName );

	int nSize = MAX_NOTES;
	nSize = LocalFileMng::readXmlInt( this, pattern, "size", nSize );
	if (nSize > MAX_NOTES) {
		errorLog( "[getPattern] Pattern size > MAX_NOTES" );
		return NULL;
	}


	pat = new Pattern( sName, nSize );

	SequenceList *sequenceList = new SequenceList();

	TiXmlNode* sequenceListNode = pattern->FirstChild("sequenceList");

	int sequence_count = 0;
	TiXmlNode* sequenceNode = 0;
	for (sequenceNode = sequenceListNode->FirstChild("sequence"); sequenceNode; sequenceNode = sequenceNode->NextSibling("sequence")) {
		sequence_count++;
		Sequence *seq = getSequence(sequenceNode, instrList);

		sequenceList->add(seq);
	}
	pat->setSequenceList(sequenceList);
	if ( sequence_count != MAX_INSTRUMENTS ) {
		warningLog( "[getPattern] sequence number != MAX_INSTRUMENTS" );

		uint nMissingSequences = MAX_INSTRUMENTS - sequence_count;
		for (uint i = 0; i < nMissingSequences; i++) {
			warningLog( "[getPattern] Creating empty Sequence");
			Sequence *seq = new Sequence();
			sequenceList->add(seq);
		}
	}
	return pat;
}



Sequence* SongReader::getSequence(TiXmlNode* sequence, InstrumentList* instrList) {
	Sequence* seq = new Sequence();

	TiXmlNode* noteListNode = sequence->FirstChild( "noteList" );

	TiXmlNode* noteNode = 0;
	for (noteNode = noteListNode->FirstChild("note"); noteNode; noteNode = noteNode->NextSibling("note")) {
		Note *note = getNote(noteNode, instrList);
		seq->m_noteList[ note->getPosition() ] = note;
	}

	return seq;
}



Note* SongReader::getNote(TiXmlNode* noteNode, InstrumentList *instrList){
	Note* note = NULL;

	uint nPosition = LocalFileMng::readXmlInt( this, noteNode, "position", 0 );
	float fVelocity = LocalFileMng::readXmlFloat( this, noteNode, "velocity", 0.8 );
	float fPan_L = LocalFileMng::readXmlFloat( this, noteNode, "pan_L", 1.0 );
	float fPan_R = LocalFileMng::readXmlFloat( this, noteNode, "pan_R", 1.0 );
	int nLength = LocalFileMng::readXmlInt( this, noteNode, "length", -1, true );
	float nPitch = LocalFileMng::readXmlFloat( this, noteNode, "pitch", 0.0, true );

	TiXmlNode* instrNode = noteNode->FirstChild("instrument");
	string instrId = instrNode->FirstChild()->Value();

	Instrument *instrRef = NULL;
	// search instrument by ref
	for (uint i = 0; i < instrList->getSize(); i++) {
		Instrument *instr = instrList->get(i);
		if (instrId == instr->getId()) {
			instrRef = instr;
			break;
		}
	}
	if (instrRef == NULL) {
		errorLog("[getNote] instrRef NULL");
	}

	note = new Note( nPosition, fVelocity, fPan_L, fPan_R, nLength, nPitch);
	note->setInstrument(instrRef);

	return note;
}






//-----------------------------------------------------------------------------
//	Implementation of SongWriter class
//-----------------------------------------------------------------------------


SongWriter::SongWriter() : Object( "SongWriter" )
{
//	infoLog("init");
}



SongWriter::~SongWriter() {
//	infoLog("destroy");
}



void SongWriter::writeSong(Song *song, string filename) {
	infoLog( "Saving song " + filename );

	// FIXME: verificare se e' possibile scrivere il file
	// FIXME: verificare che il file non sia gia' esistente
	// FIXME: effettuare copia di backup per il file gia' esistente

	TiXmlDocument doc(filename);

	TiXmlElement songNode("song");

	LocalFileMng::writeXmlString( &songNode, "version", string(VERSION) );
	LocalFileMng::writeXmlString( &songNode, "bpm", toString( song->getBpm() ) );
	LocalFileMng::writeXmlString( &songNode, "volume", toString( song->getVolume() ) );
	LocalFileMng::writeXmlString( &songNode, "metronomeVolume", toString( song->getMetronomeVolume() ) );
	LocalFileMng::writeXmlString( &songNode, "name", song->getName() );
	LocalFileMng::writeXmlString( &songNode, "author", song->getAuthor() );
	LocalFileMng::writeXmlString( &songNode, "notes", song->getNotes() );
	LocalFileMng::writeXmlBool( &songNode, "loopEnabled", song->isLoopEnabled() );

	if (song->getMode() == Song::SONG_MODE ) {
		LocalFileMng::writeXmlString( &songNode, "mode", string("song") );
	}
	else {
		LocalFileMng::writeXmlString( &songNode, "mode", string("pattern") );
	}

	LocalFileMng::writeXmlBool( &songNode, "humanizeTimeEnabled", song->isHumanizeTimeEnabled() );
	LocalFileMng::writeXmlString( &songNode, "humanize_time", toString( song->getHumanizeTimeValue() ) );
	LocalFileMng::writeXmlBool( &songNode, "humanizeVelocityEnabled", song->isHumanizeVelocityEnabled() );
	LocalFileMng::writeXmlString( &songNode, "humanize_velocity", toString( song->getHumanizeVelocityValue() ) );
	LocalFileMng::writeXmlBool( &songNode, "swingEnabled", song->isSwingEnabled() );
	LocalFileMng::writeXmlString( &songNode, "swing_factor", toString( song->getSwingFactor() ) );

	// instrument list
	TiXmlElement instrumentListNode("instrumentList");
	uint nInstrument = song->getInstrumentList()->getSize();

	// INSTRUMENT NODE
	for (uint i = 0; i < nInstrument; i++) {
		Instrument *instr = song->getInstrumentList()->get(i);

		TiXmlElement instrumentNode("instrument");

		LocalFileMng::writeXmlString( &instrumentNode, "id", instr->getId() );
		LocalFileMng::writeXmlString( &instrumentNode, "drumkit", instr->getDrumkitName() );
		LocalFileMng::writeXmlString( &instrumentNode, "name", instr->getName() );
		LocalFileMng::writeXmlString( &instrumentNode, "volume", toString( instr->getVolume() ) );
		LocalFileMng::writeXmlBool( &instrumentNode, "isMuted", instr->isMuted() );
		LocalFileMng::writeXmlString( &instrumentNode, "pan_L", toString( instr->getPan_L() ) );
		LocalFileMng::writeXmlString( &instrumentNode, "pan_R", toString( instr->getPan_R() ) );

		LocalFileMng::writeXmlString( &instrumentNode, "FX1Level", toString( instr->getFXLevel(0) ) );
		LocalFileMng::writeXmlString( &instrumentNode, "FX2Level", toString( instr->getFXLevel(1) ) );
		LocalFileMng::writeXmlString( &instrumentNode, "FX3Level", toString( instr->getFXLevel(2) ) );
		LocalFileMng::writeXmlString( &instrumentNode, "FX4Level", toString( instr->getFXLevel(3) ) );

		TiXmlElement excludeNode( "exclude" );
		if (instr->m_excludeVectId.size() != 0) {
			for (uint i = 0; i <instr->m_excludeVectId.size(); i++) {
				LocalFileMng::writeXmlString( &excludeNode, "id" , toString( instr->m_excludeVectId[ i ] ) );
			}
		}
		instrumentNode.InsertEndChild( excludeNode);

		for (uint nLayer = 0; nLayer < MAX_LAYERS; nLayer++) {
			InstrumentLayer *pLayer = instr->getLayer( nLayer );
			if (pLayer == NULL) continue;
			Sample *pSample = pLayer->getSample();
			if (pSample == NULL) continue;

			string sFilename = pSample->getFilename();

			if (instr->getDrumkitName() != "") {
				// se e' specificato un drumkit, considero solo il nome del file senza il path
				int nPos = sFilename.rfind("/");
				sFilename = sFilename.substr( nPos + 1, sFilename.length() );
			}

			TiXmlElement layerNode( "layer" );
			LocalFileMng::writeXmlString( &layerNode, "filename", sFilename );
			LocalFileMng::writeXmlString( &layerNode, "min", toString( pLayer->getStartVelocity() ) );
			LocalFileMng::writeXmlString( &layerNode, "max", toString( pLayer->getEndVelocity() ) );
			LocalFileMng::writeXmlString( &layerNode, "gain", toString( pLayer->getGain() ) );
			LocalFileMng::writeXmlString( &layerNode, "pitch", toString( pLayer->getPitch() ) );

			instrumentNode.InsertEndChild( layerNode );
		}

		instrumentListNode.InsertEndChild(instrumentNode);
	}
	songNode.InsertEndChild(instrumentListNode);


	// pattern list
	TiXmlElement patternListNode("patternList");

	uint nPatterns = song->getPatternList()->getSize();
	for (uint i = 0; i < nPatterns; i++) {
		Pattern *pat = song->getPatternList()->get(i);

		// pattern
		TiXmlElement patternNode("pattern");
		LocalFileMng::writeXmlString( &patternNode, "name", pat->getName() );
		LocalFileMng::writeXmlString( &patternNode, "size", toString( pat->getSize() ) );


		// sequence list
		TiXmlElement sequenceListNode("sequenceList");

		uint nSequences = pat->getSequenceList()->getSize();
		for (uint j = 0; j < nSequences; j++) {
			Sequence *seq = pat->getSequenceList()->get(j);

			// Sequence
			TiXmlElement sequenceNode("sequence");

			// Note List
			TiXmlElement noteListNode("noteList");

			uint nNotes = MAX_NOTES;
			for (uint y = 0; y < nNotes; y++) {
				Note *note = seq->m_noteList[ y ];
				if (note != NULL) {
					// note
					TiXmlElement noteNode("note");
					LocalFileMng::writeXmlString( &noteNode, "position", toString( note->getPosition() ) );
					LocalFileMng::writeXmlString( &noteNode, "velocity", toString( note->getVelocity() ) );
					LocalFileMng::writeXmlString( &noteNode, "pan_L", toString( note->getPan_L() ) );
					LocalFileMng::writeXmlString( &noteNode, "pan_R", toString( note->getPan_R() ) );
					LocalFileMng::writeXmlString( &noteNode, "pitch", toString( note->getPitch() ) );
					LocalFileMng::writeXmlString( &noteNode, "length", toString( note->getLength() ) );
					LocalFileMng::writeXmlString( &noteNode, "instrument", note->getInstrument()->getId() );
					noteListNode.InsertEndChild(noteNode);
				}
			}
			sequenceNode.InsertEndChild(noteListNode);
			sequenceListNode.InsertEndChild(sequenceNode);
		}
		patternNode.InsertEndChild(sequenceListNode);
		patternListNode.InsertEndChild(patternNode);
	}
	songNode.InsertEndChild(patternListNode);


	// pattern sequence
	TiXmlElement patternSequenceNode( "patternSequence" );

	uint nPatternGroups = (song->getPatternGroupVector())->size();
	for ( uint i = 0; i < nPatternGroups; i++ ) {
		TiXmlElement groupNode( "group" );

		PatternList *pList = (*song->getPatternGroupVector())[i];
		for (uint j = 0; j < pList->getSize(); j++) {
			Pattern *pPattern = pList->get(j);
			LocalFileMng::writeXmlString( &groupNode, "patternID", pPattern->getName() );
		}

		patternSequenceNode.InsertEndChild(groupNode);
	}

	songNode.InsertEndChild( patternSequenceNode );


	// LADSPA FX
	TiXmlElement ladspaFxNode( "ladspa" );

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		TiXmlElement fxNode( "fx" );

		LadspaFX *pFX = song->getLadspaFX(nFX);

		if ( pFX ) {
			LocalFileMng::writeXmlString( &fxNode, "name", pFX->getPluginLabel() );
			LocalFileMng::writeXmlString( &fxNode, "filename", pFX->getLibraryPath() );
			LocalFileMng::writeXmlBool( &fxNode, "enabled", pFX->isEnabled() );
			LocalFileMng::writeXmlString( &fxNode, "volume", toString( pFX->getVolume() ) );
			for (uint nControl = 0; nControl < pFX->inputControlPorts.size(); nControl++) {
				LadspaControlPort *pControlPort = pFX->inputControlPorts[ nControl ];
				TiXmlElement controlPortNode( "inputControlPort" );
				LocalFileMng::writeXmlString( &controlPortNode, "name", pControlPort->sName );
				LocalFileMng::writeXmlString( &controlPortNode, "value", toString( pControlPort->fControlValue ) );
				fxNode.InsertEndChild( controlPortNode );
			}
			for (uint nControl = 0; nControl < pFX->outputControlPorts.size(); nControl++) {
				LadspaControlPort *pControlPort = pFX->inputControlPorts[ nControl ];
				TiXmlElement controlPortNode( "outputControlPort" );
				LocalFileMng::writeXmlString( &controlPortNode, "name", pControlPort->sName );
				LocalFileMng::writeXmlString( &controlPortNode, "value", toString( pControlPort->fControlValue ) );
				fxNode.InsertEndChild( controlPortNode );
			}
		}
		else {
			LocalFileMng::writeXmlString( &fxNode, "name", string("no plugin") );
			LocalFileMng::writeXmlString( &fxNode, "filename", string("-") );
			LocalFileMng::writeXmlBool( &fxNode, "enabled", false );
			LocalFileMng::writeXmlString( &fxNode, "volume", toString( 0.0 ) );
		}

		ladspaFxNode.InsertEndChild( fxNode );
	}

	songNode.InsertEndChild( ladspaFxNode );




	doc.InsertEndChild(songNode);
        doc.SaveFile();

	song->setModified(false);
	song->setFilename(filename);
}



