/***************************************************************************
                          guidedog.cpp  -  description
                             -------------------
    begin                : Thu Sep 27 22:23:16 CEST 2001
    copyright            : (C) 2001 by Simon Edwards
    email                : simon@simonzone.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <stdlib.h>
#include <qhbox.h>
#include <qlabel.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <ktempfile.h>
#include <kregexp.h>
#include <kseparator.h>
#include <kiconloader.h>

#include "guidedog.h"
#include "commandrunner.h"

#define INDENT_SIZE 16

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
AddressValidator::AddressValidator(QWidget *parent,const char *name) : QValidator(parent,name) {
}

///////////////////////////////////////////////////////////////////////////
AddressValidator::~AddressValidator() {
}

///////////////////////////////////////////////////////////////////////////
QValidator::State AddressValidator::validate(QString &input, int &pos) const {
    KRegExp sanity("^[0-9a-zA-Z./-]*$");
    KRegExp domainnametest("^([a-zA-Z][a-zA-Z0-9-]*\\.)+[a-zA-Z][a-zA-Z0-9-]*$");
    KRegExp iptest("^([0-9]+)\\.([0-9]+)\\.([0-9]+)\\.([0-9]+)$");
    KRegExp ipmaskedtest("^([0-9]+)\\.([0-9]+)\\.([0-9]+)\\.([0-9]+)/([0-9]+)$");
    KRegExp ipmasked2test("^([0-9]+)\\.([0-9]+)\\.([0-9]+)\\.([0-9]+)/([0-9]+)\\.([0-9]+)\\.([0-9]+)\\.([0-9]+)$");
    long ipbyte;

    if(input.isNull()) {
        return QValidator::Intermediate;
    }

        // Smoke text
    if(sanity.match((const char *)input)==false) {
        return QValidator::Invalid;
    }
    
    if(input.length()==0) {
        return QValidator::Intermediate;
    }

        // Test against the domainname regexp.
    if(domainnametest.match((const char *)input)) {
        return QValidator::Acceptable;
    }

        // Ok, now lets try the IP address regexp.
    if(iptest.match((const char *)input)==true) {
        ipbyte = atol(iptest.group(1));    // Yep, it returns char *.
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(iptest.group(2));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(iptest.group(3));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(iptest.group(4));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        return QValidator::Acceptable;
    }

        // Ok, now lets try the IP address regexp.
    if(ipmaskedtest.match((const char *)input)==true) {
        ipbyte = atol(ipmaskedtest.group(1));    // Yep, it returns char *.
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmaskedtest.group(2));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmaskedtest.group(3));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmaskedtest.group(4));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmaskedtest.group(5));
        if(ipbyte<0 || ipbyte>32) {
            return QValidator::Intermediate;
        }
        return QValidator::Acceptable;
    }
    if(ipmasked2test.match((const char *)input)==true) {
        ipbyte = atol(ipmasked2test.group(1));    // Yep, it returns char *.
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(2));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(3));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(4));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(5));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(6));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(7));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(8));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        return QValidator::Acceptable;
    }
    return QValidator::Intermediate;
}

///////////////////////////////////////////////////////////////////////////
void AddressValidator::fixup(QString &input) const {
    QString clean;
    QString tmp;
    QString mask;
    uint i;
    int slashcount;
    char c;
    long ipbyte;
    KRegExp snarfnumber("^([0-9]+)");
    uint l;
    int pos;

        // This is real DWIM (Do What I Mean) code.
        // Somehow it is meant to take what the user entered and work out
        // what they meant and then correct the entered string.
        // It's just a bunch of guesses, hunches and heristics.

    if(input.isNull()) {    // Just in case.
        input = "0.0.0.0";
        return;
    }

        // Filter out any bad characters.
    clean = "";
    slashcount = 0;
    for(i=0; i<input.length(); i++) {
        c = input.at(i).latin1();
        if(c=='/') {
            if(slashcount==0) {
                clean.append('/');
                slashcount++;
            }
        } else if((c>='0' && c<='9') || c=='.' || c=='-' || (c>='A' && c<='Z') || (c>='a' && c<='z')) {
            clean.append(c);
        }
    }

    clean.replace(QRegExp("^\\.*"),QString(""));  // No dots at the start please.
    clean.replace(QRegExp("\\.*$"),QString(""));  // No dots at the end please.

        // Remove double dots.
    do {
        l = clean.length();
        clean.replace(QRegExp("\\.\\."),QString("."));
    } while(l!=clean.length());

        // Do we still have a string?
    if(clean.length()==0) {
        input = "0.0.0.0";  // This should not match much.
        return;
    }

        // Look at the first character and take a guess as to
        // what kind of value the user attempted to enter.
    if(clean.at(0).isDigit()) {
            // Ok, we expect some kind of IP address maybe with a netmask.
        clean.replace(QRegExp("[A-Za-z-]"),QString(""));   // Kill any funny chars.

        clean.replace(QRegExp("^\\.*"),QString(""));  // No dots at the start please.
        clean.replace(QRegExp("\\.*$"),QString(""));  // No dots at the end please.

            // Remove double dots.
        do {
            l = clean.length();
            clean.replace(QRegExp("\\.\\."),QString("."));
        } while(l!=clean.length());

        pos = clean.find('/');
        if(pos!=-1) {
            mask = clean.right(clean.length()-pos-1);
            clean = clean.left(pos);
        }

        i = 0;
        tmp = "";
        while(snarfnumber.match(clean) && i!=4) {
            ipbyte = atol(snarfnumber.group(1));
            if(ipbyte>255) {
                ipbyte = 255;
            }
            i++;
            tmp.append(QString::number(ipbyte));
            tmp.append(".");
            clean = clean.right(clean.length()-strlen(snarfnumber.group(1)));
            clean.replace(QRegExp("^[^0-9]*"),QString(""));
        }
        for(;i<4; i++) {
            tmp.append("0.");
        }
        tmp.replace(QRegExp("\\.$"),QString(""));

        if(mask.length()!=0) { // We still have not consumed all the input.
                                // There must be some kind of netmask left.
            if(mask.contains('.')==0) {    // It must be a single number netmask.
                tmp.append("/");
                ipbyte = mask.toLong();
                if(ipbyte>32) {
                    ipbyte = 32;
                }
                tmp.append(QString::number(ipbyte));
            } else {
                    // Expecting a dotted quad netmask.
                tmp.append("/");
                i = 0;
                while(snarfnumber.match(mask) && i!=4) {
                    ipbyte = atol(snarfnumber.group(1));
                    if(ipbyte>255) {
                        ipbyte = 255;
                    }
                    i++;
                    tmp.append(QString::number(ipbyte));
                    tmp.append(".");
                    mask = mask.right(mask.length()-strlen(snarfnumber.group(1)));
                    mask.replace(QRegExp("^[^0-9]*"),QString(""));
                }
                for(;i<4; i++) {
                    tmp.append("0.");
                }
                tmp.replace(QRegExp("\\.$"),QString(""));
            }
        }
        clean = tmp;

    }

    pos = 0;
    if(validate(clean, pos)!=QValidator::Acceptable) {
        input ="0.0.0.0";
    } else {
        input = clean;
    }
    return;
}

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
IPValidator::IPValidator(QWidget *parent,const char *name) : QValidator(parent,name) {
}

///////////////////////////////////////////////////////////////////////////
IPValidator::~IPValidator() {
}

///////////////////////////////////////////////////////////////////////////
QValidator::State IPValidator::validate(QString &input, int &pos) const {
    KRegExp sanity("^[0-9./]*$");
    KRegExp iptest("^([0-9]+)\\.([0-9]+)\\.([0-9]+)\\.([0-9]+)$");
    KRegExp ipmaskedtest("^([0-9]+)\\.([0-9]+)\\.([0-9]+)\\.([0-9]+)/([0-9]+)$");
    KRegExp ipmasked2test("^([0-9]+)\\.([0-9]+)\\.([0-9]+)\\.([0-9]+)/([0-9]+)\\.([0-9]+)\\.([0-9]+)\\.([0-9]+)$");
    long ipbyte;

    if(input.isNull()) {
        return QValidator::Intermediate;
    }

        // Smoke text
    if(sanity.match((const char *)input)==false) {
        return QValidator::Invalid;
    }

    if(input.length()==0) {
        return QValidator::Intermediate;
    }

        // Ok, now lets try the IP address regexp.
    if(iptest.match((const char *)input)==true) {
        ipbyte = atol(iptest.group(1));    // Yep, it returns char *.
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(iptest.group(2));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(iptest.group(3));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(iptest.group(4));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        return QValidator::Acceptable;
    }

        // Ok, now lets try the IP address regexp.
    if(ipmaskedtest.match((const char *)input)==true) {
        ipbyte = atol(ipmaskedtest.group(1));    // Yep, it returns char *.
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmaskedtest.group(2));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmaskedtest.group(3));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmaskedtest.group(4));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmaskedtest.group(5));
        if(ipbyte<0 || ipbyte>32) {
            return QValidator::Intermediate;
        }
        return QValidator::Acceptable;
    }
    if(ipmasked2test.match((const char *)input)==true) {
        ipbyte = atol(ipmasked2test.group(1));    // Yep, it returns char *.
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(2));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(3));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(4));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(5));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(6));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(7));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        ipbyte = atol(ipmasked2test.group(8));
        if(ipbyte<0 || ipbyte>255) {
            return QValidator::Intermediate;
        }
        return QValidator::Acceptable;
    }
    return QValidator::Intermediate;
}

///////////////////////////////////////////////////////////////////////////
void IPValidator::fixup(QString &input) const {
    QString clean;
    QString tmp;
    QString mask;
    uint i;
    int slashcount;
    char c;
    long ipbyte;
    KRegExp snarfnumber("^([0-9]+)");
    uint l;
    int pos;

        // This is real DWIM (Do What I Mean) code.
        // Somehow it is meant to take what the user entered and work out
        // what they meant and then correct the entered string.
        // It's just a bunch of guesses, hunches and heristics.

    if(input.isNull()) {    // Just in case.
        input = "0.0.0.0";
        return;
    }

        // Filter out any bad characters.
    clean = "";
    slashcount = 0;
    for(i=0; i<input.length(); i++) {
        c = input.at(i).latin1();
        if(c=='/') {
            if(slashcount==0) {
                clean.append('/');
                slashcount++;
            }
        } else if((c>='0' && c<='9') || c=='.') {
            clean.append(c);
        }
    }

    clean.replace(QRegExp("^\\.*"),QString(""));  // No dots at the start please.
    clean.replace(QRegExp("\\.*$"),QString(""));  // No dots at the end please.

        // Remove double dots.
    do {
        l = clean.length();
        clean.replace(QRegExp("\\.\\."),QString("."));
    } while(l!=clean.length());

        // Do we still have a string?
    if(clean.length()==0) {
        input = "0.0.0.0";  // This should not match much.
        return;
    }

        // Look at the first character and take a guess as to
        // what kind of value the user attempted to enter.
    if(clean.at(0).isDigit()) {
            // Ok, we expect some kind of IP address maybe with a netmask.
        clean.replace(QRegExp("^\\.*"),QString(""));  // No dots at the start please.
        clean.replace(QRegExp("\\.*$"),QString(""));  // No dots at the end please.

            // Remove double dots.
        do {
            l = clean.length();
            clean.replace(QRegExp("\\.\\."),QString("."));
        } while(l!=clean.length());

        pos = clean.find('/');
        if(pos!=-1) {
            mask = clean.right(clean.length()-pos-1);
            clean = clean.left(pos);
        }

        i = 0;
        tmp = "";
        while(snarfnumber.match(clean) && i!=4) {
            ipbyte = atol(snarfnumber.group(1));
            if(ipbyte>255) {
                ipbyte = 255;
            }
            i++;
            tmp.append(QString::number(ipbyte));
            tmp.append(".");
            clean = clean.right(clean.length()-strlen(snarfnumber.group(1)));
            clean.replace(QRegExp("^[^0-9]*"),QString(""));
        }
        for(;i<4; i++) {
            tmp.append("0.");
        }
        tmp.replace(QRegExp("\\.$"),QString(""));

        if(mask.length()!=0) { // We still have not consumed all the input.
                                // There must be some kind of netmask left.
            if(mask.contains('.')==0) {    // It must be a single number netmask.
                tmp.append("/");
                ipbyte = mask.toLong();
                if(ipbyte>32) {
                    ipbyte = 32;
                }
                tmp.append(QString::number(ipbyte));
            } else {
                    // Expecting a dotted quad netmask.
                tmp.append("/");
                i = 0;
                while(snarfnumber.match(mask) && i!=4) {
                    ipbyte = atol(snarfnumber.group(1));
                    if(ipbyte>255) {
                        ipbyte = 255;
                    }
                    i++;
                    tmp.append(QString::number(ipbyte));
                    tmp.append(".");
                    mask = mask.right(mask.length()-strlen(snarfnumber.group(1)));
                    mask.replace(QRegExp("^[^0-9]*"),QString(""));
                }
                for(;i<4; i++) {
                    tmp.append("0.");
                }
                tmp.replace(QRegExp("\\.$"),QString(""));
            }
        }
        clean = tmp;

    }

    pos = 0;
    if(validate(clean, pos)!=QValidator::Acceptable) {
        input ="0.0.0.0";
    } else {
        input = clean;
    }
    return;
}

///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////
GuidedogApp::GuidedogApp(const QString &caption) : KDialogBase(KJanusWidget::Tabbed,caption,
        KDialogBase::Help|KDialogBase::Ok|KDialogBase::Apply|KDialogBase::Cancel|KDialogBase::User1,
        KDialogBase::Ok) {
    // The real work is done in initialise().
    doc = 0;
    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
GuidedogApp::~GuidedogApp() {
    delete doc;
}

///////////////////////////////////////////////////////////////////////////
bool GuidedogApp::initialise(bool god) {
    QGroupBox *tmpqgroupbox;
    QHBox *tmpqbox,*tmpqbox2;
    QLabel *tmplabel;
    QWidget *tmpwidget;
    KSeparator *ksep;

    waspreviousconfiguration = false;
    systemconfigmodified = false;

    superusermode = god;
    if(superusermode==false) {
        enableButtonOK(false);
        enableButtonApply(false);
    }

    doc = new GuidedogDoc();

    aboutus = new KAboutApplication(this);
    setButtonText(KDialogBase::User1,i18n("About"));

        // Routing page
    routingpage = addVBoxPage(i18n("Routing"));
    routingpage->setSpacing(spacingHint());

    routingcheckbox = new QCheckBox(i18n("Enable Routing"),(QWidget *)routingpage);
    connect(routingcheckbox,SIGNAL(toggled(bool)),this,SLOT(slotRouting(bool)));

    tmpqgroupbox = new QVGroupBox(i18n("IP Masquerade"),(QWidget *)routingpage);
    masqueradecheckbox = new QCheckBox(i18n("Enable IP Masquerade"),(QWidget *)tmpqgroupbox);
    connect(masqueradecheckbox,SIGNAL(toggled(bool)),this,SLOT(slotMasquerade(bool)));

    tmpqbox = new QHBox((QWidget *)tmpqgroupbox);
    tmpwidget = new QWidget(tmpqbox);
    tmpwidget->setMinimumWidth(INDENT_SIZE);
    tmpqbox->setStretchFactor(tmpwidget,0);
    masqueradeftpcheckbox = new QCheckBox(i18n("Masquerade FTP"),(QWidget *)tmpqbox);
    connect(masqueradeftpcheckbox,SIGNAL(toggled(bool)),this,SLOT(slotMasqueradeFTP(bool)));
    tmpqbox->setStretchFactor(masqueradeftpcheckbox,1);
    
    tmpqbox = new QHBox((QWidget *)tmpqgroupbox);
    tmpwidget = new QWidget(tmpqbox);
    tmpwidget->setMinimumWidth(INDENT_SIZE);
    tmpqbox->setStretchFactor(tmpwidget,0);
    masqueradeirccheckbox = new QCheckBox(i18n("Masquerade IRC"),(QWidget *)tmpqbox);
    connect(masqueradeirccheckbox,SIGNAL(toggled(bool)),this,SLOT(slotMasqueradeIRC(bool)));
    tmpqbox->setStretchFactor(masqueradeirccheckbox,1);

        // No Masquerage Addresses
    tmpqbox = new QHBox((QWidget *)tmpqgroupbox);
    tmpqbox->setSpacing(spacingHint());
    tmplabel = new QLabel(i18n("No-masquerade Addresses"),tmpqbox);
    tmpqbox->setStretchFactor(tmplabel,0);
    ksep = new KSeparator(KSeparator::HLine,tmpqbox);
    tmpqbox->setStretchFactor(ksep,1);

    tmplabel = new QLabel(i18n("Network destinations that should contacted without using IP masquerade:"),tmpqgroupbox);
    tmplabel->setAlignment(Qt::AlignLeft|Qt::WordBreak);

    
    nomasqlistbox = new KListBox(tmpqgroupbox);
    nomasqlistbox->setSelectionMode(QListBox::Single);
    connect(nomasqlistbox,SIGNAL(clicked(QListBoxItem *)),this,SLOT(slotNoMasqueradeListBox(QListBoxItem *)));
    
    tmpqbox2 = new QHBox(tmpqgroupbox);
    tmpqbox2->setSpacing(spacingHint());
    newnomasqbutton = new QPushButton(i18n("New Address"),tmpqbox2);
    connect(newnomasqbutton,SIGNAL(clicked()),this,SLOT(slotNewAddressButton()));
    deleteenomasqbutton = new QPushButton(i18n("Delete Address"),tmpqbox2);
    connect(deleteenomasqbutton,SIGNAL(clicked()),this,SLOT(slotDeleteAddressButton()));

    tmpqbox2 = new QHBox(tmpqgroupbox);
    tmpqbox2->setSpacing(spacingHint());
    tmplabel = new QLabel(i18n("Address: "),tmpqbox2);
    tmpqbox2->setStretchFactor(tmplabel,0);
    nomasqlineedit = new SimeLineEdit(tmpqbox2);
    addressvalidator = new AddressValidator(nomasqlineedit);
    nomasqlineedit->setValidator(addressvalidator);

    tmpqbox2->setStretchFactor(nomasqlineedit,1);
    connect(nomasqlineedit,SIGNAL(textChanged(const QString &)),this,SLOT(slotAddressLineEdit(const QString &)));
    connect(nomasqlineedit,SIGNAL(returnPressed()),this,SLOT(slotAddressLineEditReturn()));

    tmplabel = new QLabel(i18n("Addresses can be host names, network names or IP addresses. "
        "Masks can be specified as network masks or a plain number. "
        "e.g. 192.168.1.0/255.255.255.0 or 192.168.1.0/24"),tmpqgroupbox);
    tmplabel->setAlignment(Qt::AlignLeft|Qt::WordBreak);
                                    
        // Forwarding page
    forwardingpage = addVBoxPage(i18n("Forwarding"));
    forwardingpage->setSpacing(spacingHint());

    tmplabel = new QLabel(i18n("Port forwarding rules:"),forwardingpage);
    forwardingpage->setStretchFactor(tmplabel,0);
    forwardlistbox = new KListBox(forwardingpage);
    forwardlistbox->setSelectionMode(QListBox::Single);
    connect(forwardlistbox,SIGNAL(clicked(QListBoxItem *)),this,SLOT(slotForwardListBox(QListBoxItem *)));
    forwardingpage->setStretchFactor(forwardingpage,1);

    tmpqbox2 = new QHBox(forwardingpage);
    tmpqbox2->setSpacing(spacingHint());
    newforwardbutton = new QPushButton(i18n("New Rule"),tmpqbox2);
    connect(newforwardbutton,SIGNAL(clicked()),this,SLOT(slotNewForwardButton()));
    deleteforwardbutton = new QPushButton(i18n("Delete Rule"),tmpqbox2);
    connect(deleteforwardbutton,SIGNAL(clicked()),this,SLOT(slotDeleteForwardButton()));
    
    tmpqbox2 = new QHBox(forwardingpage);
    tmpqbox2->setSpacing(spacingHint());

        // Original Destination box.
    tmpqgroupbox = new QVGroupBox(i18n("Original Destination"),(QWidget *)tmpqbox2);
    tmpqbox2->setStretchFactor(tmpqgroupbox,1);

        // Port
    tmpqbox = new QHBox(tmpqgroupbox);
    tmpqbox->setSpacing(spacingHint());
    tmplabel = new QLabel(i18n("Port: "),tmpqbox);
    tmpqbox->setStretchFactor(tmplabel,0);

    portprotocolcombobox = new KComboBox((QWidget *)tmpqbox);
    tmpqbox->setStretchFactor(portprotocolcombobox,0);
    portprotocolcombobox->insertItem(i18n("TCP"));
    portprotocolcombobox->insertItem(i18n("UDP"));
    connect(portprotocolcombobox,SIGNAL(activated(int)),this,SLOT(slotPortProtocolComboBox(int)));

    originalportspinbox = new KIntSpinBox(0,65535,1,1,10,tmpqbox);
    connect(originalportspinbox,SIGNAL(valueChanged(int)),this,SLOT(slotOriginalPortSpinBox(int)));
            
    tmplabel = new QLabel(i18n("IP Address: "),tmpqgroupbox);

        // "( ) This machine"
    tmpqbox = new QHBox(tmpqgroupbox);
    tmpqbox->setSpacing(spacingHint());
    tmpwidget = new QWidget(tmpqbox);
    tmpwidget->setMinimumWidth(INDENT_SIZE);
    tmpqbox->setStretchFactor(tmpwidget,0);
    originalthismachineradio = new QRadioButton(i18n("This machine"),tmpqbox);
    tmpqbox->setStretchFactor(originalthismachineradio,1);
    connect(originalthismachineradio,SIGNAL(toggled(bool)),this,SLOT(slotOriginalMachineRadio(bool)));
    
        // "( ) Specify: "
    tmpqbox = new QHBox(tmpqgroupbox);
    tmpqbox->setSpacing(spacingHint());
    tmpwidget = new QWidget(tmpqbox);
    tmpwidget->setMinimumWidth(INDENT_SIZE);
    tmpqbox->setStretchFactor(tmpwidget,0);
    originalspecifyradio = new QRadioButton(i18n("Specify: "),tmpqbox);
    tmpqbox->setStretchFactor(originalspecifyradio,0);
    connect(originalspecifyradio,SIGNAL(toggled(bool)),this,SLOT(slotOriginalSpecifyRadio(bool)));
    originalspecifylineedit = new SimeLineEdit(tmpqbox);
    tmpqbox->setStretchFactor(originalspecifylineedit,0);
    connect(originalspecifylineedit,SIGNAL(textChanged(const QString &)),this,SLOT(slotOriginalSpecifyLineEdit(const QString &)));
    originalspecifyaddressvalidator = new IPValidator(originalspecifylineedit);
    originalspecifylineedit->setValidator(originalspecifyaddressvalidator);

    
    // Arrow icon
    tmplabel = new QLabel(tmpqbox2);
    tmplabel->setPixmap(UserIcon("little_right_arrow"));
    tmpqbox2->setStretchFactor(tmplabel,0);
            
        // "New Destination"
    tmpqgroupbox = new QVGroupBox(i18n("New Destination"),(QWidget *)tmpqbox2);
    tmpqbox2->setStretchFactor(tmpqgroupbox,1);

    tmpqbox = new QHBox(tmpqgroupbox);
    tmpqbox->setSpacing(spacingHint());
    tmplabel = new QLabel(i18n("Port: "),tmpqbox);
    tmpqbox->setStretchFactor(tmplabel,0);
    newportspinbox = new KIntSpinBox(0,65535,1,1,10,tmpqbox);
    connect(newportspinbox,SIGNAL(valueChanged(int)),this,SLOT(slotNewPortSpinBox(int)));

    tmplabel = new QLabel(i18n("IP Address: "),tmpqgroupbox);

        // "( ) This machine"
    tmpqbox = new QHBox(tmpqgroupbox);
    tmpqbox->setSpacing(spacingHint());
    tmpwidget = new QWidget(tmpqbox);
    tmpwidget->setMinimumWidth(INDENT_SIZE);
    tmpqbox->setStretchFactor(tmpwidget,0);
    newthismachineradio = new QRadioButton(i18n("This machine"),tmpqbox);
    tmpqbox->setStretchFactor(newthismachineradio,1);
    connect(newthismachineradio,SIGNAL(toggled(bool)),this,SLOT(slotNewMachineRadio(bool)));
    
        // "( ) Specify: "
    tmpqbox = new QHBox(tmpqgroupbox);
    tmpqbox->setSpacing(spacingHint());
    tmpwidget = new QWidget(tmpqbox);
    tmpwidget->setMinimumWidth(INDENT_SIZE);
    tmpqbox->setStretchFactor(tmpwidget,0);
    newspecifyradio = new QRadioButton(i18n("Specify: "),tmpqbox);
    tmpqbox->setStretchFactor(newspecifyradio,0);
    connect(newspecifyradio,SIGNAL(toggled(bool)),this,SLOT(slotNewSpecifyRadio(bool)));

    newspecifylineedit = new SimeLineEdit(tmpqbox);
    tmpqbox->setStretchFactor(newspecifylineedit,0);
    connect(newspecifylineedit,SIGNAL(textChanged(const QString &)),this,SLOT(slotNewSpecifyLineEdit(const QString &)));
    newspecifyaddressvalidator = new IPValidator(newspecifylineedit);
    newspecifylineedit->setValidator(newspecifyaddressvalidator);
    
        // Comment line edit
    tmpqbox2 = new QHBox(forwardingpage);
    tmpqbox2->setSpacing(spacingHint());
    tmplabel = new QLabel(i18n("Comment: "),tmpqbox2);
    tmpqbox2->setStretchFactor(tmplabel,0);
    commentlineedit = new SimeLineEdit(tmpqbox2);
    connect(commentlineedit,SIGNAL(textChanged(const QString &)),this,SLOT(slotCommentLineEdit(const QString &)));
           
    
        // Advanced Page
    advancedpage = addVBoxPage(i18n("Advanced"));
    advancedpage->setSpacing(spacingHint());

    disablecheckbox = new QCheckBox(i18n("Disable Guidedog"),(QWidget *)advancedpage);
    connect(disablecheckbox,SIGNAL(toggled(bool)),this,SLOT(slotDisableGuidedog(bool)));

        // Import/Export.
    tmpqgroupbox = new QVGroupBox(i18n("Import/Export"),(QWidget *)advancedpage);

    tmplabel = new QLabel(i18n("Description:"),tmpqgroupbox);
    descriptionedit = new KEdit(tmpqgroupbox);
    connect(descriptionedit,SIGNAL(textChanged()),this,SLOT(slotDescriptionChanged()));

    tmpqbox = new QHBox((QWidget *)tmpqgroupbox);
    tmpqbox->setSpacing(spacingHint());
    importbutton = new QPushButton(i18n("Import..."),tmpqbox);
    tmpqbox->setStretchFactor(importbutton,1);
    connect(importbutton,SIGNAL(clicked()),this,SLOT(slotImportButton()));
    exportbutton = new QPushButton(i18n("Export..."),tmpqbox);
    tmpqbox->setStretchFactor(exportbutton,1);
    connect(exportbutton,SIGNAL(clicked()),this,SLOT(slotExportButton()));
	
	readOptions();
	openDefault();
    updatinggui = true;
    syncGUIFromDoc();
    updatinggui = false;
    return true;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::saveOptions() {
    KConfig *config;
    config = kapp->config();

    config->setGroup("General");
    config->writeEntry("Geometry", size());
    config->writeEntry("CommandrunnerGeometry",commandrunnersize);
    config->sync();
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::readOptions() {
    KConfig *config;
    config = kapp->config();

    config->setGroup("General");
    QSize size = config->readSizeEntry("Geometry");
    if(!size.isEmpty()) {
        resize(size);
    }
    commandrunnersize = config->readSizeEntry("CommandrunnerGeometry");
    if(commandrunnersize.isEmpty()) {
        commandrunnersize.setWidth(400);
        commandrunnersize.setHeight(300);
    }
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::syncGUIFromDoc() {
    int i;

    routingcheckbox->setChecked(doc->isRouting());
    masqueradecheckbox->setChecked(doc->isMasquerade());
    masqueradeftpcheckbox->setChecked(doc->isMasqueradeFTP());
    masqueradeirccheckbox->setChecked(doc->isMasqueradeIRC());
    descriptionedit->setText(doc->description);
    disablecheckbox->setChecked(doc->isDisabled());

        // Populate the nomasq list box.
    nomasqlistbox->clear();
    for(i=0; i<doc->nomasqueradelist.count(); i++) {
        nomasqlistbox->insertItem(doc->nomasqueradelist.at(i)->getAddress());
    }
        // Select the first item.
    if(doc->nomasqueradelist.count()!=0) {
        nomasqlistbox->setCurrentItem(0);
        nomasqlineedit->setText(doc->nomasqueradelist.at(nomasqlistbox->currentItem())->getAddress());
    }

        // Populate the forward rule box.
    forwardlistbox->clear();
    for(i=0; i<doc->forwardrulelist.count(); i++) {
        forwardlistbox->insertItem(doc->forwardrulelist.at(i)->getSummary());
    }
        // Select the first item.    
    if(doc->forwardrulelist.count()!=0) {
        forwardlistbox->setCurrentItem(0);
        
        if(forwardlistbox->currentItem()!=-1) {
            setForwardRule(doc->forwardrulelist.at(forwardlistbox->currentItem()));
        } else {
            setForwardRule(0);
        }
    }
    enabledGUIStuff();
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::setForwardRule(GuidedogPortForwardRule *rule) {

    if(rule==0) {
        originalportspinbox->setValue(0);
        portprotocolcombobox->setCurrentItem(0);
        originalthismachineradio->setChecked(true);
        originalspecifyradio->setChecked(false);
        originalspecifylineedit->setText("");
        newportspinbox->setValue(0);
        newthismachineradio->setChecked(true);
        newspecifyradio->setChecked(true);
        newspecifylineedit->setText("");
        commentlineedit->setText("");
    } else {
        originalportspinbox->setValue(rule->originalport);
        portprotocolcombobox->setCurrentItem(rule->iptype==IPPROTO_TCP ? 0 : 1 );
        originalthismachineradio->setChecked(rule->specifyoriginal!=true);
        originalspecifyradio->setChecked(rule->specifyoriginal);
        originalspecifylineedit->setText(rule->originaladdress.getAddress());
        newportspinbox->setValue(rule->newport);
        newthismachineradio->setChecked(rule->specifynewaddress!=true );
        newspecifyradio->setChecked(rule->specifynewaddress);
        newspecifylineedit->setText(rule->newaddress.getAddress());
        commentlineedit->setText(rule->comment);
    }
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::enabledGUIStuff() {
    bool active,gotaddys,gotrules;
    GuidedogPortForwardRule *rule;
    
    active = !doc->isDisabled();
    routingcheckbox->setEnabled(active);
    masqueradecheckbox->setEnabled(doc->isRouting() && active);
    masqueradeftpcheckbox->setEnabled(doc->isRouting() && doc->isMasquerade() && active);
    masqueradeirccheckbox->setEnabled(doc->isRouting() && doc->isMasquerade() && active);
    gotaddys = doc->nomasqueradelist.count()!=0;
    nomasqlistbox->setEnabled(doc->isRouting() && doc->isMasquerade() && active && gotaddys);
    nomasqlineedit->setEnabled(doc->isRouting() && doc->isMasquerade() && active && gotaddys);
    newnomasqbutton->setEnabled(doc->isRouting() && doc->isMasquerade() && active);
    deleteenomasqbutton->setEnabled(doc->isRouting() && doc->isMasquerade() && active && gotaddys);

    gotrules = doc->forwardrulelist.count()!=0;
    forwardlistbox->setEnabled(active && gotrules);
    newforwardbutton->setEnabled(active);
    deleteforwardbutton->setEnabled(active && gotrules);

    originalportspinbox->setEnabled(active && gotrules);
    portprotocolcombobox->setEnabled(active && gotrules);
    originalthismachineradio->setEnabled(active && gotrules);
    originalspecifyradio->setEnabled(active && gotrules);
    originalspecifylineedit->setEnabled(active && gotrules);

    newportspinbox->setEnabled(active && gotrules);
    newthismachineradio->setEnabled(active && gotrules);
    newspecifyradio->setEnabled(active && gotrules);
    newspecifylineedit->setEnabled(active && gotrules);

    commentlineedit->setEnabled(active && gotrules);

    if(gotrules) {
        rule = doc->forwardrulelist.at(forwardlistbox->currentItem());
        originalspecifylineedit->setEnabled(active && rule->specifyoriginal);
        newspecifylineedit->setEnabled(active && rule->specifynewaddress);
    }
    
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotOk() {
	QString errorstring;
    QString filename(SYSTEM_RC_GUIDEDOG);

    if(doc->saveScript(filename,errorstring)==false) {
		KMessageBox::error(this,i18n("An error occurred while writing the script to disk.\n\n"
            "(Detailed message: \"%1\")").arg(errorstring));
		return;
	}
	if(applyScript(true)) {
        saveOptions();
        accept();
	}
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotCancel() {
    QString errorstring;

    if(waspreviousconfiguration &&  systemconfigmodified) {
            // This is where things become complex.
            // Should we try to restore things to how they were before this program started?
        switch(KMessageBox::warningYesNoCancel(this,
  		    	i18n("The system's routing configuration has been modified.\n\n"
                "Shall I restore it to the previous configuration?\n\n"
                "These changes may disrupt current network connections."))) {
                // "Yes, revert to the previous settings."
            case KMessageBox::Yes:
                openDefault();
        		if(applyScript(false)) {
                    saveOptions();
                    accept();
                }
                break;

                // "Just leave the settings alone and piss off!!"
            case KMessageBox::No:
                saveOptions();
                accept();
                break;

                // "Forget I ever pressed the Cancel button."
            case KMessageBox::Cancel:
                break;
        }
    } else {
            // Simple Cancel.
        saveOptions();
        accept();
    }	
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotApply() {
    applyScript(true);
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotUser1() {
    aboutus->show();
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotRouting(bool on) {
    if(updatinggui) return;
    updatinggui = true;

    doc->setRouting(on);

    enabledGUIStuff();
    updatinggui = false;
}
///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotMasquerade(bool on) {
    if(updatinggui) return;
    updatinggui = true;

    doc->setMasquerade(on);

    enabledGUIStuff();
    updatinggui = false;
}
///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotMasqueradeFTP(bool on) {
    if(updatinggui) return;
    updatinggui = true;
    doc->setMasqueradeFTP(on);
    updatinggui = false;
}
///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotMasqueradeIRC(bool on) {
    if(updatinggui) return;
    updatinggui = true;
    doc->setMasqueradeIRC(on);
    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotForwardListBox(QListBoxItem *item) {
    GuidedogPortForwardRule *rule;
    
    if(item==0) {
        return;
    }
    if(updatinggui) return;
    updatinggui = true;

    rule = doc->forwardrulelist.at(forwardlistbox->currentItem());
    setForwardRule(rule);
    enabledGUIStuff();

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotDisableGuidedog(bool on) {

    if(updatinggui) return;
    updatinggui = true;

    doc->setDisabled(on);
    syncGUIFromDoc();

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotImportButton() {
    QString filename;
    QString errorstring;
    GuidedogDoc *tmpdoc;

    if(updatinggui) return;
    updatinggui = true;

    filename = KFileDialog::getOpenFileName(QString::null,QString::null,this,i18n("Import Configuration"));
    if(filename.isEmpty()) {
        updatinggui = false;
        return;
    }

    tmpdoc = new GuidedogDoc();
    if(tmpdoc->openScript(filename,errorstring)==false) {
            // Stick up a good ol' error message.
        KMessageBox::error(0,i18n("Guidedog was unable to read the file at %1 as being a Guidedog script.\n"
            "This probably means that this file in not actually a Guidedog script.\n\n"
            "(Detailed message \"%2\")").arg(filename).arg(errorstring));
        delete tmpdoc;
        updatinggui = false;
        return;
    }

        // That loaded ok. Re-configure the GUI.

    delete doc;     //Switcherroo
    doc = tmpdoc;
    syncGUIFromDoc();
    updatinggui = false;
}
///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotExportButton() {
    QString filename;
    QString errorstring;

    if(updatinggui) return;
    updatinggui = true;

    filename = KFileDialog::getSaveFileName(QString::null,QString::null,this,i18n("Export Configuration"));
    if(filename.isEmpty()) {
        updatinggui = false;
        return;
    }

    if(doc->saveScript(filename, errorstring)==false) {
		KMessageBox::error(this,i18n("An error occurred while writing the script to %1.\n\n"
            "(Detailed message: \"%2\")").arg(filename).arg(errorstring));
    }
    updatinggui = false;

}
///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotDescriptionChanged() {
    if(updatinggui) return;
    updatinggui = true;

    doc->description = descriptionedit->text();

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotNoMasqueradeListBox(QListBoxItem *item) {
    if(updatinggui) return;
    updatinggui = true;

    if(item==0) {
        return;
    }

    nomasqlineedit->setText(doc->nomasqueradelist.at(nomasqlistbox->currentItem())->getAddress());
    
    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotNewAddressButton() {
    if(updatinggui) return;
    updatinggui = true;

    doc->nomasqueradelist.append(new IPRange("new.address"));
    nomasqlistbox->insertItem((doc->nomasqueradelist.getLast())->getAddress());
    nomasqlineedit->setText((doc->nomasqueradelist.getLast())->getAddress());
    nomasqlistbox->setCurrentItem(nomasqlistbox->count()-1);
    enabledGUIStuff();
    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotDeleteAddressButton() {
    IPRange *thisiprange;
    int i;
        
    if(updatinggui) return;
    updatinggui = true;
    thisiprange = doc->nomasqueradelist.at(nomasqlistbox->currentItem());

    i = nomasqlistbox->currentItem();
    if(i>=0) {
        nomasqlistbox->removeItem(i);
        doc->nomasqueradelist.remove(i);
        if(doc->nomasqueradelist.count()) {
            nomasqlineedit->setText(doc->nomasqueradelist.at(0)->getAddress());
            nomasqlistbox->setSelected(nomasqlistbox->item(0),true);
        } else {
            nomasqlineedit->setText("");
            nomasqlistbox->setDisabled(true);
            deleteenomasqbutton->setDisabled(true);
            nomasqlineedit->setDisabled(true);
        }
        if(nomasqlistbox->currentItem()==-1) {
            nomasqlistbox->setSelected(nomasqlistbox->item(0),true);
        }
    }
    enabledGUIStuff();
        
    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotAddressLineEdit(const QString &s) {
    int i;

    if(updatinggui) return;
    updatinggui = true;

    i = nomasqlistbox->currentItem();
    doc->nomasqueradelist.at(i)->setAddress(s);
    nomasqlistbox->changeItem(s,i);

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotAddressLineEditReturn() {

}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotNewForwardButton() {
    GuidedogPortForwardRule *newrule;

    if(updatinggui) return;
    updatinggui = true;

    newrule = new GuidedogPortForwardRule();
    doc->forwardrulelist.append(newrule);
    forwardlistbox->insertItem((doc->forwardrulelist.getLast())->getSummary());
    setForwardRule(newrule);
    forwardlistbox->setCurrentItem(forwardlistbox->count()-1);
    enabledGUIStuff();

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotDeleteForwardButton() {
    GuidedogPortForwardRule *rule;
    int i;

    if(updatinggui) return;
    updatinggui = true;

    rule = doc->forwardrulelist.at(forwardlistbox->currentItem());

    i = forwardlistbox->currentItem();
    if(i>=0) {
        forwardlistbox->removeItem(i);
        doc->forwardrulelist.remove(i);
        forwardlistbox->setCurrentItem(0);
        if(forwardlistbox->currentItem()==-1) {
            setForwardRule(0);
        } else {
            setForwardRule(doc->forwardrulelist.at(forwardlistbox->currentItem()));
        }
        enabledGUIStuff();
    }
    updatinggui = false;
}


///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotOriginalMachineRadio(bool on) {
    int i;
    GuidedogPortForwardRule *rule;

    if(updatinggui) return;
    updatinggui = true;

    originalthismachineradio->setChecked(true);
    originalspecifyradio->setChecked(false);

    i = forwardlistbox->currentItem();
    rule = doc->forwardrulelist.at(i);
    if(rule!=0) {
        rule->specifyoriginal = false;
        forwardlistbox->changeItem(rule->getSummary(),i);
    }
    enabledGUIStuff();

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotOriginalSpecifyRadio(bool on) {
    int i;
    GuidedogPortForwardRule *rule;

    if(updatinggui) return;
    updatinggui = true;

    originalspecifyradio->setChecked(true);
    originalthismachineradio->setChecked(false);

    i = forwardlistbox->currentItem();
    rule = doc->forwardrulelist.at(i);
    if(rule!=0) {
        rule->specifyoriginal = true;
        forwardlistbox->changeItem(rule->getSummary(),i);
    }
    enabledGUIStuff();

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotNewMachineRadio(bool on) {
    int i;
    GuidedogPortForwardRule *rule;

    if(updatinggui) return;
    updatinggui = true;

    newthismachineradio->setChecked(true);
    newspecifyradio->setChecked(false);

    i = forwardlistbox->currentItem();
    rule = doc->forwardrulelist.at(i);
    if(rule!=0) {
        rule->specifynewaddress = false;
        forwardlistbox->changeItem(rule->getSummary(),i);
    }
    enabledGUIStuff();

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotNewSpecifyRadio(bool on) {
    int i;
    GuidedogPortForwardRule *rule;

    if(updatinggui) return;
    updatinggui = true;

    newspecifyradio->setChecked(true);
    newthismachineradio->setChecked(false);

    i = forwardlistbox->currentItem();
    rule = doc->forwardrulelist.at(i);
    if(rule!=0) {
        rule->specifynewaddress = true;
        forwardlistbox->changeItem(rule->getSummary(),i);
    }
    enabledGUIStuff();

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotOriginalPortSpinBox(int x) {
    int i;
    GuidedogPortForwardRule *rule;

    if(updatinggui) return;
    updatinggui = true;

    i = forwardlistbox->currentItem();
    rule = doc->forwardrulelist.at(i);
    if(rule!=0) {
        rule->originalport = x;
        // Update the forward listbox.
        forwardlistbox->changeItem(rule->getSummary(),i);
    }
    
    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotNewPortSpinBox(int x) {
    int i;
    GuidedogPortForwardRule *rule;

    if(updatinggui) return;
    updatinggui = true;

    i = forwardlistbox->currentItem();
    rule = doc->forwardrulelist.at(i);
    if(rule!=0) {
        rule->newport = x;
        // Update the forward listbox.
        forwardlistbox->changeItem(rule->getSummary(),i);
    }

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotPortProtocolComboBox(int x) {
    int i;
    GuidedogPortForwardRule *rule;

    if(updatinggui) return;
    updatinggui = true;

    i = forwardlistbox->currentItem();
    rule = doc->forwardrulelist.at(i);
    if(rule!=0) {
        rule->iptype = x==0 ? IPPROTO_TCP : IPPROTO_UDP;
        // Update the forward listbox.
        forwardlistbox->changeItem(rule->getSummary(),i);
    }

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotOriginalSpecifyLineEdit(const QString &s) {
    int i;
    GuidedogPortForwardRule *rule;

    if(updatinggui) return;
    updatinggui = true;

    i = forwardlistbox->currentItem();
    rule = doc->forwardrulelist.at(i);
    if(rule!=0) {
        rule->originaladdress.setAddress(s);
        // Update the forward listbox.
        forwardlistbox->changeItem(rule->getSummary(),i);
    }

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotNewSpecifyLineEdit(const QString &s) {
    int i;
    GuidedogPortForwardRule *rule;

    if(updatinggui) return;
    updatinggui = true;

    i = forwardlistbox->currentItem();
    rule = doc->forwardrulelist.at(i);
    if(rule!=0) {
        rule->newaddress.setAddress(s);
        // Update the forward listbox.
        forwardlistbox->changeItem(rule->getSummary(),i);
    }

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::slotCommentLineEdit(const QString &s) {
    int i;
    GuidedogPortForwardRule *rule;

    if(updatinggui) return;
    updatinggui = true;

    i = forwardlistbox->currentItem();
    rule = doc->forwardrulelist.at(i);
    if(rule!=0) {
        rule->comment = s;
        // Update the forward listbox.
        forwardlistbox->changeItem(rule->getSummary(),i);
    }

    updatinggui = false;
}

///////////////////////////////////////////////////////////////////////////
bool GuidedogApp::applyScript(bool warnfirst) {
    QString errorstring;
    KTempFile tmpfile(0,0,0700);
    CommandRunner cr(this);

    tmpfile.setAutoDelete(true);

    if(doc->isDisabled()==false) {
            // Normal apply.
        if(warnfirst==false || KMessageBox::warningContinueCancel(this,
                i18n("You are about to modify the system's routing configuration.\n"
                "These changes may disrupt current network connections.\n\n"
                "Do you wish to continue?"),0,i18n("Continue"))==KMessageBox::Continue) {

                // Is our temp file working?
            if(tmpfile.status()!=0) {
                KMessageBox::error(this,i18n("An error occurred while trying to modify system's routing configuration.\nThe operating system has this to report about the error: %1")
                    .arg(strerror(tmpfile.status())));
                return false;
            }
                // Write the firewall script into the temp file.
            if(doc->writeScript(*(tmpfile.textStream()),errorstring)==false) {
                return false;
            }
                // Close and flush the file.
            if(!tmpfile.close()) {
                KMessageBox::error(this,i18n("An error occurred while modifying the routing configuration.\nThe operating system has this to report about the error: %1")
                    .arg(strerror(tmpfile.status())));
                return false;
            }
                // Now we run the tmp script in our super friendly window.
            if(!commandrunnersize.isEmpty()) {
                cr.resize(commandrunnersize);
            }
            cr.setPlainCaption(i18n("Modify Routing Configuration"));
            cr.setHeading(i18n("Configuring...\n\nOutput:"));
            cr.run(QString("export GUIDEDOG_VERBOSE=1;")+tmpfile.name());
            systemconfigmodified = true;
            commandrunnersize = cr.size();
            return true;
        }
        return false;
    } else {
            // If we are to actually disable the firewall because the user has set
            // the Disable checkbox in the advanced section, then we use a different
            // warning question and use a different method to reset the network stack.
        if(warnfirst==false || KMessageBox::warningContinueCancel(this,
                i18n("You are about to reset the system's routing configuration.\n"
                "These changes may also disrupt current network connections.\n\n"
                "Do you wish to continue?"),0,i18n("Continue"))==KMessageBox::Continue) {

            if(resetSystemConfiguration()) {
                return false;
            } else {
                systemconfigmodified = true;
                return true;
            }
        }
        return false;
    }
    return false;
}
///////////////////////////////////////////////////////////////////////////
bool GuidedogApp::resetSystemConfiguration() {
    CommandRunner cr(this);

    if(!commandrunnersize.isEmpty()) {
        cr.resize(commandrunnersize);
    }

    cr.setPlainCaption(i18n("Resetting system configuration"));
    cr.setHeading(i18n("Resetting system configuration...\n\nOutput:"));
    cr.run(QString(
        "echo \"Using ipchains.\"\n"
        "echo \"Resetting nat table rules.\"\n"
        "PATH=/bin:/sbin:/usr/bin:/usr/sbin\n"
        "/sbin/sysctl -w net.ipv4.ip_forward=0\n"
        "iptables -t nat -F\n"
        "iptables -t nat -X\n"
        "iptables -t nat -P PREROUTING ACCEPT\n"
        "iptables -t nat -P POSTROUTING ACCEPT\n"
        "iptables -t nat -P OUTPUT ACCEPT\n"
        "echo \"Finished.\"\n"));
    commandrunnersize = cr.size();
    return true;
}

///////////////////////////////////////////////////////////////////////////
void GuidedogApp::openDefault() {
    QString filename(SYSTEM_RC_GUIDEDOG);
    QString errorstring;
    QFileInfo fileinfo(SYSTEM_RC_GUIDEDOG);

    if(superusermode==false) {
        return; // Sorry, if you are not root then you get no default firewall.
    }

	if(fileinfo.exists()==false) {
            // There doesn't appear to be a previous Guarddog firewall script.
            // Just warn the user about the ramifications.
        KMessageBox::information(0,i18n("Guidedog was unable to find a Guidedog script at %1.\n"
            "This is probably ok, it just means that this is the first time Guidedog has been run on this system.\n"
            "But please be aware that the settings shown may not represent the system's current routing configuration.\n"
            "Your Guidedog settings will take effect once you use the 'Apply' button or exit Guidedog using 'Ok'.").arg(filename));
    } else {
	    if(doc->openScript(filename,errorstring)==false) {
	        doc->factoryDefaults();
                // We were unable to open the guarddog firewall.
            KMessageBox::error(0,i18n("Guidedog was unable to read the file at %1 as being a Guidedog script.\n"
            "Please be aware that the settings shown may not represent the system's current routing configuration.\n\n"
            "(Detailed message \"%2\")").arg(filename).arg(errorstring));
        } else {
            waspreviousconfiguration = true;
        }
    }
}
