/* routines to look up cd and track title based on cddb number and track number
 * stored in the track's data structure
 * Once a cddb lookup has been completed the track is assigned it's final name
 * automatically.
 * The cddb_lookup() function is getting called by the track itself as soon
 * as it's cddb field contains a number !=0 */

/* default lookup is being done at freedb.freedb.org port 888 */

#ifndef CDDB_H
#define CDDB_H

#include <time.h>

#include "tracks.h"

/* max. query size. This is enough for a cd with 99 tracks,
 * so this shouldn't be subject to an overflow as long as the
 * standard is taken care of. 
 * an additional if clause is making sure that only cds with less or
 * equal to 99 tracks are being looked up */
#define CDDB_MAXQUERYSIZE 2048

/* this contains all we need to identify a compact disc with the cddb server */
typedef struct
{
   int cddb_key;
   int playingtime;
   int tracks;
   int offsets[99];
} cddb_essentials;   

typedef struct 
{
   /* the jobnumber returned by cddb_lookup and used by cddb_killjob 
    * is simply a pointer to cddb_infostruct. */
   char cd_title[255];
   char track_title[255];
   char artist[255];
   /* the following flag becomes 1 as an indication that the
    * request has been completed successfully. */
   int  complete;
   
   gpointer callback; /* the callback to be called once the job is complete */
   gpointer data;     /* the data parameter passed to the callback 
		       * on completion */
   
   int tracknum;      /* track No. to look up (starting at 1) */
   cddb_essentials es;
   
   /* this is used to control how often failing requests are being 
    * retried */
   time_t last_lookup;
   int timeout;
   
} cddb_infostruct;

typedef void (*cddb_datahandler)(cddb_infostruct *result,
				 gpointer data);

/* the following function is rather lowlevel. It looks up an entry
 * based on the cddb_essentials struct passed to it
 * (cddb_lookup is making a copy of that struct so the original is no
 * longer needed and can safely be freed)
 * and calls a handler with the result as it's parameter.
 * This function also returns a job identifier with which the job can be 
 * cancelled. 
 * data can point to additional informations passed to the callback
 * but may otherwise safely be set to NULL */
int cddb_lookup(cddb_essentials *es,
		int tracknum,
		cddb_datahandler callback,
		gpointer data);
/* kill a cddb lookup job. this is done e.g. in tracks.c when the last reference
 * to a track has been cleared and the track is about to be deleted */
void cddb_killjob(int jobid);
/* The following two functions are primarily used to restart the
 * cddb lookup process in case the server adress has been changed in
 * the preferences setup */
void cddb_start_lookup();
void cddb_kill_lookup();
/* control functions called by main.c to bring up/down cddb interface */
void cddb_init();
void cddb_destroy();

#endif // CDDB_H
