/* doodle.c - Shared doodling program for etalk
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.ai.mit.edu.
 *
 * $Log: doodle.c,v $
 * Revision 1.2  1997/10/15 02:11:14  zappo
 * Added check for display return value.
 *
 * Revision 1.1  1996/03/17 15:23:21  zappo
 * Initial revision
 *
 * ::Header:: doodle.h
 */

#include "config.h"
#include "headers.h"

#include <X11/Xlib.h>
#include <X11/keysym.h>

/* Some Athena widgets we need */
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/Xaw/Command.h> 
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/AsciiTextP.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Toggle.h>

#include "gnuicon.xbm"
#include "doodle.h"

int main(argc, argv)
     int argc;
     char *argv[];
{
  struct Xcontext XCtxt;
  Pixmap icon;
  XGCValues values;

  printf("%s\n", DOODLE_);

  if(argc < 2)
    {
      fprintf(stderr, "Usage: %s <sock fd>\n", argv[0]);
      exit(1);
    }

  XCtxt.currentTool = Point;
  XCtxt.currentColor = Black;

  XtToolkitInitialize();	/* get things rolling */

  XCtxt.app_context = XtCreateApplicationContext(); /* get a context */

  /* Open the display, and read the database... */
  XCtxt.display = XtOpenDisplay(XCtxt.app_context,
				NULL, /* name of display */
				"Doodle",
				"Doodle",
				NULL, 0, /* resource options */
				&argc, argv);

  if(XCtxt.display == NULL) {
    fprintf(stderr, "%s: Cannot open display!\n", argv[0]);
  }

  /* Load in the GC we will use in our application */
  XCtxt.gc = DefaultGC(XCtxt.display, DefaultScreen(XCtxt.display));

  icon = XCreateBitmapFromData(XCtxt.display,
			       XDefaultRootWindow(XCtxt.display),
			       gnuicon_bits, gnuicon_width, gnuicon_height);

  /* Ok, now make a top-level shell */
  XCtxt.topLevel = XtVaAppCreateShell("Doodle",
				      "Doodle",
				      applicationShellWidgetClass,
				      XCtxt.display,
				      XtNallowShellResize, True,
				      XtNheight, 400,
				      XtNwidth, 400,
				      /* don't check, a null means 
					 no icon anyway */
				      XtNiconPixmap, icon,
				      NULL
				      );

  /* Now, initialize the shells we need to put our widgets into */
  XCtxt.topPane = 
    XtVaCreateManagedWidget("DoodlePane", panedWidgetClass, XCtxt.topLevel,
			    XtNorientation, XtorientVertical,
			    NULL);

  /* Create a menu up on the top. */
  XCtxt.menuBox =
    XtVaCreateManagedWidget("MenuBox", boxWidgetClass, XCtxt.topPane,
			    XtNallowResize,       True,
			    XtNresizeToPreferred, True,
			    XtNshowGrip,          False,
			    XtNmin,               10,
			    XtNorientation,       XtorientHorizontal,
			    NULL);

  /* A pane with tools on the left */
  XCtxt.doodlePane = 
    XtVaCreateManagedWidget("DoodlePane", panedWidgetClass, XCtxt.topPane,
			    XtNorientation, XtorientHorizontal,
			    NULL);

  /* Create a menu up on the top. */
  XCtxt.toolBox =
    XtVaCreateManagedWidget("toolBox", boxWidgetClass, XCtxt.doodlePane,
			    XtNallowResize,       False,
			    XtNresizeToPreferred, True,
			    XtNshowGrip,          False,
			    XtNmin,               10,
			    XtNorientation,       XtorientVertical,
			    NULL);

  /* Create toolbox for the drawing tools */
  XCtxt.drawToolBox =
    XtVaCreateManagedWidget("drawToolBox", boxWidgetClass, XCtxt.toolBox,
			    XtNallowResize,       False,
			    XtNresizeToPreferred, True,
			    XtNshowGrip,          False,
			    XtNorientation,       XtorientVertical,
			    NULL);
  XW_build_radio_tools(XCtxt.drawToolBox, &XCtxt);

  /* Create the porthole in which we will view our painting */
  XCtxt.doodlePort =
    XtVaCreateManagedWidget("doodlePort", viewportWidgetClass, XCtxt.doodlePane,
			    XtNallowResize,       True,
			    XtNresizeToPreferred, False,
			    XtNshowGrip,          False,
			    XtNorientation,       XtorientVertical,
			    XtNallowHoriz,        True,
			    XtNallowVert,         True,
			    XtNforceBars,         True,
			    XtNuseBottom,         True,
			    XtNuseRight,          True,
			    NULL);

  /* Create a canvas type thing */
  XCtxt.doodlePix = XCreatePixmap(XCtxt.display,
				  DefaultRootWindow(XCtxt.display),
				  600,
				  600,
				  DefaultDepth(XCtxt.display,
					       DefaultScreen(XCtxt.display)));

  XCtxt.doodleCanvas = XtVaCreateManagedWidget("DoodleBox",
					       coreWidgetClass,
					       XCtxt.doodlePort,
					       XtNborderWidth, 0,
					       XtNwidth, 600,
					       XtNheight, 600,
					       NULL);

  /* Add handlers to core so we can manipulate the image in it */
  XtAddEventHandler(XCtxt.doodleCanvas, ButtonPressMask | PointerMotionMask
		    | ButtonReleaseMask, False,
		    (XtEventHandler)X_mouse_doodle, &XCtxt);
  XtAddEventHandler(XCtxt.doodleCanvas, ExposureMask, False,
		    (XtEventHandler)X_expose_doodle, &XCtxt);

  /* Lets create a bunch of GCs so we can draw in our space */
  values.foreground = BlackPixel(XCtxt.display, XDefaultScreen(XCtxt.display));
  values.background = WhitePixel(XCtxt.display, XDefaultScreen(XCtxt.display));
  XCtxt.black = XtGetGC(XCtxt.doodleCanvas, GCForeground | GCBackground, &values);

  values.foreground = WhitePixel(XCtxt.display, XDefaultScreen(XCtxt.display));
  values.background = BlackPixel(XCtxt.display, XDefaultScreen(XCtxt.display));
  XCtxt.white = XtGetGC(XCtxt.doodleCanvas, GCForeground | GCBackground, &values);

  values.function = GXxor;
  values.foreground = ~0;
  values.background = 0;
  XCtxt.xor = XtGetGC(XCtxt.doodleCanvas, GCForeground|GCBackground|GCFunction,
		      &values);

  /* Initialize the image */
  XFillRectangle(XCtxt.display, XCtxt.doodlePix, XCtxt.white,
		 0, 0, 600, 600);

  /* Create toolbox for the drawing tools */
  XCtxt.colorToolBox =
    XtVaCreateManagedWidget("drawToolBox", boxWidgetClass, XCtxt.toolBox,
			    XtNallowResize,       False,
			    XtNresizeToPreferred, True,
			    XtNshowGrip,          False,
			    XtNorientation,       XtorientVertical,
			    NULL);
  XW_build_radio_colors(XCtxt.colorToolBox, &XCtxt);

  X_make_menus(XCtxt.menuBox);

  /* Put it all up */
  XtRealizeWidget (XCtxt.topLevel);

  /* Initialize the network part. */
  dnet_init(argv[1], &XCtxt);

  /* Event Loop */
  XtAppMainLoop(XCtxt.app_context);  
 
  return 0;
}


/*
 * Function: X_expose_doodle
 *
 *   Whenever the widget displaying our image is damaged, we must
 * refresh our display.  May someday also have temp data to refresh as
 * well.
 *
 * Returns:     Nothing
 * Parameters:  w         - Widget w
 *              XCtxt     - Context
 *              call_data - Number of call data
 * History:
 * zappo   3/8/96     Created
 */
void X_expose_doodle(w, XCtxt, call_data)
     Widget w;
     struct Xcontext *XCtxt;
     XtPointer call_data;
{
  XCopyArea(XCtxt->display, XCtxt->doodlePix, XtWindow(XCtxt->doodleCanvas),
	    XCtxt->gc, 0, 0, 600, 600, 0, 0);
} /* X_expose_doodle( ) */

void XW_build_radio_colors(parent, XCtxt)
     Widget parent;
     struct Xcontext *XCtxt;
{
  Widget prev = NULL, new;
  Pixel p, f;
  int i;
  Colormap cmap;
  XGCValues newvalues;

  cmap = DefaultColormap(XCtxt->display, DefaultScreen(XCtxt->display));

  for(i = White; i < MAXCOLOR; i++)
    {
      if(i == White)
	{
	  XCtxt->gcs[White] = XCtxt->white;
	  p = WhitePixel(XCtxt->display, XDefaultScreen(XCtxt->display));
	  f = BlackPixel(XCtxt->display, XDefaultScreen(XCtxt->display));
	}
      else if(i == Black)
	{
	  XCtxt->gcs[Black] = XCtxt->black;
	  p = BlackPixel(XCtxt->display, XDefaultScreen(XCtxt->display));
	  f = WhitePixel(XCtxt->display, XDefaultScreen(XCtxt->display));
	}
      else
	{
	  XColor tc1, tc2;
	  /* Parse out the new color */
	  if(XAllocNamedColor(XCtxt->display, cmap, color_names[i], &tc1, &tc2))
	    {
	      newvalues.background = newvalues.foreground = p = tc1.pixel;
	      XCtxt->gcs[i] = XCreateGC(XCtxt->display,
					XCtxt->doodlePix,
					GCForeground | GCBackground,
					&newvalues);
	      f = BlackPixel(XCtxt->display, XDefaultScreen(XCtxt->display));
	    }
	  else
	    {
	      fprintf(stderr, "Error allocating color %s.\n", color_names[i]);
	      return;
	    }
	}
      new = XtVaCreateManagedWidget(color_names[i], toggleWidgetClass, 
				    parent,
				    XtNlabel, color_names[i],
				    XtNstate, (i==XCtxt->currentColor)?True:False,
				    XtNradioGroup, prev,
				    XtNradioData, i,
				    XtNbackground, p,
				    XtNforeground, f,
				    NULL);
      XtAddCallback(new, XtNcallback, (XtCallbackProc)X_color_cb, XCtxt);
      prev = new;
    } /* for */
}


/*
 * Function: XW_build_radio_group
 *
 *   Builds a group of widgets for a radio box full of tools.
 *
 * Returns:     Nothing
 * Parameters:  parent - Widget parent
 *              XCtxt  - Context
 * History:
 * zappo   3/8/96     Created
 */
void XW_build_radio_tools(parent, XCtxt)
     Widget parent;
     struct Xcontext *XCtxt;
{
  Widget prev = NULL, new;
  int i;
  
  for(i = Point; i < MAXTOOL; i++)
    {
      new = XtVaCreateManagedWidget(tool_names[i], toggleWidgetClass, 
				    parent,
				    XtNlabel, tool_names[i],
				    XtNstate, (i==XCtxt->currentTool)?True:False,
				    XtNradioGroup, prev,
				    XtNradioData, i,
				    NULL);
      XtAddCallback(new, XtNcallback, (XtCallbackProc)X_toggle_cb, XCtxt);
      prev = new;
    } /* for */
  
} /* XW_build_radio_group( ) */


/*
 * Function: XW_build_menu
 *
 *   Use this to build one widget menu with callbacks.  All callbacks
 * are given data 
 *
 * Returns:     Nothing
 * Parameters:  parent   - Widget parent
 *              name     - Name of
 *              items    -  items
 *              numitems - Number of number
 * History:
 * zappo   9/16/95    Created
 */
void XW_build_menu(parent, name, items, numitems)
     Widget parent;
     char *name;
     struct MenuItemConstructor *items;
     int numitems;
{
  Widget mbutton, menu, item;
  int i;

  mbutton = XtVaCreateManagedWidget(name, menuButtonWidgetClass, parent,
				    XtNlabel,    name,
				    XtNmenuName, name,
				    XtNresize,   False,
				    NULL);

  menu = XtVaCreatePopupShell(name, simpleMenuWidgetClass, mbutton,
			      XtNlabel, name,
			      NULL);

  for (i=0; i < numitems; i++) {
    item = XtVaCreateManagedWidget(items[i].button_name, smeBSBObjectClass,
				   menu, 
				   XtNlabel, items[i].button_name,
				   NULL);

    XtAddCallback(item, XtNcallback, items[i].callback, items[i].data);
  }
}

/*
 * Function: X_make_menus
 *
 *   This function is called from the widget control file since the
 * menus will always be the same across multiple X widget interfaces.
 *
 * Returns:     Nothing
 * Parameters:  parent - Widget parent to place menu buttons in
 *
 * History:
 * zappo   9/22/95    Created
 */
void X_make_menus(parent)
     Widget parent;
{
  static struct MenuItemConstructor file[] =
    {
      { "Quit", "Exit Doodle",
	  (XtCallbackProc)X_quit, NULL }
    };

  /* Put some menu items into it */
  XW_build_menu(parent, "File", file, 
		sizeof(file)/sizeof(struct MenuItemConstructor));
  
}
