/* gt_filter.c - Filter management
 *
 * Copyright (C) 1998, 1999 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * 
 * Description:
 *
 *   This file will serv as a starting point for adding in new
 * filters.  At the moment, a simple ROT13 obscurement is here, but the
 * ability to pipe the data through translators (english->spanish) or
 * character filters (remove umlots, ets) can be implemented by those
 * who might better know how to handle them.
 *   
 * 
 * History:
 *
 * Tokens: ::Header:: gtalkc.h 
 */
#include "gtalklib.h"
#include "gtalkc.h"

enum filter_type { no_filter = 0, ROT13 = 1 };

static char *etalk_filter_names[] =
{
  "None",
  "ROT13"
};

static char *etalk_filter_describe[] =
{
  "Do not use a filter, or turn off filtration.",
  "Rotate printable alphanumeric characters by 13.",
};

struct generic_filter_data
{
  int  counter;
  int  data_length;
  char data[1];		/* leave unbounded for exact sizing */
};


/*
 * Function: FILTER_describe
 *
 *   Describes the available methods for filtration.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   11/28/99    Created
 */
void FILTER_describe(Ctxt)
     struct TalkContext *Ctxt;
{
  int i;

#ifndef NO_IFACE
  DISP_create_popup(Ctxt, 70, 4+(sizeof(etalk_filter_names)/sizeof(char *)));
#endif

  DISP_message(Ctxt, "Available filtration types");

  for(i = 0; i < sizeof(etalk_filter_names)/sizeof(char *); i++)
    {
      char buffer[100];

      sprintf(buffer, "%s\t%s", etalk_filter_names[i], 
	      etalk_filter_describe[i]);

      DISP_message(Ctxt, buffer);
    } 
}


/*
 * Function: FILTER_name
 *
 *   Provides a name for the type encoded number internally.
 *
 * Returns:     char * - 
 * Parameters:  type - Type of
 *
 * History:
 * zappo   11/28/99    Created
 */
char *FILTER_name(int type)
{
  if ((type < sizeof(etalk_filter_names)/sizeof(char *)) &&
      (type >= 0))
    return etalk_filter_names[type];
  else
    return "Error";
}


/*
 * Function: FILTER_index
 *
 *   Return the local filtration index based on the filter name.
 *
 * Returns:     int - 
 * Parameters:  name - Name of filter method we want.
 *
 * History:
 * zappo   11/28/99   Created
 */
int FILTER_index(char *name)
{
  int i;

  for (i = 0; i < sizeof(etalk_filter_names)/sizeof(char *); i++)
    {
      if(GTL_scmp(etalk_filter_names[i], name, strlen(name)))
	{
	  return i;
	}
    }
  return -1; /* Does not exist */
}


/*
 * Function: FILTER_init
 *
 *   Initializes uo so that all future data is filtered
 * with the remote user.
 *
 * Returns:     int  - 
 * Parameters:  Ctxt - Context
 *              uo   - Pointer to User Object uo
 *              type - Type of
 * History:
 * zappo   11/28/99    Created
 */
int FILTER_init(Ctxt, uo, type)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
     int                 type;
{
  char obuff[30];
  char answer;

  /* First, make sure we can contact them */
  if(!((uo->type == GNUTALK) || 
       ((uo->type == ETALK) && (((uo->ver == 0) && (uo->num >= 11)) ||
				(uo->ver > 0)))))
    {
      DISP_message(Ctxt, "\03Selected user does not support filtration.");
      return Fail;
    }

  DISP_message(Ctxt, "\03Negotiating filter...");

  sprintf(obuff, "%c%c%s %s\n", ETALK_ESCAPE, TCP_FILTER_CONNECTION, 
	  etalk_filter_names[type], "foo");

  USER_send(Ctxt, uo, obuff, strlen(obuff));

  answer = USER_get_answer(Ctxt, uo);

  if(answer == TCP_ANSWER_YES)
    {
      /* turn off any allocated filter parts */
      if(uo->outbound_filter_data)
	{
	  free (uo->outbound_filter_data);
	  uo->outbound_filter_data = NULL;
	}

      /* Lastly, add the filter starting now. */
      uo->outbound_filter = type;

      if(Ctxt->runstate == Socket) {
	printf("\03%c%c%d:%s\n", TTY_FILTER_MESSAGE, TTY_FILTER_OUTBOUND,
	       uo->id, etalk_filter_names[type]);
      }

      DISP_message(Ctxt, "\03Negotiating filter...done");
      DISP_update_user(Ctxt, uo);
    }

  return Success;
}


/*
 * Function: FILTER_respond
 *
 *   When a filter request comes in from a remote, then we must
 * respond YES or NO depending on if we support that filter
 * method.  If yes, we must also find our filter data, and pass that back
 * as well.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              uo   - Pointer to user Object
 *              type - Type of filter
 *              data - Extra filter info
 * History:
 * zappo   11/28/99     Created
 */
void FILTER_respond(Ctxt, uo, type, data)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
     char *type;
     char *data;
{
  char obuff[100];
  int i, j;

  for (i = 0, j = 0; i < sizeof(etalk_filter_names)/sizeof(char *); i++)
    {
      if(GTL_scmp(etalk_filter_names[i], type, strlen(type)))
	{
	  j = 1;
	  break;
	}
    }

  if(j)
    {
      sprintf(obuff, "%c%c%c\n", ETALK_ESCAPE, TCP_ANSWER, TCP_ANSWER_YES);
      USER_send(Ctxt, uo, obuff, strlen(obuff));

      /* turn off any allocated filter parts */
      if(uo->inbound_filter_data)
	{
	  free (uo->inbound_filter_data);
	  uo->inbound_filter_data = NULL;
	}

      /* enable the filter */
      uo->inbound_filter = i;

      if(Ctxt->runstate == Socket) {
	printf("%c%c%c%d:%s\n", ETALK_ESCAPE, 
	       TTY_FILTER_MESSAGE, TTY_FILTER_INBOUND,
	       uo->id, etalk_filter_names[i]);
	
      }

      DISP_message(Ctxt, "\03Remove filter enabled.");
      DISP_update_user(Ctxt, uo);
    }
  else
    {
      sprintf(obuff, "%c%c%cFilter type not supported.\n",
	      ETALK_ESCAPE, TCP_ANSWER, TCP_ANSWER_NO);
      USER_send(Ctxt, uo, obuff, strlen(obuff));
      DISP_message(Ctxt, "Failed to initialize remote filter.");
    }
}


/*
 * Function: ROT13_buffer
 *
 *   Locally defined function which will decode or encode a buffer
 * using ROT13.
 *
 * Returns:     static char * - 
 * Parameters:  inbuff  - buffer to rotate
 *              outbuff - buffer to fill
 *              size    - Number of size
 * History:
 * zappo   11/28/99    Created
 */
static void ROT13_buffer(inbuff, outbuff, size)
     char *inbuff;
     char *outbuff;
     int   size;
{
  int i;

  for(i = 0; i < size; i++)
    {
      if((inbuff[i] >= 'a') && (inbuff[i] <= 'z'))
	{
	  outbuff[i] = (((inbuff[i] - 'a') + 13) % ('z' - 'a' + 1)) + 'a';
	}
      else if((inbuff[i] >= 'A') && (inbuff[i] <= 'Z'))
	{
	  outbuff[i] = (((inbuff[i] - 'A') + 13) % ('Z' - 'A' + 1)) + 'A';
	}
      else
	{
	  outbuff[i] = inbuff[i];
	}
    }
  outbuff[i] = 0;
}


/*
 * Function: FILTER_outbound
 *
 *   Filters BUFF into a newly allocated buffer of the the correct size.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt    - Context
 *              uo      - Pointer  uo
 *              inbuff  - Raw buffer
 *              outbuff - filtered buffer
 *              insize  - size of the input buffer
 *              outsize - size of the output buffer
 * History:
 * zappo   11/28/99    Created
 */
void FILTER_outbound(Ctxt, uo, inbuff, outbuff, insize, outsize)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
     char               *inbuff;
     char               *outbuff;
     int                 insize;
     int                *outsize;
{
  switch(uo->outbound_filter)
    {
    case no_filter:		/* none */
      break;
    case ROT13:			/* ROT13 */
      ROT13_buffer(inbuff, outbuff, insize);
      *outsize = insize;
      break;
    default:			/* error */
      break;
    }

  return;
}


/*
 * Function: FILTER_inbound
 *
 *   Filters BUFF and returns an allocated buffer.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt    - Context
 *              uo      - Pointer  uo
 *              inbuff  - raw buffer
 *              outbuff - filtered buffer
 *              insize  - Characters in input buffer
 *              outsize - Characters in output buffer
 * History:
 * zappo   11/28/99    Created
 */
void FILTER_inbound(Ctxt, uo, inbuff, outbuff, insize, outsize)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
     char               *inbuff;
     char               *outbuff;
     int                 insize;
     int                *outsize;
{
  switch(uo->inbound_filter)
    {
    case no_filter:		/* none */
      break;
    case ROT13:			/* ROT13 */
      ROT13_buffer(inbuff, outbuff, insize);
      *outsize = insize;
      break;
    default:			/* error */
      break;
    }

  return;
}


