#
# Gramps - a GTK+/GNOME based genealogy program
#
# Copyright (C) 2000-2005  Donald N. Allingham
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

# $Id: EditSource.py,v 1.25.2.9 2005/03/14 02:35:51 rshura Exp $

#-------------------------------------------------------------------------
#
# Python modules
#
#-------------------------------------------------------------------------
from gettext import gettext as _

#-------------------------------------------------------------------------
#
# GTK/Gnome modules
#
#-------------------------------------------------------------------------
import gtk.glade
import gnome

#-------------------------------------------------------------------------
#
# gramps modules
#
#-------------------------------------------------------------------------
import const
import Utils
import GrampsCfg
import ImageSelect
import ListModel

#-------------------------------------------------------------------------
#
# Constants
#
#-------------------------------------------------------------------------

class EditSource:

    def __init__(self,source,db,parent_window=None,func=None):
        self.source = source
        self.db = db
        self.callback = func
        self.path = db.getSavePath()
        self.not_loaded = 1
        self.ref_not_loaded = 1
        self.lists_changed = 0
        self.gallery_ok = 0

        self.top_window = gtk.glade.XML(const.gladeFile,"sourceEditor","gramps")
        self.top = self.top_window.get_widget("sourceEditor")

        Utils.set_titles(self.top,self.top_window.get_widget('title'),
                         _('Source Editor'))
        
        plwidget = self.top_window.get_widget("iconlist")
        self.gallery = ImageSelect.Gallery(source, self.path, plwidget, db, self, self.top)
        self.author = self.top_window.get_widget("author")
        self.pubinfo = self.top_window.get_widget("pubinfo")
        self.abbrev = self.top_window.get_widget("abbrev")
        self.note = self.top_window.get_widget("source_note")
        self.notes_buffer = self.note.get_buffer()
        self.gallery_label = self.top_window.get_widget("gallerySourceEditor")
        self.refs_label = self.top_window.get_widget("refsSourceEditor")
        self.notes_label = self.top_window.get_widget("notesSourceEditor")
        self.flowed = self.top_window.get_widget("source_flowed")
        self.preform = self.top_window.get_widget("source_preform")
        
        self.refinfo = self.top_window.get_widget("refinfo")
        
        self.title = self.top_window.get_widget("source_title")
        self.title.set_text(source.getTitle())
        self.author.set_text(source.getAuthor())
        self.pubinfo.set_text(source.getPubInfo())
        self.abbrev.set_text(source.getAbbrev())

        if source.getNote():
            self.notes_buffer.set_text(source.getNote())
            Utils.bold_label(self.notes_label)
            if source.getNoteFormat() == 1:
                self.preform.set_active(1)
            else:
                self.flowed.set_active(1)

        if self.source.getPhotoList():
            Utils.bold_label(self.gallery_label)

        self.top_window.signal_autoconnect({
            "on_switch_page" : self.on_switch_page,
            "on_addphoto_clicked" : self.gallery.on_add_photo_clicked,
            "on_selectphoto_clicked"    : self.gallery.on_select_photo_clicked,
            "on_deletephoto_clicked" : self.gallery.on_delete_photo_clicked,
            "on_editphoto_clicked"     : self.gallery.on_edit_photo_clicked,
            "on_edit_properties_clicked": self.gallery.popup_change_description,
            "on_sourceEditor_help_clicked" : self.on_help_clicked,
            })

        if self.source.getId() == "":
            self.top_window.get_widget("edit_photo").set_sensitive(0)
            self.top_window.get_widget("delete_photo").set_sensitive(0)

        if parent_window:
            self.top.set_transient_for(parent_window)

        self.display_references()
        self.top.show()
        self.val = self.top.run()
        if self.val == gtk.RESPONSE_OK:
            self.on_source_apply_clicked()
        self.top.destroy()

    def on_help_clicked(self,obj):
        """Display the relevant portion of GRAMPS manual"""
        gnome.help_display('gramps-manual','gramps-edit-complete')
        self.val = self.top.run()

    def close(self,obj):
        self.gallery.close(self.gallery_ok)
        self.top.destroy()
        
    def display_references(self):
        p_top_list = []
        p_event_list = []
        p_event_list_photo = []
        p_event_list_photo_attr = []
        p_lds_list = []
        p_addr_list = []
        p_attr_list = []
        p_name_list = []
        p_photo_list = []
        p_photo_attr_list = []
        f_top_list = []
        f_event_list = []
        f_event_list_photo = []
        f_event_list_photo_attr = []
        f_attr_list = []
        f_photo_list = []
        f_photo_attr_list = []
        f_lds_list = []
        m_list = []
        m_attr_list = []
        pl_list = []
        pl_photo_list = []
        pl_photo_attr_list = []
        s_photo_list = []
        s_photo_attr_list = []

        # Person
        for key in self.db.getPersonKeys():
            p = self.db.getPerson(key)
            name = GrampsCfg.nameof(p)
            for sref in p.getSourceRefList():
                if sref.getBase() == self.source:
                    p_top_list.append(name)
            for v in p.getEventList() + [p.getBirth(), p.getDeath()]:
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        p_event_list.append((name,v.getName()))
                for vv in v.getPhotoList():
                    for sref in vv.getSourceRefList():
                        if sref.getBase() == self.source:
                            p_event_list_photo.append((name,vv.getReference().getDescription()))
                    for vvv in vv.getAttributeList():
                        for sref in vvv.getSourceRefList():
                            if sref.getBase() == self.source:
                                p_event_list_photo_attr.append((name,vvv.getType()))
            if p.getLdsBaptism():
                for sref in p.getLdsBaptism().getSourceRefList():
                    if sref.getBase() == self.source:
                        p_lds_list.append((name,_('LDS Baptism')))
            if p.getLdsEndowment():
                for sref in p.getLdsEndowment().getSourceRefList():
                    if sref.getBase() == self.source:
                        p_lds_list.append((name,_('Endowment')))
            if p.getLdsSeal():
                for sref in p.getLdsSeal().getSourceRefList():
                    if sref.getBase() == self.source:
                        p_lds_list.append((name,_('Sealed to parents')))

            for v in p.getAttributeList():
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        p_attr_list.append((name,v.getType()))
            for v in p.getAlternateNames() + [p.getPrimaryName()]:
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        p_name_list.append((name,v.getName()))
            for v in p.getAddressList():
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        p_addr_list.append((name,v.getStreet()))
            for v in p.getPhotoList():
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        p_photo_list.append((name,v.getReference().getDescription()))
                for vv in v.getAttributeList():
                    for sref in vv.getSourceRefList():
                        if sref.getBase() == self.source:
                            p_photo_attr_list.append((name,vv.getType()))
        # Family
        for p in self.db.getFamilyMap().values():
            f = p.getFather()
            m = p.getMother()
            if f and m:
                name = _("%(father)s and %(mother)s") % {
                    "father" : GrampsCfg.nameof(f),
                    "mother" : GrampsCfg.nameof(m)}
            elif f:
                name = GrampsCfg.nameof(f)
            else:
                name = GrampsCfg.nameof(m)
            for sref in p.getSourceRefList():
                if sref.getBase() == self.source:
                    f_top_list.append(name)
            for v in p.getEventList():
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        f_event_list.append((name,v.getName()))
                for vv in v.getPhotoList():
                    for sref in vv.getSourceRefList():
                        if sref.getBase() == self.source:
                            f_event_list_photo.append((name,vv.getReference().getDescription()))
                    for vvv in vv.getAttributeList():
                        for sref in vvv.getSourceRefList():
                            if sref.getBase() == self.source:
                                f_event_list_photo_attr.append((name,vvv.getType()))
            if p.getLdsSeal():
                for sref in p.getLdsSeal().getSourceRefList():
                    if sref.getBase() == self.source:
                        f_lds_list.append((name,_('Sealed to spouse')))
            for v in p.getAttributeList():
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        f_attr_list.append((name,v.getType()))
            for v in p.getPhotoList():
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        f_photo_list.append((name,v.getReference().getDescription()))
                for vv in v.getAttributeList():
                    for sref in vv.getSourceRefList():
                        if sref.getBase() == self.source:
                            f_photo_attr_list.append((name,vv.getType()))
        # Media
        for p in self.db.getObjectMap().values():
            name = p.getDescription()
            for sref in p.getSourceRefList():
                if sref.getBase() == self.source:
                    m_list.append(name)
            for v in p.getAttributeList():
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        m_attr_list.append((name,v.getType()))
        # Place
        for key in self.db.getPlaceKeys():
            p = self.db.getPlace(key) 
            name = p.get_title()
            for sref in p.getSourceRefList():
                if sref.getBase() == self.source:
                    pl_list.append(name)
            for v in p.getPhotoList():
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        pl_photo_list.append((name,v.getReference().getDescription()))
                for vv in v.getAttributeList():
                    for sref in vv.getSourceRefList():
                        if sref.getBase() == self.source:
                            pl_photo_attr_list.append((name,vv.getType()))
        # Source
        for key in self.db.getSourceKeys():
            p = self.db.getSource(key) 
            name = p.getTitle()
            for v in p.getPhotoList():
                for sref in v.getSourceRefList():
                    if sref.getBase() == self.source:
                        s_photo_list.append((name,v.getReference().getDescription()))
                for vv in v.getAttributeList():
                    for sref in vv.getSourceRefList():
                        if sref.getBase() == self.source:
                            s_photo_attr_list.append((name,vv.getType()))

        slist = self.top_window.get_widget('slist')

        titles = [(_('Source Type'),0,150),(_('Object'),1,150),(_('Value'),2,150)]
        
        self.model = ListModel.ListModel(slist,titles)
        any = 0
        if len(p_top_list) > 0:
            any = 1
            for p in p_top_list:
                self.model.add([_("Person"),p])

        if len(p_event_list) > 0:
            any = 1
            for p in p_event_list:
                self.model.add([_("Personal Event"),p[0],
                                const.display_pevent(p[1])])

        if len(p_event_list_photo) > 0:
            any = 1
            for p in p_event_list_photo:
                self.model.add([_("Personal Event Gallery"),p[0],p[1]])

        if len(p_event_list_photo_attr) > 0:
            any = 1
            for p in p_event_list_photo_attr:
                self.model.add([_("Personal Event Gallery Attribute"),p[0],p[1]])

        if len(p_attr_list) > 0:
            any = 1
            for p in p_attr_list:
                self.model.add([_("Personal Attribute"),p[0],
                               const.display_pattr(p[1])])
        if len(p_addr_list) > 0:
            any = 1
            for p in p_addr_list:
                self.model.add([_("Address"),p[0],p[1]])

        if len(p_name_list) > 0:
            any = 1
            for p in p_name_list:
                self.model.add([_("Name"),p[0],p[1]])

        if len(p_photo_list) > 0:
            any = 1
            for p in p_photo_list:
                self.model.add([_("Personal Gallery"),p[0],p[1]])

        if len(p_photo_attr_list) > 0:
            any = 1
            for p in p_photo_attr_list:
                self.model.add([_("Personal Gallery Attribute"),p[0],p[1]])

        if len(p_lds_list) > 0:
            any = 1
            for p in p_lds_list:
                self.model.add([_("Personal LDS event"),p[0],p[1]])

        if len(f_top_list) > 0:
            any = 1
            for p in f_top_list:
                self.model.add([_("Family"),p])

        if len(f_event_list) > 0:
            any = 1
            for p in f_event_list:
                self.model.add([_("Family Event"),p[0],
                                const.display_fevent(p[1])])

        if len(f_lds_list) > 0:
            any = 1
            for p in f_lds_list:
                self.model.add([_("Family LDS event"),p[0],p[1]])

        if len(f_event_list_photo) > 0:
            any = 1
            for p in f_event_list_photo:
                self.model.add([_("Family Event Gallery"),p[0],p[1]])

        if len(f_event_list_photo_attr) > 0:
            any = 1
            for p in f_event_list_photo_attr:
                self.model.add([_("Family Event Gallerie Attribute"),p[0],p[1]])

        if len(f_attr_list) > 0:
            any = 1
            for p in f_attr_list:
                self.model.add([_("Family Attribute"),p[0],
                                const.display_fattr(p[1])])

        if len(f_photo_list) > 0:
            any = 1
            for p in f_photo_list:
                self.model.add([_("Family Gallery"),p[0],p[1]])

        if len(f_photo_attr_list) > 0:
            any = 1
            for p in f_photo_attr_list:
                self.model.add([_("Family Gallery Attribute"),p[0],p[1]])

        if len(m_list) > 0:
            any = 1
            for p in m_list:
                self.model.add([_("Media Object"),p,''])

        if len(m_attr_list) > 0:
            any = 1
            for p in m_attr_list:
                self.model.add([_("Media Attribute"),p[0],p[1]])

        if len(pl_list) > 0:
            any = 1
            for p in pl_list:
                self.model.add([_("Place"),p,''])
        
        if len(pl_photo_list) > 0:
            any = 1
            for p in pl_photo_list:
                self.model.add([_("Place Gallery"),p[0],p[1]])

        if len(pl_photo_attr_list) > 0:
            any = 1
            for p in pl_photo_attr_list:
                self.model.add([_("Place Gallery Attribute"),p[0],p[1]])

        if len(s_photo_list) > 0:
            any = 1
            for p in s_photo_list:
                self.model.add([_("Source Gallery"),p[0],p[1]])

        if len(s_photo_attr_list) > 0:
            any = 1
            for p in s_photo_attr_list:
                self.model.add([_("Source Gallery Attribute"),p[0],p[1]])

        if any:
            Utils.bold_label(self.refs_label)
        else:
            Utils.unbold_label(self.refs_label)
            
        self.ref_not_loaded = 0

    def on_source_apply_clicked(self):

        title = unicode(self.title.get_text())
        author = unicode(self.author.get_text())
        pubinfo = unicode(self.pubinfo.get_text())
        abbrev = unicode(self.abbrev.get_text())
        note = unicode(self.notes_buffer.get_text(self.notes_buffer.get_start_iter(),
                                  self.notes_buffer.get_end_iter(),False))
        format = self.preform.get_active()

        if author != self.source.getAuthor():
            self.source.setAuthor(author)
            Utils.modified()
        
        if title != self.source.getTitle():
            self.source.setTitle(title)
            Utils.modified()
        
        if pubinfo != self.source.getPubInfo():
            self.source.setPubInfo(pubinfo)
            Utils.modified()
        
        if abbrev != self.source.getAbbrev():
            self.source.setAbbrev(abbrev)
            Utils.modified()
        
        if note != self.source.getNote():
            self.source.setNote(note)
            Utils.modified()

        if format != self.source.getNoteFormat():
            self.source.setNoteFormat(format)
            Utils.modified()

        if self.lists_changed:
            Utils.modified()
        
        self.gallery_ok = 1
        self.close(None)

        if self.callback:
            self.callback(self.source)

    def on_switch_page(self,obj,a,page):
        if page == 2 and self.not_loaded:
            self.not_loaded = 0
            self.gallery.load_images()
        elif page == 3 and self.ref_not_loaded:
            self.ref_not_loaded = 0
            self.display_references()
        text = unicode(self.notes_buffer.get_text(self.notes_buffer.get_start_iter(),
                                self.notes_buffer.get_end_iter(),False))
        if text:
            Utils.bold_label(self.notes_label)
        else:
            Utils.unbold_label(self.notes_label)


class DelSrcQuery:
    def __init__(self,source,db,update):
        self.source = source
        self.db = db
        self.update = update

    def delete_source(self,object):
        m = 0
        l = []
        for sref in object.getSourceRefList():
            if sref.getBase() != self.source:
                l.append(sref)
            else:
                m = 1
        if m:
            object.setSourceRefList(l)

    def query_response(self):
        self.db.removeSource(self.source.getId())
        Utils.modified()

        # Person
        for key in self.db.getPersonKeys():
            p = self.db.getPerson(key)
            self.delete_source(p)
            for v in p.getEventList() + [p.getBirth(), p.getDeath()]:
                self.delete_source(v)
                for vv in v.getPhotoList():
                    self.delete_source(vv)
                    for vvv in vv.getAttributeList():
                        self.delete_source(vvv)
            if p.getLdsBaptism():
                self.delete_source(p.getLdsBaptism())
            if p.getLdsEndowment():
                self.delete_source(p.getLdsEndowment())
            if p.getLdsSeal():
                self.delete_source(p.getLdsSeal())

            for v in p.getAttributeList():
                self.delete_source(v)

            for v in p.getAlternateNames() + [p.getPrimaryName()]:
                self.delete_source(v)

            for v in p.getAddressList():
                self.delete_source(v)

            for v in p.getPhotoList():
                self.delete_source(v)
                for vv in v.getAttributeList():
                    self.delete_source(vv)

        # Family
        for p in self.db.getFamilyMap().values():
            self.delete_source(p)
            for v in p.getEventList():
                self.delete_source(v)
                for vv in v.getPhotoList():
                    self.delete_source(vv)
                    for vvv in vv.getAttributeList():
                        self.delete_source(vvv)
            if p.getLdsSeal():
                self.delete_source(p.getLdsSeal())

            for v in p.getAttributeList():
                self.delete_source(v)

            for v in p.getPhotoList():
                self.delete_source(v)
                for vv in v.getAttributeList():
                    self.delete_source(vv)

        # Media
        for p in self.db.getObjectMap().values():
            self.delete_source(p)
            for v in p.getAttributeList():
                self.delete_source(v)

        # Place
        for key in self.db.getPlaceKeys():
            p = self.db.getPlace(key) 
            self.delete_source(p)
            for v in p.getPhotoList():
                self.delete_source(v)
                for vv in v.getAttributeList():
                    self.delete_source(vv)

        # Source
        for key in self.db.getSourceKeys():
            p = self.db.getSource(key) 
            for v in p.getPhotoList():
                self.delete_source(v)
                for vv in v.getAttributeList():
                    self.delete_source(vv)

        self.update(0)
