/***************************************************************************
                          gpc_film.c  -  description
                             -------------------
    begin                : Wed May 17 2000
    copyright            : (C) 2000 by Thierry Florac
    email                : tflorac@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <dirent.h>
#include <gnome.h>

#include "gpc_app.h"
#include "gpc_film.h"
#include "gpc_film_infos.h"
#include "gpc_utils.h"

#include "pg_connection.h"
#include "pg_query.h"

#include "support.h"


/** Data structure to store current film informations */
struct _GpcFilmInfos {
  gint      ID;
  gchar     *pathname;
  gchar     *kind;
  gchar     *number;
  gchar     *maker;
  gchar     *model;
  gshort    iso;
  gshort    size;
  gchar     *camera_maker;
  gchar     *camera_model;
  gchar     *period;
  gchar     *description;
  GList     *keywords;
};


/** Create a new GpcFilmInfos structure */
GpcFilmInfos *gpc_film_infos_new(void)
{
  GpcFilmInfos *infos;

  infos = g_new0(GpcFilmInfos, 1);

  return infos;
}


/** Get informations corresponding to a given film ID */
GpcFilmInfos *gpc_film_get_infos (pgConnection *connection, gint ID)
{
  GpcFilmInfos *infos;
  pgQuery *query;
  gchar   *sql;

  if (!connection || !pg_connection_is_active (connection))
    return NULL;
  sql = g_strdup_printf ("select * from GPC_FILM "
                         "where (ID = %d)", ID);
  query = pg_query_new_with_sql (connection, sql);
  g_free (sql);
  if (!query)
    return NULL;
  if (pg_query_open (query) > 0) {
    infos = gpc_film_infos_new ();
    infos->ID = atoi (pg_query_get_fieldvalue_byname (query, "ID"));
    infos->pathname = g_strdup (pg_query_get_fieldvalue_byname (query, "PATHNAME"));
    infos->kind = g_strdup (pg_query_get_fieldvalue_byname (query, "KIND"));
    infos->number = g_strdup (pg_query_get_fieldvalue_byname (query, "NUMBER"));
    infos->maker = g_strdup (pg_query_get_fieldvalue_byname (query, "MAKER"));
    infos->model = g_strdup (pg_query_get_fieldvalue_byname (query, "MODEL"));
    infos->iso = atoi (pg_query_get_fieldvalue_byname (query, "ISO"));
    infos->size = atoi (pg_query_get_fieldvalue_byname (query, "SIZE"));
    infos->camera_maker = g_strdup (pg_query_get_fieldvalue_byname (query, "CAMERA_MAKER"));
    infos->camera_model = g_strdup (pg_query_get_fieldvalue_byname (query, "CAMERA_MODEL"));
    infos->period = g_strdup (pg_query_get_fieldvalue_byname (query, "PERIOD"));
    infos->description = g_strdup (pg_query_get_fieldvalue_byname (query, "DESCRIPTION"));
    pg_query_free (query);
    sql = g_strdup_printf ("select * from GPC_KEYWORD "
                           "where (FILM_ID = %d)", ID);
    query = pg_query_new_with_sql (connection, sql);
    g_free (sql);
    if (query && (pg_query_open (query) > 0)) {
      pg_query_move_first (query);
      while (! pg_query_at_EOF (query)) {
        infos->keywords = g_list_insert_sorted (infos->keywords, g_strdup (pg_query_get_fieldvalue_byname (query, "CODE")), (GCompareFunc) g_strcasecmp);
        pg_query_move_next (query);
      }
      pg_query_free (query);
    }
    return infos;
  }
  else {
    pg_query_free (query);
    return NULL;
  }

}


/* Retrieve ID of a film, given a RELATIVE path */
gint gpc_film_get_id_from_path (pgConnection *connection, gchar *path)
{
  pgQuery   *query;
  gchar     *sql;
  gchar     *sql_path;
  gint       result = -1;
  
  if (!connection || !pg_connection_is_active (connection))
    return result;
  sql_path = get_sql_string (path);
  sql = g_strdup_printf ("select ID from GPC_FILM where PATHNAME = %s", sql_path);
  query = pg_query_new_with_sql (connection, sql);
  g_free (sql);
  g_free (sql_path);
  if (!query)
    return result;
  if (pg_query_open (query) > 0)
    result = atoi (pg_query_get_fieldvalue_byname (query, "ID"));
  pg_query_free (query);
  return result;
}


/** Check to see if a film exists with specified RELATIVE directory */
gboolean gpc_film_exists (pgConnection *connection, gchar *path)
{
  return (gpc_film_get_id_from_path (connection, path) != -1);
}


/** Create a new reference to an existing ABSOLUTE directory, saved later as relative */
gint gpc_film_append (pgConnection *connection, gchar *path)
{
  gchar        *sql;
  gint          ID;
  gchar        *filmpath;
  gchar        *gpc_path;
  gchar        *sql_path;
  gchar        *fields[4];
  pgQuery      *query;

  if (!connection || !path || (strlen (path) == 0))
    return 0;
  if (path[strlen (path) - 1] != '/')
    filmpath = g_strconcat (path, "/", NULL);
  else
    filmpath = g_strdup (path);
  gpc_path = gpc_path_relname (filmpath); /* Convert absolute path to a repository relative one */
  if (gpc_film_exists (connection, gpc_path)) { /* film already exists */
    ID = gpc_film_get_id_from_path (connection, gpc_path);
    gpc_film_update (connection, ID);
    g_free (gpc_path);
  }
  else {
    sql_path = get_sql_string (gpc_path);
    sql = g_strdup_printf ("insert into GPC_FILM (PATHNAME, DESCRIPTION) values (%s, %s)", sql_path, sql_path);
    query = pg_query_new_with_sql (connection, sql);
    pg_query_execute (query);
    pg_query_free (query);
    g_free (sql);
    g_free (sql_path);
    ID = gpc_film_get_id_from_path (connection, gpc_path);
    if (ID) {
      gpc_film_update (connection, ID);
      if (!gpc_film_dialog)
        on_film_properties_view_menu_activate (NULL, NULL);
      if (films_clist) {
        fields[0] = g_strdup_printf ("%d", ID);
        fields[1] = "   ";
        fields[2] = g_strdup (gpc_path);;
        fields[3] = NULL;
        gtk_clist_select_row (films_clist, gtk_clist_append (films_clist, fields), 0);
      }
    }
  }
  g_free (filmpath);
  return ID;
}


/** Create a new reference to an existing ABSOLUTE directory, saved later as relative */
gint gpc_film_append_without_update (pgConnection *connection, gchar *path)
{
  gchar        *sql;
  gint          ID;
  gchar        *filmpath;
  gchar        *gpc_path;
  gchar        *sql_path;
  gchar        *fields[4];
  pgQuery      *query;

  if (!connection || !path || (strlen (path) == 0))
    return 0;
  if (path[strlen (path) - 1] != '/')
    filmpath = g_strconcat (path, "/", NULL);
  else
    filmpath = g_strdup (path);
  gpc_path = gpc_path_relname (filmpath); /* Convert absolute path to a repository relative one */
  if (gpc_film_exists (connection, gpc_path)) { /* film does already exists */
    ID = gpc_film_get_id_from_path (connection, gpc_path);
    g_free (gpc_path);
  }
  else {
    sql_path = get_sql_string (gpc_path);
    sql = g_strdup_printf ("insert into GPC_FILM (PATHNAME, DESCRIPTION) values (%s, %s)", sql_path, sql_path);
    query = pg_query_new_with_sql (connection, sql);
    pg_query_execute (query);
    pg_query_free (query);
    g_free (sql);
    g_free (sql_path);
    ID = gpc_film_get_id_from_path (connection, gpc_path);
    if (ID) {
      if (!gpc_film_dialog)
        on_film_properties_view_menu_activate (NULL, NULL);
      if (films_clist) {
        fields[0] = g_strdup_printf ("%d", ID);
        fields[1] = "   ";
        fields[2] = g_strdup (gpc_path);;
        fields[3] = NULL;
        gtk_clist_append (films_clist, fields);
      }
    }
  }
  g_free (filmpath);
  return ID;
}


gboolean gpc_film_update (pgConnection *connection, gint ID)
{
  GpcFilmInfos *infos;
  gchar        *path;
  gchar        *filepath;
  pgQuery      *query;
  gint          image_ID;
  gchar        *status;
  
  DIR           *dp;
  struct dirent *dir;
  struct stat    ent_sbuf;

  if (!connection || !pg_connection_is_active (connection))
    return FALSE;
  infos = gpc_film_get_infos (connection, ID);
  if (!infos)
    return FALSE;
  path = gpc_film_get_full_pathname (infos);
  if ((dp = opendir (path)) == NULL)          /* directory not found */
    return FALSE;
  while ((dir = readdir (dp)) != NULL) {
    if (dir->d_ino > 0) {                     /* skips removed files */
      gchar *name = dir->d_name;
      if (gpc_get_hidden_files || !file_is_hidden (name)) {
        filepath = g_strconcat (path, name, NULL);
        if (stat (filepath, &ent_sbuf) >= 0 && !S_ISDIR (ent_sbuf.st_mode)) {
          if (gpc_image_exists (connection, gpc_film_get_pathname (infos), name)) {
            status = g_strdup_printf (_("Updating image file: %s"), name);
            gnome_appbar_push (GNOME_APPBAR (lookup_widget (gpc_app, "appbar")), status);
            while (gtk_events_pending ())
              gtk_main_iteration ();
            gpc_image_update (connection, ID, gpc_film_get_pathname (infos), name);
          }
          else {
            status = g_strdup_printf (_("Inserting image file: %s"), name);
            gnome_appbar_push (GNOME_APPBAR (lookup_widget (gpc_app, "appbar")), status);
            while (gtk_events_pending ())
              gtk_main_iteration ();
            gpc_image_insert (connection, ID, gpc_film_get_pathname (infos), name);
          }
          g_free (status);
        }
        gnome_appbar_pop (GNOME_APPBAR (lookup_widget (gpc_app, "appbar")));
        while (gtk_events_pending ())
          gtk_main_iteration ();
        g_free (filepath);
      }
    }
  }
  closedir (dp);
  g_free (path);
  gpc_film_infos_free (infos);
  if (gpc_delete_removed_files) {
    query = gpc_image_get_list_from_film (connection, ID);
    while (!pg_query_at_EOF (query)) {
      image_ID = atoi (pg_query_get_fieldvalue_byname (query, "ID"));
      if (!gpc_image_file_exists (connection, image_ID))
        gpc_image_delete (connection, image_ID);
      pg_query_move_next (query);    
    }
    pg_query_free (query);
  }
  return TRUE;
}


/** Update a film, corresponding with a given RELATIVE path */
gboolean 
gpc_film_update_with_path (pgConnection *connection, gchar *path)
{
  gint          ID = -1;
  gchar        *sql;
  gchar        *sql_path;
  pgQuery      *query;
  
  if (!connection || !pg_connection_is_active (connection) || !path || (strlen (path) == 0))
    return FALSE;
  sql_path = get_sql_string (path);
  sql = g_strdup_printf ("select ID from GPC_FILM where PATHNAME = %s", sql_path);
  query = pg_query_new_with_sql (connection, sql);
  if (query && pg_query_open (query) > 0)
    ID = atoi (pg_query_get_fieldvalue_byname (query, "ID"));
  pg_query_free (query);
  g_free (sql);
  g_free (sql_path);
  return ((ID > -1) && gpc_film_update (connection, ID));
}


gboolean 
gpc_film_delete (pgConnection *connection, gint ID)
{
  gchar        *sql;
  pgQuery      *query;
  gboolean      result;

  if (connection == NULL)
    return FALSE;
  sql = g_strdup_printf ("delete from GPC_FILM where (ID = %d)", ID);
  query = pg_query_new_with_sql (connection, sql);
  result = pg_query_execute (query);
  if (!result)
    pg_connection_show_error (connection, _("PostgreSQL error :"), _("Can't delete current film...!"));
  pg_query_free (query);
  g_free (sql);
  return result;
}


/** Execute a query extracting every film in database */
pgQuery* 
gpc_film_get_list (pgConnection *connection)
{
  pgQuery *query;

  if (!connection || !pg_connection_is_active (connection))
    return NULL;
  query = pg_query_new_with_sql (connection, "select ID, NUMBER, DESCRIPTION "
                                             "from GPC_FILM "
                                             "order by NUMBER");
  if (!query)
    return NULL;
  if (pg_query_open (query) < 0) {
    pg_connection_show_error (connection, _("An error occured :"), _("Can't open query on GPC_FILM table...!"));
    pg_query_free (query);
    return NULL;
  }
  return query;
}


/** Execute a query extracting films corresponding to a given condition.
    Condition is given in SQL language without the "WHERE" keyword        */
pgQuery* 
gpc_film_get_list_where (pgConnection *connection, gchar *where)
{
  gchar   *sql;
  pgQuery *query;

  if (!connection || !pg_connection_is_active (connection))
    return NULL;
  sql = g_strdup_printf ("select ID, NUMBER, DESCRIPTION "
                         "from GPC_FILM "
                         "where %s "
                         "order by NUMBER", where);
  query = pg_query_new_with_sql (connection, sql);
  g_free (sql);
  if (!query)
    return NULL;
  if (pg_query_open (query) < 0) {
    pg_connection_show_error (connection, _("An error occured :"), _("Can't open query on GPC_FILM table...!"));
    pg_query_free (query);
    return NULL;
  }
  return query;
}


gint gpc_film_get_id (GpcFilmInfos *infos)
{
  if (!infos)
    return -1;
  return infos->ID;
}


gchar* gpc_film_get_pathname (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->pathname;
}

gchar* gpc_film_get_full_pathname (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  if (infos->pathname[0] == '.')
    return g_strconcat (gpc_repository, infos->pathname, NULL);
  else
    return g_strdup (infos->pathname);
}

gchar* gpc_film_get_kind (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->kind;
}

gchar* gpc_film_get_number (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->number;
}

gchar* gpc_film_get_maker (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->maker;
}

gchar* gpc_film_get_model (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->model;
}

gshort gpc_film_get_iso (GpcFilmInfos *infos)
{
  if (!infos)
    return -1;
  return infos->iso;
}

gshort gpc_film_get_size (GpcFilmInfos *infos)
{
  if (!infos)
    return -1;
  return infos->size;
}

gchar* gpc_film_get_camera_maker (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->camera_maker;
}

gchar* gpc_film_get_camera_model (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->camera_model;
}

gchar* gpc_film_get_period (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->period;
}

gchar* gpc_film_get_description (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->description;
}

GList* gpc_film_get_keywords (GpcFilmInfos *infos)
{
  if (!infos)
    return NULL;
  return infos->keywords;
}


void gpc_film_set_pathname (GpcFilmInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->pathname)
    g_free (infos->pathname);
//  infos->pathname = g_strdup (value);
  infos->pathname = gpc_path_relname (value);
}

void gpc_film_set_kind (GpcFilmInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->kind)
    g_free (infos->kind);
  infos->kind = g_strdup (value);
}

void gpc_film_set_number (GpcFilmInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->number)
    g_free (infos->number);
  infos->number = g_strdup (value);
}

void gpc_film_set_maker (GpcFilmInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->maker)
    g_free (infos->maker);
  infos->maker = g_strdup (value);
}

void gpc_film_set_model (GpcFilmInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->model)
    g_free (infos->model);
  infos->model = g_strdup (value);
}

void gpc_film_set_iso (GpcFilmInfos *infos, gshort value)
{
  if (!infos)
    return;
  infos->iso = value;
}

void gpc_film_set_size (GpcFilmInfos *infos, gshort value)
{
  if (!infos)
    return;
  infos->size = value;
}

void gpc_film_set_camera_maker (GpcFilmInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->camera_maker)
    g_free (infos->camera_maker);
  infos->camera_maker = g_strdup (value);
}

void gpc_film_set_camera_model (GpcFilmInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->camera_model)
    g_free (infos->camera_model);
  infos->camera_model = g_strdup (value);
}

void gpc_film_set_period (GpcFilmInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->period)
    g_free (infos->period);
  infos->period = g_strdup (value);
}

void gpc_film_set_description (GpcFilmInfos *infos, gchar *value)
{
  if (!infos)
    return;
  if (infos->description)
    g_free (infos->description);
  infos->description = g_strdup (value);
}

void gpc_film_set_keywords (GpcFilmInfos *infos, GList *value, gboolean free_it)
{
  if (!infos)
    return;
  if (infos->keywords && free_it) {
    g_list_foreach (infos->keywords, (GFunc) g_free, NULL);
    g_list_free (infos->keywords);
  }
  infos->keywords = value;
}


gboolean gpc_film_infos_store (pgConnection *connection, GpcFilmInfos *infos)
{
  gchar   *sql;
  gchar   *kind;
  gchar   *number;
  gchar   *maker;
  gchar   *model;
  gchar   *camera_maker;
  gchar   *camera_model;
  gchar   *period;
  gchar   *description;

  pgQuery *query;
  gboolean result;

  if (!connection || !infos || !pg_connection_is_active (connection))
    return FALSE;
  if (!gpc_film_modified)
    return TRUE;
  if (!pg_connection_start (connection))
    return FALSE;
  kind = get_sql_string (gpc_film_get_kind (infos));
  number = get_sql_string (gpc_film_get_number (infos));
  maker = get_sql_string (gpc_film_get_maker (infos));
  model = get_sql_string (gpc_film_get_model (infos));
  camera_maker = get_sql_string (gpc_film_get_camera_maker (infos));
  camera_model = get_sql_string (gpc_film_get_camera_model (infos));
  period = get_sql_string (gpc_film_get_period (infos));
  description = get_sql_string (gpc_film_get_description (infos));
  sql = g_strdup_printf ("update GPC_FILM set "
                         " KIND = %s,"
                         " NUMBER = %s,"
                         " MAKER = %s,"
                         " MODEL = %s,"
                         " ISO = %d,"
                         " SIZE = %d,"
                         " CAMERA_MAKER = %s,"
                         " CAMERA_MODEL = %s,"
                         " PERIOD = %s,"
                         " DESCRIPTION = %s "
                         "where ID = %d",
                         kind,
                         number,
                         maker,
                         model,
                         gpc_film_get_iso (infos),
                         gpc_film_get_size (infos),
                         camera_maker,
                         camera_model,
                         period,
                         description,
                         gpc_film_get_id (infos));
  query = pg_query_new_with_sql (connection, sql);
  result = pg_query_execute (query);
  pg_query_free (query);
  g_free (sql);
  g_free (kind);
  g_free (number);
  g_free (maker);
  g_free (model);
  g_free (camera_maker);
  g_free (camera_model);
  g_free (period);
  g_free (description);
  result = result && gpc_film_infos_store_keywords (connection, infos);
  result = result && gpc_film_store_references (connection, infos);
  result = result && pg_connection_commit (connection);
  if (!result)
    pg_connection_rollback (connection);
  return result;
}


gboolean gpc_film_infos_store_keywords (pgConnection *connection, GpcFilmInfos *infos)
{
  guint   elements;
  guint   index;
  gchar   *code;
  gchar   *sql;
  GList   *keywords;

  pgQuery *query;
  gboolean result;

  if (!connection || !infos || !pg_connection_is_active(connection))
    return FALSE;
  if (!gpc_film_modified)
    return TRUE;
  sql = g_strdup_printf ("delete from GPC_KEYWORD "
                         "where (FILM_ID = %d)",
                         gpc_film_get_id (infos));
  query = pg_query_new_with_sql (connection, sql);
  result = pg_query_execute (query);
  pg_query_free (query);
  g_free (sql);
  keywords = gpc_film_get_keywords (infos);
  elements = g_list_length (keywords);
  for (index = 0; index < elements; index++) {
    code = get_sql_string ((gchar *) g_list_nth_data (keywords, index));
    sql = g_strdup_printf ("insert into GPC_KEYWORD (CODE, FILM_ID, IMAGE_ID) "
                           "values (%s, %d, 0)",
                           code,
                           gpc_film_get_id (infos));
    query = pg_query_new_with_sql (connection, sql);
    result = result && pg_query_execute (query);
    pg_query_free (query);
    g_free (sql);
    g_free (code);
  }
  return result;
}


gboolean gpc_film_store_references (pgConnection *connection, GpcFilmInfos *infos)
{
  gboolean result;
  
  if (!connection || !infos || !pg_connection_is_active (connection))
    return FALSE;
  result = TRUE;  
  if (gpc_added_film_kinds) {
    result = result && gpc_app_store_keywords (connection, "GPC_FILM_KIND", gpc_added_film_kinds);
    g_list_foreach (gpc_added_film_kinds, (GFunc) g_free, NULL);
    g_list_free (gpc_added_film_kinds);
    gpc_added_film_kinds = NULL;
  }
  if (gpc_added_film_makers) {
    result = result && gpc_app_store_keywords (connection, "GPC_FILM_MAKER", gpc_added_film_makers);
    g_list_foreach (gpc_added_film_makers, (GFunc) g_free, NULL);
    g_list_free (gpc_added_film_makers);
    gpc_added_film_makers = NULL;
  }
  if (gpc_added_film_maker_codes) {
    result = result && gpc_app_store_maker_codes (connection, "GPC_FILM_MODEL", gpc_film_get_maker (infos), gpc_added_film_maker_codes);
    g_list_foreach (gpc_added_film_maker_codes, (GFunc) g_free, NULL);
    g_list_free (gpc_added_film_maker_codes);
    gpc_added_film_maker_codes = NULL;
  }
  if (gpc_added_camera_makers) {
    result = result && gpc_app_store_keywords (connection, "GPC_CAMERA_MAKER", gpc_added_camera_makers);
    g_list_foreach (gpc_added_camera_makers, (GFunc) g_free, NULL);
    g_list_free (gpc_added_camera_makers);
    gpc_added_camera_makers = NULL;
  }
  if (gpc_added_camera_maker_codes) {
    result = result && gpc_app_store_maker_codes (connection, "GPC_CAMERA_MODEL", gpc_film_get_camera_maker (infos), gpc_added_camera_maker_codes);
    g_list_foreach (gpc_added_camera_maker_codes, (GFunc) g_free, NULL);
    g_list_free (gpc_added_camera_maker_codes);
    gpc_added_camera_maker_codes = NULL;
  }
  if (gpc_added_keywords) {
    result = result && gpc_app_store_keywords (connection, "GPC_KEY", gpc_added_keywords);
    g_list_foreach (gpc_added_keywords, (GFunc) g_free, NULL);
    g_list_free (gpc_added_keywords);
    gpc_added_keywords = NULL;
  }
  return result;
}


/** Free memory allocated to store film informations */
void gpc_film_infos_free(GpcFilmInfos *infos)
{
  if (!infos)
    return;
  if (infos->pathname)
    g_free (infos->pathname);
  if (infos->kind)
    g_free (infos->kind);
  if (infos->number)
    g_free (infos->number);
  if (infos->maker)
    g_free (infos->maker);
  if (infos->model)
    g_free (infos->model);
  if (infos->camera_maker)
    g_free (infos->camera_maker);
  if (infos->camera_model)
    g_free (infos->camera_model);
  if (infos->period)
    g_free (infos->period);
  if (infos->description)
    g_free (infos->description);
  if (infos->keywords) {
    g_list_foreach (infos->keywords, (GFunc) g_free, NULL);
    g_list_free (infos->keywords);
  }
  g_free(infos);
}


/** Show window with all film informations */
void gpc_film_infos_show(GpcFilmInfos *infos)
{

}


