/*
**  FaceController.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "FaceController.h"

#include "Face.h"
#include "Constants.h"
#include "MailHeaderCell.h"

#include <Pantomime/Message.h>

static FaceController *singleInstance = nil;

@implementation FaceController

//
//
//
- (id) initWithOwner: (id) theOwner
{
  NSBundle *aBundle;
  
  self = [super init];

  owner = theOwner;
 
  aBundle = [NSBundle bundleForClass: [self class]];
  
  resourcePath = [aBundle resourcePath];
  RETAIN(resourcePath);

  allFaceViews = [[NSMutableArray alloc] init];

  cache = [[NSMutableDictionary alloc] init];

  return self;
}


//
//
//
- (void) dealloc
{
  NSDebugLog(@"FaceController: -dealloc");

  RELEASE(resourcePath);
  RELEASE(allFaceViews);
  
  RELEASE(cache);

  [super dealloc];
}



//
//
//
+ (id) singleInstance
{
  NSDebugLog(@"FaceController: -singleInstance");

  if ( !singleInstance )
    {
      singleInstance = [[FaceController alloc] initWithOwner: nil];
    }

  return singleInstance;
}


//
// access / mutation methods
//
- (NSString *) name
{
  return @"Face";
}

- (NSString *) description
{
  return @"This is a simple Face bundle supporting X-Face and X-Image-URL.";
}

- (NSString *) version
{
  return @"v0.2.0";
}

- (void) setOwner: (id) theOwner
{
  owner = theOwner;
}

//
// UI elements
//
- (BOOL) hasPreferencesPanel
{
  return NO;
}

- (BOOL) hasComposeViewAccessory
{
  return NO;
}

- (BOOL) hasViewingViewAccessory
{
  return YES;
}


//
//
//
- (id) viewingViewAccessory
{  
  Face *aFace;

  aFace = [[Face alloc] initWithResourcePath: resourcePath];

  [allFaceViews addObject: aFace];

  return AUTORELEASE(aFace);
}


//
//
//
- (enum ViewingViewType) viewingViewAccessoryType
{
  return ViewingViewTypeHeaderCell;
}


//
//
//
- (void) viewingViewAccessoryWillBeRemovedFromSuperview: (id) theView
{
  if ( theView == nil )
    {
      return;
    }
  else
    {
      Face *aFace;
      int i;
      
      for (i = 0; i < [allFaceViews count]; i++)
	{
	  aFace = [allFaceViews objectAtIndex: i];
	  
	  if ( [theView containsView: aFace] )
	    {
	      [allFaceViews removeObject: aFace];
	      break;
	    }
	}
    }
}


//
//
//
- (void) setCurrentSuperview: (NSView *) theView
{
  superview = theView;
}


//
//
//
- (NSArray *) submenuForMenu: (NSMenu *) theMenu
{
  return nil;
}


//
//
//
- (NSArray *) menuItemsForMenu: (NSMenu *) theMenu
{
  return nil;
}


//
// Pantomime related methods
//
- (void) messageWillBeDisplayed: (Message *) theMessage
			 inView: (NSTextView *) theTextView
{
  NSEnumerator *theEnumerator;
  NSString *aKey;
  Face *aFace;
  
  BOOL hasFoundHeader;
  int i;
  
  hasFoundHeader = NO;

  for (i = 0; i < [allFaceViews count]; i++)
    {      
      // We verify if our header is present.
      theEnumerator = [[theMessage allHeaders] keyEnumerator];
      
      while ( (aKey = [theEnumerator nextObject]) )
	{
	  if ( [aKey caseInsensitiveCompare: @"X-Image-URL"] == NSOrderedSame ||
	       [aKey caseInsensitiveCompare: @"X-Face"] == NSOrderedSame )
	    {
	      hasFoundHeader = YES;
	      break;
	    }
	}

      aFace = [allFaceViews objectAtIndex: i];

      if ( hasFoundHeader  )
	{
	  NSString *aString;
	  
	  aString = [theMessage headerValueForName: aKey];
	  
	  // We search for X-Image-URL / X-Face
	  if ( aString )
	    {
	      NSImage *aImage;
	        
	      // We verify if the image is in our cache. If not, we create it and add it to our cache.
	      aImage = [cache objectForKey: aString];
	      
	      if ( !aImage )
		{
		  // If it's an X-URL-Image
		  if ( [aKey caseInsensitiveCompare: @"X-Image-URL"] == NSOrderedSame )
		    {
		      aImage = [[NSImage alloc] initWithData:
						  [NSData dataWithContentsOfURL: [NSURL URLWithString: aString]]];
		    }
		  else
		    {
		      aImage = [[NSImage alloc] initWithXFaceString: aString];
		    }
		  
		  if ( aImage )
		    {
		      [cache setObject: aImage
			     forKey: aString];
		      
		      RELEASE(aImage);
		    }
		}
	      
	      [aFace setImage: aImage];
	    }
	}
      else
	{
	  [aFace setImage: nil];
	}
	  
      [aFace setNeedsDisplay: YES];
    }
}

@end
