#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# mailmerge.py
#
# DESCRIPTION:
# Base file for the GNUe Reports merge demos
#
# NOTES:
#

import string, sys, re
import xml.sax

from gnue.reports.base.GROutputAdapter import TransformAdapter as Base


class TransformAdapter(Base, xml.sax.ContentHandler):

  FIELD_ENCLOSURE = '::'
  NEWPAGE = "\p"
  NEWLINE = ""
  HEAD_DELIMITER = r"^"
  TAIL_DELIMITER = r"$"
  REGEX = r"\A(.*"            \
          + HEAD_DELIMITER \
          + r")(.+)("           \
          + TAIL_DELIMITER \
          + r".*)\Z"

  COMMENT_BEGIN = "#"
  COMMENT_END = "\n"
  MULTILINE_COMMENTS = 0
  MIMETYPE = 'text/plain'

  def __init__(self, *args, **params):
    Base.__init__(self, *args, **params)
    self.fieldre = re.compile("%s(\\(\\w|[\\[,\\]]\\)+)%s" % \
          (self.FIELD_ENCLOSURE, self.FIELD_ENCLOSURE))
    self.recordCount = 1
    self.newPage = 0

  ###
  ### Methods for TransformAdapter support
  ### (these over-ride TransformAdapter methods)
  ###

  def open(self):

    # We need a temp file
    self.input, self.infile = self.createTempFile()
    return self.input

  def close(self):

    # We are finished with the intermediate file, so
    # close in order for sablotron to be able to open it.
    self.input.close()

    # Get a file for output from the destination adapter.
    outfile = self.destination.getOutputHandle()

    # Get the name of the transform script
    try:
      template = open(self.parameters['template'],'r')
    except KeyError:
      raise "Filter configuration file is missing the 'template=' parameter"

    infile = open(self.infile,'r')


    self.merge(infile, template, outfile)

    infile.close()
    template.close()

    try:
      mimetype = self.parameters['mimetype']
    except:
      mimetype = self.MIMETYPE


    # Let the destination adapter do its thing
    self.destination.close(mimetype=mimetype)

    # clean up our temp files
    self.deleteTempFile(self.infile)


  ###
  ### Methods for Merge support
  ### (these are new methods)
  ###

  def escape(self, s):
    return s

  def printComment (self, out, comment):
    if self.COMMENTABLE:
      if not self.MULTILINE_COMMENTS:
        comments = string.split(comment,"\n")
      else:
        comments=[comment];

      out.write("%s%s%s" % (self.COMMENT_BEGIN,
           string.join(comments, self.COMMENT_END + self.COMMENT_BEGIN),
           self.COMMENT_END))


  def merge (self, data, template, output):
    self.printComment (output, "\n"
                  "================================\n"
                  "This file was generated from the\n"
                  "GNUe Reports / MailMerge Filter.\n"
                  "(http://www.gnue.org/)\n"
                  "================================\n")

    tmpl = string.join(template.readlines(),"")

    reg = re.search(self.REGEX, tmpl, re.DOTALL)
    if reg == None:
      raise "Unable to parse the requested template file."

    header, self.repeating, footer = reg.groups()
    self.output = output

##    print '-'* 80
##    print header
##    print '-'* 80
##    print self.repeating
##    print '-'* 80
##    print footer
##    print '-'* 80


    output.write(header)
    self.printComment(output,
     "GNUe MailMerge [%s]:  Start of Repeating Section"%self.NAME)

    # Create a parser
    parser = xml.sax.make_parser()

    # Tell the parser to use our handler
    parser.setContentHandler(self)
    parser.parse(data)


    # Print out the footer text
    self.printComment(output,
     "GNUe MailMerge [%s]:  End of Repeating Section"%self.NAME)
    output.write(footer)


  # XML handling stuff
  def startElement(self, name, attrs):
    if name == "record":
      self.values = {}
      if self.newPage:
        self.output.write(self.NEWRECORD)
      self.newPage = 1
    self._attrs = attrs
    self._text = ""

  def characters(self, text):
    self._text += text.encode(gConfig('textEncoding'))

  def endElement(self, name):
    if name == "field":
      self.values[string.lower(self._attrs['name'])] = string.replace(
            string.strip(self._text.encode('utf-8')),'\n',self.NEWLINE)
    elif name == "record":
      self.printComment (self.output,
        "GNUe MailMerge [%s]:  Record #%d" % (self.NAME, self.recordCount))

      self.output.write(self.fieldre.sub(self.getFieldValue, self.repeating))

      self.recordCount = self.recordCount + 1

  def getFieldValue(self, matchObject):
    name = string.lower(matchObject.group(1))
    if name[-1] == ']' and '[' in name:
      name, extras = name[:-1].split('[',1)
      extras = extras.split(',')
      format = extras[0]
      extras = extras[1:]
    else:
      format="text"
      extras = []

    try:
      return getattr(self,"as_%s" % format)(name, extras)
    except:
      print "WARNING: this MailMerge driver doesn't support '%s'" % format
      return ""

  def as_text(self, name, extras):
    try:
      return self.escape(self.values[name])
    except KeyError:
      return ""

  def as_barcode(self, name, extras):
    if not hasattr(self,'as_image'):
      print "WARNING: This MailMerge driver doesn't support barcodes"
      return ""


