# GNU Enterprise Common - MaxDB/SAP-DB Driver - Schema Introspection
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Introspection.py 6851 2005-01-03 20:59:28Z jcater $

__all__ = ['Introspection']

import string

from gnue.common.datasources import GIntrospection


# =============================================================================
# This class implements schema introspection for MaxDB/SAP-DB backends
# =============================================================================

class Introspection (GIntrospection.Introspection):

  # list of the types of Schema objects this driver provides
  types =[ ('table',   _('Tables'),       1),
           ('view',    _('Views'),        1),
           ('synonym', _('Synonyms'),     1),
           ('result',  _('Result Table'), 1) ]

  # ---------------------------------------------------------------------------
  # Find a schema element by name or type
  # ---------------------------------------------------------------------------

  def find (self, name = None, type = None):
    """
    This function searches the schema for an element by name and/or type. If no
    name and no type is given, all elements will be retrieved.

    @param name: look for an element with this name
    @param type: look for an element of this type
    @return: A sequence of schema instances, one per element found, or None if
        no element could be found.
    """

    result = []
    cond   = ["TYPE <> 'SYSTEM'"]

    if name is not None:
      cond.append ("TABLENAME = '%s'" % name.upper ())

    if type is not None:
      cond.append ("TABLETYPE = '%s'" % type.upper ())

    cmd = u"SELECT tableid, tablename, tabletype, owner FROM DOMAIN.TABLES " \
           "WHERE %s ORDER BY tablename" % string.join (cond, " AND ")

    cursor = self._connection.makecursor (cmd)
    try:
      for rs in cursor.fetchall ():
        attrs = {'id'        : rs [0],
                 'name'      : rs [1],
                 'type'      : rs [2],
                 'owner'     : rs [3],
                 'indices'   : self.__getIndices (rs [1]),
                 'primarykey': self.__getPrimaryKey (rs [1], rs [2])}

        result.append (GIntrospection.Schema (attrs, self._getChildSchema))

    finally:
      cursor.close ()

    return len (result) and result or None


  # ---------------------------------------------------------------------------
  # Get all fields of a relation
  # ---------------------------------------------------------------------------

  def _getChildSchema (self, parent):
    """
    This function returns a list of all child elements of a given parent
    relation.

    @param parent: schema instance to fetch child elements for
    @return: sequence of schema instances, one per element found
    """

    cmd = u"SELECT columnname, mode, datatype, len, dec, nullable, " \
           "\"DEFAULT\", \"DEFAULTFUNCTION\", pos, keypos " \
           "FROM DOMAIN.COLUMNS " \
           "WHERE tablename = '%s' AND owner = '%s' AND tabletype = '%s' " \
           "ORDER BY columnname" \
           % (parent.name.upper (), parent.owner, parent.type)

    result = []
    cursor = self._connection.makecursor (cmd)

    try:
      for rs in cursor.fetchall ():
        nativetype = rs [2]
        attrs = {'id'        : "%s.%s" % (parent.name, rs [0]),
                 'name'      : rs [0],
                 'type'      : 'field',
                 'nativetype': nativetype,
                 'required'  : rs [5] == 'NO',
                 'pos'       : rs [8],
                 'keypos'    : rs [9]}

        if nativetype in ['DATE', 'TIME', 'TIMESTAMP']:
          attrs ['datatype'] = 'date'

        elif nativetype in ['FIXED', 'FLOAT', 'INTEGER', 'SMALLINT']:
          attrs ['datatype']  = 'number'
          attrs ['length']    = rs [3]

          if nativetype == 'FIXED':
            attrs ['precision'] = rs [4]

        elif nativetype in ['BOOLEAN']:
          attrs ['datatype'] = 'number'

        else:
          attrs ['datatype'] = 'text'
          attrs ['length']   = rs [3]

        if rs [6] is not None:
          attrs ['defaulttype'] = 'constant'
          attrs ['defaultval']  = rs [6]

        elif rs [7] is not None:
          attrs ['defaulttype'] = 'system'
          attrs ['defaultval']  = rs [7]

        result.append (GIntrospection.Schema (attrs))

    finally:
      cursor.close ()

    return result


  # ---------------------------------------------------------------------------
  # Get the primary key of a relation (if it has one)
  # ---------------------------------------------------------------------------

  def __getPrimaryKey (self, tablename, tabletype):
    """
    This function returns a sequence with the primary key fields or None if the
    given relation has no primary key.

    @param tablename: name of the table to fetch the primary key for
    @param tabletype: type of the table to fetch the primary key for

    @return: sequence of primary key fields or None
    """

    cmd = u"SELECT columnname FROM DOMAIN.COLUMNS " \
           "WHERE tablename = '%s' AND tabletype = '%s' AND mode = 'KEY'" \
           "ORDER BY keypos" % (tablename.upper (), tabletype)

    cursor = self._connection.makecursor (cmd)

    try:
      result = [rs [0] for rs in cursor.fetchall ()]

    finally:
      cursor.close ()

    return len (result) and result or None


  # ---------------------------------------------------------------------------
  # Get a dictionary of all indices defined for a relation
  # ---------------------------------------------------------------------------

  def __getIndices (self, tablename):
    """
    This function returns a dictionary with indices of the given table. The
    keys are the index names and the values are dictionaries describing the
    index.

    @param tablename: name of the table to fetch indices for.
    @return: dictionary with indices or None
    """

    cmd = u"SELECT indexname, type FROM INDEXES WHERE tablename = '%s'" \
          % tablename.upper ()

    result = {}
    cursor = self._connection.makecursor (cmd)

    try:
      for rs in cursor.fetchall ():
        result [rs [0]] = {'unique' : rs [1] == 'UNIQUE',
                           'primary': False,
                           'fields' : []}

        cmd = u"SELECT columnname FROM INDEXCOLUMNS " \
               "WHERE tablename = '%s' AND indexname = '%s' " \
               "ORDER BY columnno" % (tablename.upper (), rs [0])

        fcursor = self._connection.makecursor (cmd)
        try:
          result [rs [0]]['fields'] = [fr [0] for fr in fcursor.fetchall ()]

        finally:
          fcursor.close ()

    finally:
      cursor.close ()

    return len (result.keys ()) and result or None
