#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# interbase/DBdriver.py
#
# DESCRIPTION:
# Driver to provide access to data via the Kinterbasdb Interbase/Firebird Python Driver
# Requires Kinterbasdb > 3.0 (http://kinterbasdb.sourceforge.net/)
#
# NOTES:
#
#   Supported attributes (via connections.conf or <database> tag)
#
#     host=      This is the Interbase host for your connection  (required)
#     dbame=      This is the Interbase database to use (required)
#
# $Id: Connection.py 7080 2005-03-02 22:00:12Z reinhard $

__all__ = ['Connection']

####                                   ####
#### IF YOU MODIFY ANY CONNECTION      ####
#### ATTRIBUTES, PLEASE UPDATE info.py ####
####                                   ####

from string import upper, lower, rstrip
import sys
import types
from gnue.common.datasources import Exceptions, GConditions, GConnections
from gnue.common.apps import errors, i18n
from gnue.common.datasources.drivers import DBSIG2
from DataObject import *
from gnue.common.datasources.drivers.interbase.Schema.Discovery.Introspection import Introspection
from gnue.common.datasources.drivers.interbase.Schema.Creation.Creation \
    import Creation

try:
  import kinterbasdb as SIG2api
except ImportError:
  raise GConnections.DependencyError, ('Kinterbasdb', None)


######################################################################
#
#  GConnection object for Interbase drivers
#
class Connection (DBSIG2.Connection):

  _boolean_true      = 1
  _boolean_false     = 0
  _numbers_as_string = False
  defaultBehavior    = Introspection
  defaultCreator     = Creation
  _driver            = SIG2api
  _DatabaseError     = SIG2api.DatabaseError
  supportedDataObjects = {
    'object': DataObject_Object,
    'sql':    DataObject_SQL
  }
  # The date/time format used in insert/select statements
  # (based on format used for time.strftime())
  _dateTimeFormat = "cast('%Y-%m-%d %H:%M:%S' as timestamp)"

  # ---------------------------------------------------------------------------
  # Establish a new connection
  # ---------------------------------------------------------------------------

  def connect(self, connectData={}):

    gDebug (9, "Interbase database driver initializing")

    try:
      ib_encoding = ib_encTable [self._encoding]

    except KeyError:
      gDebug (1, u_("Encoding '%s' is not supported by interbase "
                              "dbdriver. Using default encoding.") % \
                              self._encoding)
      ib_encoding = ''

    if ib_encoding:
      gDebug (9, u_("Setting interbase client_encoding to %(new)s (%(old)s)") \
                 % {'new': ib_encoding, 'old': self._encoding})

    try:
      username = connectData ['_username']
      password = connectData ['_password']

      if isinstance (username, types.UnicodeType):
        username = username.encode (i18n.getencoding ())
      if isinstance (password, types.UnicodeType):
        password = password.encode (i18n.getencoding ())

      self.native = SIG2api.connect ( \
                            user     = username,
                            password = password,
                            charset  = ib_encoding,
                            database = connectData ['dbname'],
                            host     = connectData ['host'])

      # automatically start a new transaction
      self._beginTransaction ()

    except self._DatabaseError:
      raise Exceptions.LoginError, errors.getException () [2]


  # ---------------------------------------------------------------------------
  # Start a new transaction
  # ---------------------------------------------------------------------------

  def _beginTransaction (self):
    self.native.begin ()


  # ---------------------------------------------------------------------------
  # Return the current date, according to database
  # ---------------------------------------------------------------------------

  def getTimeStamp (self):
    return self.__singleQuery ("SELECT CAST('now' AS DATE) FROM rdb$database")


  # ---------------------------------------------------------------------------
  # Return a sequence number from sequence 'name'
  # ---------------------------------------------------------------------------

  def getSequence (self, name):
    return self.__singleQuery ("SELECT gen_id(%s,1) FROM rdb$database" % name)


  # ---------------------------------------------------------------------------
  # Execute an SQL statement for internal use only
  # ---------------------------------------------------------------------------

  def __singleQuery (self, statement):
    cursor = self.native.cursor ()

    try:
      cursor.execute (statement)
      rv = cursor.fetchone ()
      cursor.close ()

    except mesg:
      gDebug (1, "**** Unable to execute extension query")
      gDebug (1, "**** %s" % mesg)

      try:
        cursor.close ()

      except:
        pass

      return None

    try:
      return rv [0]

    except:
      return None


# RDB$CHARACTER_SETS.RDB$CHARACTER_SET_NAME
ib_encTable =  {'ascii'     :  'ASCII',
                ''          :  'BIG_5',
                ''          :  'CYRL',
                'cp437'     :  'DOS437',
                'cp850'     :  'DOS850',
                'cp852'     :  'DOS852',
                'cp857'     :  'DOS857',
                'cp860'     :  'DOS860',
                'cp861'     :  'DOS861',
                'cp863'     :  'DOS863',
                'cp865'     :  'DOS865',
                ''          :  'EUCJ_0208',
                ''          :  'GB_2312',
                'iso8859-1' :  'ISO8859_1',
                'iso8859-2' :  'ISO8859_2',
                'iso8859-3' :  'ISO8859_3',
                'iso8859-4' :  'ISO8859_4',
                'iso8859-5' :  'ISO8859_5',
                'iso8859-6' :  'ISO8859_6',
                'iso8859-7' :  'ISO8859_7',
                'iso8859-8' :  'ISO8859_8',
                'iso8859-9' :  'ISO8859_9',
                'iso8859-13':  'ISO8859_13',
                ''          :  'KSC_5601',
                ''          :  'NEXT',
                ''          :  'NONE',
                ''          :  'OCTETS',
                ''          :  'SJIS_0208',
                'utf-8'     :  'UNICODE_FSS',
                'cp1250'    :  'WIN1250',
                'cp1251'    :  'WIN1251',
                'cp1252'    :  'WIN1252',
                'cp1253'    :  'WIN1253',
                'cp1254'    :  'WIN1254',
                'cp1255'    :  'WIN1255',
                'cp1256'    :  'WIN1256',
                'cp1257'    :  'WIN1257'}
