/**
 ** dynamic_list.c - generation/loading of lists based on statistics and 
 **                  other phenomena
 **
 ** Copyright (C) 2000 Matthew Pratt <mattpratt@yahoo.com>
 **
 ** This software is licensed under the terms of the GNU General 
 ** Public License (GPL). Please see the file LICENSE for details.
 **/ 

#include <string.h>

#include "gnomp3.h"
#include "dynamic_list.h"
#include "mp3list.h"
#include "playlist.h"
#include "xmms_play.h"

/******************************************************************************
 * The dynamic lists genertors of gnomp3 reside here. There exists a framework
 * for easily adding generators simply by adding to this file, instead of
 * having to play with the playlist loading/saving/displaying code.
 *
 * To add a new generator:
 *   Write the generator function. Add an entry to the dynamic_lists array 
 * which conatins the name of your generator and a pointer to the generator
 * function.
 *****************************************************************************/

/*
 * Generate a playlist of a fixed size of randoms songs from all those 
 * available
 */
void dynamic_list_random()
{
    int i;
    int len = g_list_length(mp3list);
    GList *ptr;

    for(i = 0; i < gnomp3.dynamic_list_size; i++){
	ptr = g_list_nth( mp3list, 1 + (int) ((float)len * rand()/(RAND_MAX+1.0)));
	playlist_add_song( ptr->data);
    }
}

/*
 * Load the current playlist from XMMS if available.
 */
void dynamic_list_xmms()
{   
    if(!gnomp3.tight_xmms || !gnomp3.use_xmms)
	return;

    xmms_play_list_load();
}

/*
 * Gets the newsest songs.
 */
int time_cmp(gconstpointer a, gconstpointer b){
    const MP3 *m1 = a, *m2 = b;
    if(m1->file_time > m2->file_time)return -1;
    if(m2->file_time > m1->file_time)return 1;
    return 0;
}

void dynamic_list_newest()
{
    GList *ptr;
    GList *sorted = NULL;
    int i;

    for(ptr = mp3list; ptr; ptr = ptr->next){
	sorted = g_list_insert_sorted(sorted, ptr->data, time_cmp);
    }

    for(ptr = sorted, i = 0; i < gnomp3.dynamic_list_size && ptr; i++){
	playlist_add_song( ptr->data);
	ptr = ptr->next;
    }
}

struct dynamic_list dynamic_lists[] = {
    {"Dynamic list: RANDOM", dynamic_list_random},
    {"Dynamic list: NEWEST", dynamic_list_newest},
#ifdef ENABLE_XMMS
    {"Current XMMS playlist", dynamic_list_xmms},
#endif 
    {NULL,NULL}
};

/******************************************************************************
 * The following functions are the framework for displaying/activating the
 * above list generation functions. They are integrated into the playlist
 * loading/saving/displaying infrastructure of gnomp3.
 *****************************************************************************/

/*
 * Add each name in the array of dynamic lists to given list.
 */
GList *dynamic_list_add_names(GList *list)
{
    int i;
    for(i = 0; dynamic_lists[i].name; i++){
	list = g_list_append(list, dynamic_lists[i].name);
    }
    return list;
}

/*
 * Search the list array of dynamic lists for an entry with its name macthing
 * that given. Return a pointer to the entry if found, else NULL.
 */
struct dynamic_list *dynamic_list_find(char *name)
{
    int i;
    for(i = 0; dynamic_lists[i].name; i++){
	if(!strcmp(dynamic_lists[i].name, name))
	    return &dynamic_lists[i];
    }
    return NULL;
}

/*
 * Activate the dynamic list with the given name.
 */
int dynamic_list_load(char *name)
{
    struct dynamic_list *list;

    list = dynamic_list_find(name);
    if(!list)
	return FALSE;

    strcpy(gnomp3.playlist_loaded, list->name);
    list->load_callback();
    return TRUE;
}


