/*  Gnometab -- a guitar tablature editor for GNOME
    Copyright (C) 2001  William L. Guelker

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <gnome.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include "tab_canvas.h"
#include "chord_builder.h"
#include "dialogs.h"
#include "chord_lib.h"
#include "tab_objects.h"
#include "chord_canvas.h"
#include "tab_doc.h"

/* callbacks */

static void
dialog_destroyed (GtkObject *obj,  void **dialog_pointer)
{
    if (dialog_pointer != NULL)
    {
        g_free (*dialog_pointer);
        *dialog_pointer = NULL;
    }
}

static void
slur_length_response          (GtkDialog       *dlg, gint res_id,
                                        GtabDialogSlur *gtab_dialog)
{
    static gint slur_type = 0;
    
    switch (res_id) {
        case GTK_RESPONSE_OK:
            
            if (gtk_toggle_button_get_active
                    (GTK_TOGGLE_BUTTON(gtab_dialog->radio_hammer)))slur_type = SLUR_HAMMER;
            
            if (gtk_toggle_button_get_active
                    (GTK_TOGGLE_BUTTON(gtab_dialog->radio_pulloff)))slur_type = SLUR_PULLOFF;
            
            if (gtk_toggle_button_get_active
                    (GTK_TOGGLE_BUTTON(gtab_dialog->radio_slide)))    slur_type = SLUR_SLIDE;
            
            if (gtk_toggle_button_get_active
                    (GTK_TOGGLE_BUTTON(gtab_dialog->radio_trill)))    slur_type = SLUR_TRILL;
            
            create_slur(gtab_dialog->tab_doc, gtab_dialog->x, gtab_dialog->y, slur_type,
                        gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
                            (gtab_dialog->spin_slur_length)));
            break;
        default:
            
            break;
    }        
    gtk_widget_destroy(gtab_dialog->dialog);
}

static void
bar_size_response                    (GtkDialog *dlg, gint res_id,
                                        GtabDialogBar *gtab_dialog)
{
    gint bar_size = 0;
    
    switch (res_id) {
        case GTK_RESPONSE_OK:
            bar_size = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
                                            (gtab_dialog->spin_bar_size));
            
            switch (gtab_dialog->tab_doc->parent->current_tool)    {
                case TOOL_BAR:
                    create_bar(gtab_dialog->tab_doc, gtab_dialog->x, gtab_dialog->y, bar_size);
        
                    break;
                    
                case TOOL_LEGATO:
                    create_legato(gtab_dialog->tab_doc, gtab_dialog->x, gtab_dialog->y,
                                    bar_size);
                    break;
                    
            }
            
            break;
            
        default:
            break;
    }
    gtk_widget_destroy(gtab_dialog->dialog);
}

static void
chord_bar_size_response                    (GtkDialog *dlg, gint res_id,
                                        GtabDialogBar *gtab_dialog)
{
    GtabChordBuilder *builder;
    
    gint bar_size = 0;
    builder = g_object_get_data(G_OBJECT(gtab_dialog->dialog), "builder");
    
    switch (res_id) {
        case GTK_RESPONSE_OK:
            bar_size = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
                                            (gtab_dialog->spin_bar_size));
        
            create_chord_bar(popup_holder_x2, popup_holder_y2 + 8.5,
                                    bar_size, builder);
            
            break;
            
        default:
            break;
    }
    
    gtk_widget_destroy(gtab_dialog->dialog);
}

static void
strum_size_response                    (GtkDialog *dlg, gint res_id,
                                        GtabDialogStrum *gtab_dialog)
{
    gboolean strum_upstroke = FALSE;
    
    switch (res_id) {
        case GTK_RESPONSE_OK:
            if (gtk_toggle_button_get_active
                    (GTK_TOGGLE_BUTTON(gtab_dialog->radio_up))) {
                strum_upstroke = TRUE;
            }
                        
            create_strum(gtab_dialog->tab_doc, gtab_dialog->x, gtab_dialog->y, strum_upstroke,
                gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
                                            (gtab_dialog->spin_strum_size)));
            break;
        default:
            break;
    }
    
    gtk_widget_destroy(gtab_dialog->dialog);
    
}

static void
props_response                    (GtkDialog *dlg, gint res_id,
                                        GtabDialogProps *gtab_dialog)
{
    
    switch (res_id) {
        case GTK_RESPONSE_OK:
            xmlSetProp(gtab_dialog->tab_doc->doc_props, "title",
                        gtk_entry_get_text(GTK_ENTRY(gtab_dialog->title)));
            xmlSetProp(gtab_dialog->tab_doc->doc_props, "artist", 
                        gtk_entry_get_text(GTK_ENTRY(gtab_dialog->artist)));
            xmlSetProp(gtab_dialog->tab_doc->doc_props, "tuning", 
                        gtk_entry_get_text(GTK_ENTRY(gtab_dialog->tuning)));
            /*
            xmlSetProp(gtab_dialog->tab_doc->doc_props, "orientation",
                    g_object_get_data(G_OBJECT(GTK_OPTION_MENU(gtab_dialog->orient)->menu_item), 
                                        "orient"));
            */
        
            create_doc_props(gtab_dialog->tab_doc);
        
            break;
        default:
            break;
    }
    
    gtk_widget_destroy(gtab_dialog->dialog);
    
}

static void
timesig_response                    (GtkDialog *dlg, gint res_id,
                                        GtabDialogTimesig *gtab_dialog)
{
    gint beats;
    gint beat_val;
    
    switch (res_id) {
        case GTK_RESPONSE_OK:
            
            beats = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
                                                (gtab_dialog->beats));
            beat_val = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
                                                (gtab_dialog->beatval));

            create_timesig(gtab_dialog->tab_doc, gtab_dialog->x, gtab_dialog->y, beats, beat_val);
        
            break;
        default:
            break;
    }
    
    gtk_widget_destroy(gtab_dialog->dialog);
    
}

static void
comment_response                    (GtkDialog *dlg, gint res_id,
                                        GtabDialogComment *gtab_dialog)
{
    switch (res_id) {
        case GTK_RESPONSE_OK:
            
            create_comment(gtab_dialog->tab_doc, gtab_dialog->x, gtab_dialog->y, 
                        gtk_entry_get_text(GTK_ENTRY(gtab_dialog->comment)));
        
            break;
        default:
            break;
    }
    
    gtk_widget_destroy(gtab_dialog->dialog);
    
}

static void
rest_response                    (GtkDialog *dlg, gint res_id,
                                        GtabDialogRest *gtab_dialog)
{
    gboolean dotted;
    
    switch (res_id) {
        case GTK_RESPONSE_OK:
            
            dotted = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
                                                    (gtab_dialog->dotted));
    
            create_rest(gtab_dialog->tab_doc, gtab_dialog->x, gtab_dialog->y, 
                get_rest_value(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO
                                    (gtab_dialog->length)->entry))), dotted);
        
            break;
        default:
            break;
    }
    
    gtk_widget_destroy(gtab_dialog->dialog);
    
}

/* dialog creation functions */

GtabDialogSlur*
create_slur_length_popup (GtabDoc *doc)
{
    static GtabDialogSlur  *dialog = NULL;
    GtkWidget         *label;
    GtkWidget         *hbox;
    GtkWidget         *vbox;
    GtkObject         *adj;
    GSList *slur_type_group = NULL;

    dialog = g_new0(GtabDialogSlur, 1);
    
    dialog->dialog = gtk_dialog_new_with_buttons (_("Slur Length"), 
                                                    GTK_WINDOW(doc->parent->gnometab),
                                                    GTK_DIALOG_DESTROY_WITH_PARENT && 
                                                    GTK_DIALOG_MODAL,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_CANCEL,
                                                    NULL);
    
    gtk_dialog_set_default_response(GTK_DIALOG(dialog->dialog), GTK_RESPONSE_OK);
 
    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), 8);
 
    hbox = gtk_hbox_new (FALSE, 0);
 
    gtk_widget_show (hbox);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

    label = gtk_label_new (_("Number of Positions: "));

    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

    adj = gtk_adjustment_new (1, 1, 50, 1, 10, 10);
    dialog->spin_slur_length = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);
  
    gtk_widget_show (dialog->spin_slur_length);
    gtk_box_pack_start (GTK_BOX (hbox), dialog->spin_slur_length, TRUE, TRUE, 0);

    dialog->radio_hammer = gtk_radio_button_new_with_label (slur_type_group, _("Hammer-On"));
    slur_type_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (dialog->radio_hammer));

    gtk_widget_show (dialog->radio_hammer);
    gtk_box_pack_start (GTK_BOX (vbox), dialog->radio_hammer, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->radio_hammer), TRUE);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (dialog->radio_hammer), FALSE);

    dialog->radio_pulloff = gtk_radio_button_new_with_label (slur_type_group, _("Pull-Off"));
    slur_type_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (dialog->radio_pulloff));
 
    gtk_widget_show (dialog->radio_pulloff);
    gtk_box_pack_start (GTK_BOX (vbox), dialog->radio_pulloff, FALSE, FALSE, 0);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (dialog->radio_pulloff), FALSE);

    dialog->radio_slide = gtk_radio_button_new_with_label (slur_type_group, _("Slide"));
    slur_type_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (dialog->radio_slide));

    gtk_widget_show (dialog->radio_slide);
    gtk_box_pack_start (GTK_BOX (vbox), dialog->radio_slide, FALSE, FALSE, 0);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (dialog->radio_slide), FALSE);

    dialog->radio_trill = gtk_radio_button_new_with_label (slur_type_group, _("Trill"));
    slur_type_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (dialog->radio_trill));

    gtk_widget_show (dialog->radio_trill);
    gtk_box_pack_start (GTK_BOX (vbox), dialog->radio_trill, FALSE, FALSE, 0);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (dialog->radio_trill), FALSE);

    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
 
    dialog->tab_doc = doc;
    
      g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                  G_CALLBACK(dialog_destroyed), &dialog);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "response",
                  G_CALLBACK(slur_length_response), dialog);
                  
    
    return dialog;
}

GtabDialogBar*
create_bar_size_popup (GtabDoc *doc)
{
    static GtabDialogBar  *dialog = NULL;
    GtkWidget         *label;
    GtkWidget         *hbox;
    GtkWidget         *vbox;
    GtkObject         *adj;
    
    dialog = g_new0(GtabDialogBar, 1);
    
    dialog->dialog = gtk_dialog_new_with_buttons (_("Bar Size"), 
                                                  GTK_WINDOW(doc->parent->gnometab),
                                                    GTK_DIALOG_DESTROY_WITH_PARENT && GTK_DIALOG_MODAL,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_CANCEL,
                                                    NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog->dialog), GTK_RESPONSE_OK);
    
    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), 8);
 
    hbox = gtk_hbox_new (FALSE, 0);
 
    gtk_widget_show (hbox);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

    label = gtk_label_new (_("Number of Strings: "));

    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

    adj = gtk_adjustment_new (1, 1, 50, 1, 10, 10);
    dialog->spin_bar_size = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);
  
    gtk_widget_show (dialog->spin_bar_size);
    gtk_box_pack_start (GTK_BOX (hbox), dialog->spin_bar_size, TRUE, TRUE, 0);

    dialog->tab_doc = doc;
    
      g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                  G_CALLBACK(dialog_destroyed), &dialog);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "response",
                  G_CALLBACK(bar_size_response), dialog);
                  
    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
    
    return dialog;
}

GtabDialogBar*
create_chord_bar_size_popup (GtabChordBuilder *chord_dialog)
{
    static GtabDialogBar  *dialog = NULL;
    GtkWidget         *label;
    GtkWidget         *hbox;
    GtkWidget         *vbox;
    GtkObject         *adj;
    
    dialog = g_new0(GtabDialogBar, 1);
    
    dialog->dialog = gtk_dialog_new_with_buttons (_("Bar Size"), 
                                                  GTK_WINDOW(chord_dialog->dialog),
                                                    GTK_DIALOG_DESTROY_WITH_PARENT && GTK_DIALOG_MODAL,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_CANCEL,
                                                    NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog->dialog), GTK_RESPONSE_OK);
    g_object_set_data(G_OBJECT(dialog->dialog), "builder", chord_dialog);
    
    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), 8);
 
    hbox = gtk_hbox_new (FALSE, 0);
 
    gtk_widget_show (hbox);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

    label = gtk_label_new (_("Number of Strings: "));

    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

    adj = gtk_adjustment_new (1, 1, 50, 1, 10, 10);
    dialog->spin_bar_size = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);
  
    gtk_widget_show (dialog->spin_bar_size);
    gtk_box_pack_start (GTK_BOX (hbox), dialog->spin_bar_size, TRUE, TRUE, 0);

      g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                  G_CALLBACK(dialog_destroyed), &dialog);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "response",
                  G_CALLBACK(chord_bar_size_response), dialog);
                  
    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
    
    return dialog;
}

GtabDialogStrum*
create_strum_size_select (GtabDoc *doc)
{
    static GtabDialogStrum  *dialog = NULL;
    GtkWidget         *label;
    GtkWidget         *hbox1;
    GtkWidget         *hbox2;
    GtkWidget         *vbox;
    GtkObject         *adj;
    GSList *strum_type_group = NULL;

    dialog = g_new0(GtabDialogStrum, 1);
    
    dialog->dialog = gtk_dialog_new_with_buttons (_("Strum Parameters"), 
                                                    GTK_WINDOW(doc->parent->gnometab),
                                                    GTK_DIALOG_DESTROY_WITH_PARENT && 
                                                    GTK_DIALOG_MODAL,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_CANCEL,
                                                    NULL);
    
    gtk_dialog_set_default_response(GTK_DIALOG(dialog->dialog), GTK_RESPONSE_OK);
 
    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), 8);
 
    hbox1 = gtk_hbox_new (FALSE, 0);
 
    gtk_widget_show (hbox1);
    gtk_box_pack_start (GTK_BOX (vbox), hbox1, FALSE, FALSE, 0);

    label = gtk_label_new (_("Number of Strings: "));

    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 0);

    adj = gtk_adjustment_new (1, 1, 50, 1, 10, 10);
    dialog->spin_strum_size = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);
  
    gtk_widget_show (dialog->spin_strum_size);
    gtk_box_pack_start (GTK_BOX (hbox1), dialog->spin_strum_size, TRUE, TRUE, 0);

    hbox2 = gtk_hbox_new(FALSE, 8);
    gtk_widget_show(hbox2);
    
    gtk_box_pack_start(GTK_BOX (vbox), hbox2, FALSE, FALSE, 0);
    
    dialog->radio_down = gtk_radio_button_new_with_label (strum_type_group, _("Down Stroke"));
    strum_type_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (dialog->radio_down));

    gtk_widget_show (dialog->radio_down);
    gtk_box_pack_start (GTK_BOX (hbox2), dialog->radio_down, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->radio_down), TRUE);
    
    dialog->radio_up = gtk_radio_button_new_with_label (strum_type_group, _("Up Stroke"));
    strum_type_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (dialog->radio_up));

    gtk_widget_show (dialog->radio_up);
    gtk_box_pack_start (GTK_BOX (hbox2), dialog->radio_up, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->radio_up), FALSE);

      g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                  G_CALLBACK(dialog_destroyed), &dialog);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "response",
                  G_CALLBACK(strum_size_response), dialog);
                  
    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
 
    dialog->tab_doc = doc;
    
    return dialog;
}

GtabDialogProps*
create_props_popup (GtabDoc *doc)
{
    static GtabDialogProps  *dialog = NULL;
    GtkWidget         *label1;
    GtkWidget         *label2;
    GtkWidget         *label3;
    GtkWidget         *table;
    GtkWidget         *vbox;
    /* GtkWidget         *menu;
    GtkWidget         *menuitem1;
    GtkWidget         *menuitem2;
    gchar             *orient; */
    
    dialog = g_new0(GtabDialogProps, 1);
    
    dialog->dialog = gtk_dialog_new_with_buttons (_("Properties"), 
                                                    GTK_WINDOW(doc->parent->gnometab),
                                                    GTK_DIALOG_DESTROY_WITH_PARENT && 
                                                    GTK_DIALOG_MODAL,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_CANCEL,
                                                    NULL);
    
    gtk_dialog_set_default_response(GTK_DIALOG(dialog->dialog), GTK_RESPONSE_OK);
 
    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), 8);
 
    table = gtk_table_new (3, 2, FALSE);
    gtk_widget_show(table);
    gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 0);
    gtk_table_set_row_spacings (GTK_TABLE (table), 6);
    gtk_table_set_col_spacings (GTK_TABLE (table), 6);
    
    
    dialog->title = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(dialog->title), 50);
    
    gtk_widget_show(dialog->title);
    gtk_table_attach (GTK_TABLE (table), dialog->title, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
    
    dialog->artist = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(dialog->artist), 50);
    
    gtk_widget_show(dialog->artist);
    gtk_table_attach (GTK_TABLE (table), dialog->artist, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
    
    dialog->tuning = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(dialog->tuning), 50);
    
    gtk_widget_show(dialog->tuning);
    gtk_table_attach (GTK_TABLE (table), dialog->tuning, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
    
    
    label1 = gtk_label_new (_("Title: "));
    gtk_widget_show (label1);
    gtk_table_attach (GTK_TABLE (table), label1, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_label_set_justify (GTK_LABEL (label1), GTK_JUSTIFY_RIGHT);
    
    
    label2 = gtk_label_new (_("Artist: "));
    gtk_widget_show (label2);
    gtk_table_attach (GTK_TABLE (table), label2, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_label_set_justify (GTK_LABEL (label2), GTK_JUSTIFY_RIGHT);
    
    label3 = gtk_label_new (_("Tuning: "));
    gtk_widget_show (label3);
    gtk_table_attach (GTK_TABLE (table), label3, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_label_set_justify (GTK_LABEL (label3), GTK_JUSTIFY_RIGHT);

    /*
    
    label1 = gtk_label_new(_("Orientation: "));
    gtk_widget_show(label1);
    gtk_table_attach (GTK_TABLE (table), label1, 0, 1, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_label_set_justify (GTK_LABEL (label1), GTK_JUSTIFY_RIGHT);
    
    orient = get_prop(doc->doc_props, "orientation");
        
    menu = gtk_menu_new();
    gtk_widget_show(menu);
    
    menuitem1 = gtk_menu_item_new_with_label(_("Portrait (6 staffs/page)"));
    gtk_widget_show(menuitem1);
    g_object_set_data(G_OBJECT(menuitem1), "orient", "portrait");
    
    menuitem2 = gtk_menu_item_new_with_label(_("Landscape (5 staffs/page)"));
    gtk_widget_show(menuitem2);
    g_object_set_data(G_OBJECT(menuitem2), "orient", "landscape");
    
    if (orient == NULL) {
        gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem1);
        gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem2);
    }
    else {
        if (g_strcasecmp(orient, "landscape") == 0) {
            gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem2);
            gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem1);
        }
        else {
            gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem1);
            gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem2);
        }
    }
    
    g_free(orient);
    
    dialog->orient = gtk_option_menu_new();
    gtk_widget_show(dialog->orient);
    gtk_option_menu_set_menu(GTK_OPTION_MENU(dialog->orient), menu);
    
    gtk_table_attach (GTK_TABLE (table), dialog->orient, 1, 2, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    */
    
    g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                  G_CALLBACK(dialog_destroyed), &dialog);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "response",
                  G_CALLBACK(props_response), dialog);
                  
    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
 
    dialog->tab_doc = doc;
    
    return dialog;
}

GtabDialogTimesig*
create_timesig_popup (GtabDoc *doc)
{
    static GtabDialogTimesig  *dialog = NULL;
    GtkWidget         *label1;
    GtkWidget         *label2;
    GtkWidget         *hbox1;
    GtkWidget         *hbox2;
    GtkWidget         *vbox;
    GtkObject         *adj1;
    GtkObject         *adj2;

    dialog = g_new0(GtabDialogTimesig, 1);
    
    dialog->dialog = gtk_dialog_new_with_buttons (_("Properties"), 
                                                    GTK_WINDOW(doc->parent->gnometab),
                                                    GTK_DIALOG_DESTROY_WITH_PARENT && 
                                                    GTK_DIALOG_MODAL,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_CANCEL,
                                                    NULL);
    
    gtk_dialog_set_default_response(GTK_DIALOG(dialog->dialog), GTK_RESPONSE_OK);
 
    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), 8);
 
    hbox1 = gtk_hbox_new(TRUE, 0);
    gtk_widget_show (hbox1);
    gtk_box_pack_start (GTK_BOX (vbox), hbox1, FALSE, FALSE, 0);
    
    label1 = gtk_label_new (_("Beats Per Measure: "));
    
    gtk_widget_show (label1);
    gtk_box_pack_start (GTK_BOX (hbox1), label1, FALSE, FALSE, 0);
    gtk_label_set_justify (GTK_LABEL (label1), GTK_JUSTIFY_RIGHT);
    
    adj1 = gtk_adjustment_new (4, 1, 50, 1, 10, 10);
    dialog->beats = gtk_spin_button_new (GTK_ADJUSTMENT (adj1), 1, 0);
    gtk_widget_show (dialog->beats);
    gtk_box_pack_start (GTK_BOX (hbox1), dialog->beats, TRUE, TRUE, 0);
    
    hbox2 = gtk_hbox_new(TRUE, 0);
    gtk_widget_show (hbox2);
    gtk_box_pack_start (GTK_BOX (vbox), hbox2, TRUE, TRUE, 0);
    
    label2 = gtk_label_new (_("Beat Value: "));
    gtk_widget_show (label2);
    gtk_box_pack_start (GTK_BOX (hbox2), label2, FALSE, FALSE, 0);
    gtk_label_set_justify (GTK_LABEL (label2), GTK_JUSTIFY_RIGHT);
    
    adj2 = gtk_adjustment_new (4, 1, 50, 1, 10, 10);
    dialog->beatval = gtk_spin_button_new (GTK_ADJUSTMENT (adj2), 1, 0);
    gtk_widget_show (dialog->beatval);
    gtk_box_pack_start (GTK_BOX (hbox2), dialog->beatval, TRUE, TRUE, 0);

    g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                  G_CALLBACK(dialog_destroyed), &dialog);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "response",
                  G_CALLBACK(timesig_response), dialog);
                  
    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
 
    dialog->tab_doc = doc;
    
    return dialog;
}

GtabDialogComment*
create_comment_popup (GtabDoc *doc)
{
    static GtabDialogComment  *dialog = NULL;
    GtkWidget         *vbox;

    dialog = g_new0(GtabDialogComment, 1);
    
    dialog->dialog = gtk_dialog_new_with_buttons (_("Comment"), 
                                                    GTK_WINDOW(doc->parent->gnometab),
                                                    GTK_DIALOG_DESTROY_WITH_PARENT && 
                                                    GTK_DIALOG_MODAL,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_CANCEL,
                                                    NULL);
    
    gtk_dialog_set_default_response(GTK_DIALOG(dialog->dialog), GTK_RESPONSE_OK);
 
    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), 8);
 
    dialog->comment = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(dialog->comment), 100);
    
    gtk_widget_show (dialog->comment);
    gtk_box_pack_start (GTK_BOX (vbox), dialog->comment, FALSE, FALSE, 0);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                  G_CALLBACK(dialog_destroyed), &dialog);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "response",
                  G_CALLBACK(comment_response), dialog);
                  
    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
    
    gtk_widget_grab_focus(dialog->comment);
    
    dialog->tab_doc = doc;
    
    return dialog;
}

GtabDialogRest*
create_rest_popup (GtabDoc *doc)
{
    static GtabDialogRest  *dialog = NULL;
    GtkWidget         *vbox;
    GtkWidget         *table;
    GtkWidget         *label;
    GList             *value_items = NULL;
    
    dialog = g_new0(GtabDialogRest, 1);
    
    dialog->dialog = gtk_dialog_new_with_buttons (_("Comment"), 
                                                    GTK_WINDOW(doc->parent->gnometab),
                                                    GTK_DIALOG_DESTROY_WITH_PARENT && 
                                                    GTK_DIALOG_MODAL,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_CANCEL,
                                                    NULL);
    
    gtk_dialog_set_default_response(GTK_DIALOG(dialog->dialog), GTK_RESPONSE_OK);
 
    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), 8);
 
    table = gtk_table_new (1, 3, FALSE);
    gtk_widget_show (table);
    gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 0);
    gtk_table_set_row_spacings (GTK_TABLE (table), 6);
    gtk_table_set_col_spacings (GTK_TABLE (table), 6);
 
    label = gtk_label_new (_("Value:"));
    
    gtk_widget_show (label);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);
    gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
    gtk_misc_set_padding (GTK_MISC (label), 9, 0);
    
    dialog->length = gtk_combo_new ();
    
    gtk_widget_show (dialog->length);
    gtk_table_attach (GTK_TABLE (table), dialog->length, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
                    
    gtk_combo_set_value_in_list (GTK_COMBO (dialog->length), TRUE, FALSE);
    value_items = g_list_append (value_items, (gpointer) _("whole"));
    value_items = g_list_append (value_items, (gpointer) _("1/2"));
    value_items = g_list_append (value_items, (gpointer) _("1/4"));
    value_items = g_list_append (value_items, (gpointer) _("1/8"));
    value_items = g_list_append (value_items, (gpointer) _("1/16"));
    value_items = g_list_append (value_items, (gpointer) _("1/32"));
    gtk_combo_set_popdown_strings (GTK_COMBO (dialog->length), value_items);
    g_list_free (value_items);
    
    dialog->dotted = gtk_toggle_button_new_with_label (_("Dotted"));
    gtk_widget_show (dialog->dotted);
    gtk_table_attach (GTK_TABLE (table), dialog->dotted, 2, 3, 0, 1,
                    (GtkAttachOptions) (0),
                    (GtkAttachOptions) (0), 0, 0);
        
    g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                  G_CALLBACK(dialog_destroyed), &dialog);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "response",
                  G_CALLBACK(rest_response), dialog);
                  
    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
    
    dialog->tab_doc = doc;
    
    return dialog;
    
}

/* PREFS DIALOG STUFF */

static void
on_cas_toggled                     (GtkToggleButton *button,
                                    GtabDialogPrefs *gtab_dialog)
{
    gtab_dialog->cas_is_changed = TRUE;
}

static void
on_font_changed                    (GnomeFontPicker *font_picker,
                                    GtabDialogPrefs *gtab_dialog)
{
    gtab_dialog->font_is_changed = TRUE;
}

static void 
on_lib_changed                     (GtkButton *button,
                                    GtabDialogPrefs *gtab_dialog)
{
    gtab_dialog->lib_is_changed = TRUE;
}

static void 
on_new_library_clicked            (GtkButton *button, 
                                    GtabDialogPrefs *gtab_dialog)
{
    GtkWidget *dlg;
    gint response;
    const gchar *file;
    gboolean test;
    
    dlg = gtk_file_selection_new("Enter New Chord Library Filename");
    response = gtk_dialog_run(GTK_DIALOG(dlg));
        
    if (response == GTK_RESPONSE_OK) {
        file = gtk_file_selection_get_filename(GTK_FILE_SELECTION(dlg));
        test = g_file_test(file, G_FILE_TEST_EXISTS);
        if (!test && strlen(file) > 1) {
            gnome_file_entry_set_filename(GNOME_FILE_ENTRY(gtab_dialog->current_chord_lib), 
                                gtk_file_selection_get_filename(GTK_FILE_SELECTION(dlg)));
            gtab_dialog->lib_is_changed = TRUE;
        }
        else {
            if (test) {
                gtab_warn(gtab_dialog->app, "Chord Library file already exists!");
            }
        }
    }
    gtk_widget_destroy(dlg);
    
}

static void
on_default_font_clicked         (GtkButton *button,
                                    GtabDialogPrefs *gtab_dialog)
{
    gint font;
    gboolean result;
    
    font = atoi(g_object_get_data(G_OBJECT(button), "font"));
    
    switch (font) {
        case 0:
            result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(gtab_dialog->tab_font), 
                                                        "Sans 10");
            gtab_dialog->font_is_changed = TRUE;
            break;
        case 1:
            result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(gtab_dialog->tab_font_large),
                                                        "Sans 14");
            gtab_dialog->font_is_changed = TRUE;
            break;
        case 2:
            result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(gtab_dialog->tab_font_small),
                                                        "Sans 8");
            gtab_dialog->font_is_changed = TRUE;
            break;
        case 3:
            result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(gtab_dialog->tab_font_tiny),
                                                        "Sans 7");
            gtab_dialog->font_is_changed = TRUE;
            break;
        case 4:
            result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(gtab_dialog->chrd_bldr_font),
                                                        "Sans 18");
            gtab_dialog->font_is_changed = TRUE;
            break;
        case 5:
            result = gnome_font_picker_set_font_name(GNOME_FONT_PICKER(gtab_dialog->chrd_bldr_font_small),
                                                        "Sans 10");
            gtab_dialog->font_is_changed = TRUE;
            break;
        default:
            break;
    }
}

static void
prefs_response                    (GtkDialog *dlg, gint res_id,
                                    GtabDialogPrefs *gtab_dialog)
{
    GConfClient *client;
    gchar             *chord_lib = NULL;
    const gchar             *tab_font;
    const gchar             *tab_font_large;
    const gchar             *tab_font_small;
    const gchar             *tab_font_tiny;
    const gchar             *chrd_bldr_font;
    const gchar             *chrd_bldr_font_small;
    gboolean                 file_exists;
    gboolean                lib_is_visible;
    gboolean                 canvas_needs_refresh = FALSE;
    gint                     i;
    GtkProgressBar             *progress;
    
    if (res_id == GTK_RESPONSE_OK) {
        client = gconf_client_get_default();
        
        if (gtab_dialog->lib_is_changed) {
            chord_lib = gnome_file_entry_get_full_path(GNOME_FILE_ENTRY
                                                    (gtab_dialog->current_chord_lib), 
                                                    FALSE);
            if (chord_lib != NULL) {
                file_exists = g_file_test(chord_lib,
                                        G_FILE_TEST_EXISTS && G_FILE_TEST_IS_REGULAR);
                if (file_exists) {
                    
                    lib_is_visible = gtk_toggle_button_get_active
                                        (GTK_TOGGLE_BUTTON(gtab_dialog->app->btn_lib_chord));
                    free_chord_lib(gtab_dialog->app->chord_lib);
                    load_chord_lib(gtab_dialog->app, chord_lib, lib_is_visible);
                }
                else {
                    file_exists = g_file_test(chord_lib, G_FILE_TEST_EXISTS);
                    if (!file_exists && strlen(chord_lib) > 1) {
                        i = new_chord_lib_file(gtab_dialog->app->chord_lib, chord_lib);
                        if (i) {
                            lib_is_visible = gtk_toggle_button_get_active
                                        (GTK_TOGGLE_BUTTON(gtab_dialog->app->btn_lib_chord));
                            free_chord_lib(gtab_dialog->app->chord_lib);
                            load_chord_lib(gtab_dialog->app, chord_lib, lib_is_visible);
                        }
                    }
                    else {
                        gtab_warn(gtab_dialog->app, "Chord Library file is not valid!");
                    }
                }
                //g_free(chord_lib);
                
            }
        }
        
        if (gtab_dialog->cas_is_changed) {
            chords_above_staff = gtk_toggle_button_get_active
                                (GTK_TOGGLE_BUTTON(gtab_dialog->chords_above_staff));
            gconf_client_set_bool(client, GCONF_CHORDS_ABOVE_STAFF, 
                                    chords_above_staff, NULL);
            canvas_needs_refresh = TRUE;
        }
        
        if (gtab_dialog->font_is_changed) {
                
            tab_font = gnome_font_picker_get_font_name(GNOME_FONT_PICKER(gtab_dialog->tab_font));
            tab_font_large = gnome_font_picker_get_font_name(GNOME_FONT_PICKER(gtab_dialog->tab_font_large));
            tab_font_small = gnome_font_picker_get_font_name(GNOME_FONT_PICKER(gtab_dialog->tab_font_small));
            tab_font_tiny = gnome_font_picker_get_font_name(GNOME_FONT_PICKER(gtab_dialog->tab_font_tiny));
            chrd_bldr_font = gnome_font_picker_get_font_name(GNOME_FONT_PICKER(gtab_dialog->chrd_bldr_font));
            chrd_bldr_font_small = gnome_font_picker_get_font_name(GNOME_FONT_PICKER(gtab_dialog->chrd_bldr_font_small));
            
            gconf_client_set_string(client, GCONF_TAB_FONT, tab_font, NULL);
            gconf_client_set_string(client, GCONF_TAB_FONT_LARGE, tab_font_large, NULL);
            gconf_client_set_string(client, GCONF_TAB_FONT_SMALL, tab_font_small, NULL);
            gconf_client_set_string(client, GCONF_TAB_FONT_TINY, tab_font_tiny, NULL);
            gconf_client_set_string(client, GCONF_CHRD_BLDR_FONT, chrd_bldr_font, NULL);
            gconf_client_set_string(client, GCONF_CHRD_BLDR_FONT_SMALL, chrd_bldr_font_small, NULL);
            
            refresh_fonts(gtab_dialog->app);
            
            canvas_needs_refresh = TRUE;
            
        }
        
        if (canvas_needs_refresh) {
            sync_loaded_doc(((GtabDoc *)gtab_dialog->app->current_doc));
        
            xmlFreeDoc(((GtabDoc *)gtab_dialog->app->current_doc)->current_doc);
            init_current_doc(((GtabDoc *)gtab_dialog->app->current_doc));
            progress = gnome_appbar_get_progress(GNOME_APPBAR
                            (gtab_dialog->app->appbar));
            
            refresh_canvas(((GtabDoc *)gtab_dialog->app->current_doc), FALSE);
            
            gnome_app_flash(GNOME_APP(gtab_dialog->app->gnometab), 
                                                "Loading page...");
            
            render_current_page(((GtabDoc *)gtab_dialog->app->current_doc), progress);
            
        }
    }
    gtk_widget_destroy(gtab_dialog->dialog);
    
}

GtabDialogPrefs*
create_prefs_popup(GtabApp *app)
{
    static GtabDialogPrefs        *dialog = NULL;
    GtkWidget             *label;
    GtkWidget             *frame;
    GtkWidget             *hbox;
    GtkWidget             *vbox;
    GtkWidget             *button;
    GtkWidget             *table;
    
    dialog = g_new0(GtabDialogPrefs, 1);
    
    dialog->dialog = gtk_dialog_new_with_buttons (_("Gnometab Preferences"), 
                                                    GTK_WINDOW(app->gnometab),
                                                    GTK_DIALOG_DESTROY_WITH_PARENT,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    NULL);
    
    gtk_dialog_set_default_response(GTK_DIALOG(dialog->dialog), GTK_RESPONSE_OK);
 
    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    gtk_box_set_spacing(GTK_BOX(vbox), 4);
 
    frame = gtk_frame_new("Chord Display Settings");
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 2);
    
    hbox = gtk_hbox_new(TRUE, 0);
    gtk_widget_show(hbox);
    gtk_container_add(GTK_CONTAINER(frame), hbox);
    
    dialog->chords_above_staff = gtk_check_button_new_with_label("Show chords above the staff");
    gtk_widget_show(dialog->chords_above_staff);
    
    gtk_box_pack_start(GTK_BOX(hbox), dialog->chords_above_staff, TRUE, TRUE, 8);
    
    frame = gtk_frame_new("Chord Library");
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 2);
    
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(hbox);
    gtk_container_add(GTK_CONTAINER(frame), hbox);
    
    dialog->current_chord_lib = gnome_file_entry_new(NULL, "Choose Chord Library");
    gtk_widget_show(dialog->current_chord_lib);
    
    gtk_box_pack_start(GTK_BOX(hbox), dialog->current_chord_lib, TRUE, TRUE, 8);
    
    button = gtk_button_new_with_label("New");
    gtk_widget_show(button);
    
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 8);
    
    g_signal_connect(G_OBJECT(button), "clicked",
                  G_CALLBACK(on_new_library_clicked), dialog);
                  
    frame = gtk_frame_new("Fonts");
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 2);
    
    table = gtk_table_new(6, 3, FALSE);
    gtk_widget_show(table);
    gtk_table_set_row_spacings(GTK_TABLE(table), 4);
    gtk_table_set_col_spacings(GTK_TABLE(table), 4);
    gtk_container_add(GTK_CONTAINER(frame), table);
    
    label = gtk_label_new("Tab Font:");
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1.00, 0.50);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
    
    dialog->tab_font = gnome_font_picker_new();
    gtk_widget_show(dialog->tab_font);
    gnome_font_picker_set_title(GNOME_FONT_PICKER(dialog->tab_font), "Tab Font");
    gnome_font_picker_set_mode(GNOME_FONT_PICKER(dialog->tab_font), GNOME_FONT_PICKER_MODE_FONT_INFO);
    gnome_font_picker_fi_set_use_font_in_label(GNOME_FONT_PICKER(dialog->tab_font), TRUE, 13);
    gnome_font_picker_set_preview_text(GNOME_FONT_PICKER(dialog->tab_font), "Regular Tab Font");
    
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->tab_font, 1, 2, 0, 1);
    
    button = gtk_button_new_with_label("Default");
    g_object_set_data(G_OBJECT(button), "font", "0");
    gtk_widget_show(button);
    gtk_table_attach_defaults(GTK_TABLE(table), button, 2, 3, 0, 1);
    g_signal_connect(G_OBJECT(button), "clicked",
                    G_CALLBACK(on_default_font_clicked), dialog);
                    
    label = gtk_label_new("Large Tab Font:");
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1.00, 0.50);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
    
    dialog->tab_font_large = gnome_font_picker_new();
    gtk_widget_show(dialog->tab_font_large);
    gnome_font_picker_set_title(GNOME_FONT_PICKER(dialog->tab_font_large), "Large Tab Font");
    gnome_font_picker_set_mode(GNOME_FONT_PICKER(dialog->tab_font_large), 
                                GNOME_FONT_PICKER_MODE_FONT_INFO);
    gnome_font_picker_fi_set_use_font_in_label(GNOME_FONT_PICKER(dialog->tab_font_large), TRUE, 13);
    gnome_font_picker_set_preview_text(GNOME_FONT_PICKER(dialog->tab_font_large), "Large Tab Font");
    
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->tab_font_large, 1, 2, 1, 2);
    
    button = gtk_button_new_with_label("Default");
    g_object_set_data(G_OBJECT(button), "font", "1");
    gtk_widget_show(button);
    gtk_table_attach_defaults(GTK_TABLE(table), button, 2, 3, 1, 2);
    
    g_signal_connect(G_OBJECT(button), "clicked",
                    G_CALLBACK(on_default_font_clicked), dialog);
                    
    label = gtk_label_new("Small Tab Font:");
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1.00, 0.50);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
    
    dialog->tab_font_small = gnome_font_picker_new();
    gtk_widget_show(dialog->tab_font_small);
    gnome_font_picker_set_title(GNOME_FONT_PICKER(dialog->tab_font_small), "Small Tab Font");
    gnome_font_picker_set_mode(GNOME_FONT_PICKER(dialog->tab_font_small), 
                                GNOME_FONT_PICKER_MODE_FONT_INFO);
    gnome_font_picker_fi_set_use_font_in_label(GNOME_FONT_PICKER(dialog->tab_font_small), TRUE, 13);
    gnome_font_picker_set_preview_text(GNOME_FONT_PICKER(dialog->tab_font_small), "Small Tab Font");
    
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->tab_font_small, 1, 2, 2, 3);
        
    button = gtk_button_new_with_label("Default");
    g_object_set_data(G_OBJECT(button), "font", "2");
    gtk_widget_show(button);
    gtk_table_attach_defaults(GTK_TABLE(table), button, 2, 3, 2, 3);
    
    g_signal_connect(G_OBJECT(button), "clicked",
                    G_CALLBACK(on_default_font_clicked), dialog);
    
    label = gtk_label_new("Tiny Tab Font:");
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1.00, 0.50);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);
    
    dialog->tab_font_tiny = gnome_font_picker_new();
    gtk_widget_show(dialog->tab_font_tiny);
    gnome_font_picker_set_title(GNOME_FONT_PICKER(dialog->tab_font_tiny), "Tiny Tab Font");
    gnome_font_picker_set_mode(GNOME_FONT_PICKER(dialog->tab_font_tiny), 
                                GNOME_FONT_PICKER_MODE_FONT_INFO);
    gnome_font_picker_fi_set_use_font_in_label(GNOME_FONT_PICKER(dialog->tab_font_tiny), TRUE, 13);
    gnome_font_picker_set_preview_text(GNOME_FONT_PICKER(dialog->tab_font_tiny), "Tiny Tab Font");
    
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->tab_font_tiny, 1, 2, 3, 4);
    
    button = gtk_button_new_with_label("Default");
    g_object_set_data(G_OBJECT(button), "font", "3");
    gtk_widget_show(button);
    gtk_table_attach_defaults(GTK_TABLE(table), button, 2, 3, 3, 4);
    g_signal_connect(G_OBJECT(button), "clicked",
                    G_CALLBACK(on_default_font_clicked), dialog);
                    
    label = gtk_label_new("Chord Builder Font:");
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1.00, 0.50);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);
    
    dialog->chrd_bldr_font = gnome_font_picker_new();
    gtk_widget_show(dialog->chrd_bldr_font);
    gnome_font_picker_set_title(GNOME_FONT_PICKER(dialog->chrd_bldr_font), "Chord Builder Font");
    gnome_font_picker_set_mode(GNOME_FONT_PICKER(dialog->chrd_bldr_font), 
                                GNOME_FONT_PICKER_MODE_FONT_INFO);
    gnome_font_picker_fi_set_use_font_in_label(GNOME_FONT_PICKER(dialog->chrd_bldr_font), TRUE, 13);
    gnome_font_picker_set_preview_text(GNOME_FONT_PICKER(dialog->chrd_bldr_font), "Chord Builder Font");
    
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->chrd_bldr_font, 1, 2, 4, 5);
    
    button = gtk_button_new_with_label("Default");
    g_object_set_data(G_OBJECT(button), "font", "4");
    gtk_widget_show(button);
    gtk_table_attach_defaults(GTK_TABLE(table), button, 2, 3, 4, 5);
    g_signal_connect(G_OBJECT(button), "clicked",
                    G_CALLBACK(on_default_font_clicked), dialog);
                    
    label = gtk_label_new("Chord Builder Small Font:");
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1.00, 0.50);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 5, 6);
    
    dialog->chrd_bldr_font_small = gnome_font_picker_new();
    gtk_widget_show(dialog->chrd_bldr_font_small);
    gnome_font_picker_set_title(GNOME_FONT_PICKER(dialog->chrd_bldr_font_small), "Chord Builder Small Font");
    gnome_font_picker_set_mode(GNOME_FONT_PICKER(dialog->chrd_bldr_font_small), 
                                GNOME_FONT_PICKER_MODE_FONT_INFO);
    gnome_font_picker_fi_set_use_font_in_label(GNOME_FONT_PICKER(dialog->chrd_bldr_font_small), TRUE, 13);
    gnome_font_picker_set_preview_text(GNOME_FONT_PICKER(dialog->chrd_bldr_font_small), "Chord Builder Small Font");
    
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->chrd_bldr_font_small, 1, 2, 5, 6);
    
    button = gtk_button_new_with_label("Default");
    g_object_set_data(G_OBJECT(button), "font", "5");
    gtk_widget_show(button);
    
    gtk_table_attach_defaults(GTK_TABLE(table), button, 2, 3, 5, 6);
    
    g_signal_connect(G_OBJECT(button), "clicked",
                    G_CALLBACK(on_default_font_clicked), dialog);
                    
    g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                  G_CALLBACK(dialog_destroyed), &dialog);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "response",
                  G_CALLBACK(prefs_response), dialog);
    
    g_signal_connect(G_OBJECT(dialog->chords_above_staff), "toggled",
                    G_CALLBACK(on_cas_toggled), dialog);
    
    g_signal_connect(G_OBJECT(dialog->current_chord_lib), "activate",
                    G_CALLBACK(on_lib_changed), dialog);
                    
    g_signal_connect(G_OBJECT(dialog->tab_font), "clicked",
                    G_CALLBACK(on_font_changed), dialog);
    
    g_signal_connect(G_OBJECT(dialog->tab_font_large), "clicked",
                    G_CALLBACK(on_font_changed), dialog);
    
    g_signal_connect(G_OBJECT(dialog->tab_font_small), "clicked",
                    G_CALLBACK(on_font_changed), dialog);
                        
    g_signal_connect(G_OBJECT(dialog->tab_font_tiny), "clicked",
                    G_CALLBACK(on_font_changed), dialog);

    g_signal_connect(G_OBJECT(dialog->chrd_bldr_font), "clicked",
                    G_CALLBACK(on_font_changed), dialog);
    
    g_signal_connect(G_OBJECT(dialog->chrd_bldr_font_small), "clicked",
                    G_CALLBACK(on_font_changed), dialog);
                    
    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
    
    dialog->app = app;
    dialog->lib_is_changed = FALSE;
    dialog->font_is_changed = FALSE;
    dialog->cas_is_changed = FALSE;

    return dialog;
    
}

/* file selection dialogs */

static void
file_load_response                (GtkButton       *button,
                                        GtkWidget     *dialog)
{
    gint response;
    gboolean file_exists;
    GtabDoc *tab_doc;
    
    response = atoi(g_object_get_data(G_OBJECT(button), "response"));
    
    if (response) {
        tab_doc = g_object_get_data(G_OBJECT(button), "tab_doc");
        
        tab_doc->filename = g_strdup(gtk_file_selection_get_filename
                                            (GTK_FILE_SELECTION
                                            (dialog)));
        
        gtk_widget_destroy(dialog);
        
        file_exists = g_file_test(tab_doc->filename,
                                G_FILE_TEST_EXISTS && G_FILE_TEST_IS_REGULAR);
        if (file_exists) {
            file_pre_load(tab_doc);
        }
        else {
            tab_doc->filename = NULL;
            gtab_warn(tab_doc->parent, "This file is not valid!");
        }
    }
    else {
        gtk_widget_destroy(dialog);
    }
}

GtkWidget*
create_file_load_select (GtabDoc *doc)
{
    GtkWidget *file_load_select;
    GtkWidget *file_load_ok;
    GtkWidget *file_load_cncl;

    file_load_select = gtk_file_selection_new (_("Select File"));
 
    gtk_container_set_border_width (GTK_CONTAINER (file_load_select), 10);
  
    gtk_window_set_position(GTK_WINDOW(file_load_select), GTK_WIN_POS_CENTER_ON_PARENT);
    gtk_window_set_modal(GTK_WINDOW (file_load_select), TRUE);
  
    file_load_ok = GTK_FILE_SELECTION (file_load_select)->ok_button;
    g_object_set_data (G_OBJECT (file_load_ok), "response", "1");
    g_object_set_data (G_OBJECT (file_load_ok), "tab_doc", doc);
    gtk_widget_show (file_load_ok);
    GTK_WIDGET_SET_FLAGS (file_load_ok, GTK_CAN_DEFAULT);

    file_load_cncl = GTK_FILE_SELECTION (file_load_select)->cancel_button;
    g_object_set_data (G_OBJECT (file_load_cncl), "response", "0");
    gtk_widget_show (file_load_cncl);
    GTK_WIDGET_SET_FLAGS (file_load_cncl, GTK_CAN_DEFAULT);

    g_signal_connect (GTK_OBJECT (file_load_ok), "clicked",
                      GTK_SIGNAL_FUNC (file_load_response),
                      file_load_select);
    g_signal_connect (GTK_OBJECT (file_load_cncl), "clicked",
                      GTK_SIGNAL_FUNC (file_load_response),
                      file_load_select);

    return file_load_select;
}

/* utility */

void
gtab_warn(GtabApp *app, const gchar *warning)
{
    GtkWidget *warning_dlg;
    gint dlg_return;
    
    warning_dlg = gtk_message_dialog_new(GTK_WINDOW(app->gnometab),
                                    GTK_DIALOG_MODAL,
                                    GTK_MESSAGE_WARNING,
                                    GTK_BUTTONS_OK,
                                    warning,
                                    NULL);
    
    dlg_return = gtk_dialog_run(GTK_DIALOG(warning_dlg));
    
    gtk_widget_destroy(warning_dlg);
}



