/*  Gnometab -- a guitar tablature editor for GNOME
    Copyright (C) 2001  William L. Guelker

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <math.h>
#include <gnome.h>
#include <libgnomecanvas/gnome-canvas.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include "tab_canvas.h"
#include "chord_builder.h"
#include "chord_lib.h"
#include "interface.h"
#include "tab_objects.h"

static gchar *text_icon[]={
    "21 21 19 1",
    ". c None",
    "c c #303030",
    "e c #323232",
    "i c #434343",
    "f c #4b4b4b",
    "b c #4c4c4c",
    "g c #4f4f4f",
    "d c #505050",
    "h c #555555",
    "a c #585858",
    "j c #636363",
    "k c #646464",
    "m c #656565",
    "l c #666666",
    "o c #676767",
    "p c #696969",
    "n c #6a6a6a",
    "# c #707070",
    "q c #d1d1d1",
    ".....................",
    "........#####a.......",
    ".......bcccccca......",
    "......deaaaaafca.....",
    ".....gca......hca....",
    ".....ia.......aca....",
    "..............jca....",
    ".............kca.....",
    "............jcl......",
    "...........mcn.......",
    "..........jco........",
    ".........kcn.........",
    "........jco..........",
    ".......acn...........",
    "......aca............",
    ".....acp.............",
    "....accq.............",
    "....accaaaaaaaaaa....",
    "....accccccccccca....",
    "....aaaaaaaaaaaaa....",
    "....................."
};

static gchar *bar_icon[]={
    "21 21 3 1",
    ". c None",
    "a c #303030",
    "# c #c00000",
    ".....................",
    ".....................",
    "......########.......",
    "....###......###.....",
    "..###..........###...",
    ".##..............##..",
    ".....aa.......aa.....",
    "....aaa......aaa.....",
    "....aaa......aaa.....",
    ".....aa.......aa.....",
    ".....aa.......aa.....",
    ".....aa.......aa.....",
    ".....aa.......aa.....",
    ".....aa.......aa.....",
    ".....aa.......aa.....",
    ".....aa.......aa.....",
    ".....aa.......aa.....",
    "....aaaa.....aaaa....",
    "....aaaa.....aaaa....",
    ".....................",
    "....................."
};

/* callbacks */

static void
builder_destroyed (GtkObject *obj,  void **dialog_pointer)
{
    if (dialog_pointer != NULL)
    {
        g_free (*dialog_pointer);
        *dialog_pointer = NULL;
    }
}

static void
on_chrd_btn_num_clicked                (GtkButton           *button,
                                        GtabChordBuilder     *gtab_dialog)
{
    
    gtab_dialog->current_tool = TOOL_NUM;
}

static void
on_chrd_btn_bar_clicked                (GtkButton           *button,
                                        GtabChordBuilder     *gtab_dialog)
{
    
    gtab_dialog->current_tool = TOOL_BAR;
}

static void
on_chrd_btn_save_clicked               (GtkButton           *button,
                                        GtabChordBuilder     *gtab_dialog)
{
    gboolean value;
    xmlNodePtr copied_node;
    xmlNodePtr new_chord;
    xmlNodePtr chrd_name_node;
    
    value = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
                                    gtab_dialog->tab_doc->parent->btn_lib_chord));
    copied_node = xmlCopyNode(gtab_dialog->chord->xmlRootNode, TRUE);
    
    if ( ((GtabChordLib *) gtab_dialog->tab_doc->parent->chord_lib)->lib_doc) {
        new_chord = xmlAddChild( ((GtabChordLib *) gtab_dialog->tab_doc->parent->chord_lib)->lib_doc->xmlRootNode, 
                                copied_node);
        chrd_name_node = xmlNewChild(new_chord, NULL, "chord_name", NULL);
        xmlSetProp(chrd_name_node, "name_text", 
            gtk_entry_get_text(GTK_ENTRY(gtab_dialog->name)));
        
        if (value == TRUE) {
            refresh_chord_canvas( ((GtabChordLib *)gtab_dialog->tab_doc->parent->chord_lib));
        }
        else {
            sort_lib(((GtabChordLib *)gtab_dialog->tab_doc->parent->chord_lib));
        }
    }

}

static void
on_chrd_btn_create_clicked             (GtkButton           *button,
                                        GtabChordBuilder     *gtab_dialog)
{
    xmlNodePtr chrd_name_node = NULL;
    
    chrd_name_node = xmlNewChild(gtab_dialog->chord->xmlRootNode, NULL, "chord_name", NULL);
    xmlSetProp(chrd_name_node, "name_text", 
        gtk_entry_get_text(GTK_ENTRY(gtab_dialog->name)));
    
    gtab_dialog->y = (floor(gtab_dialog->y/140))*140 - 4;
    translate_chord(gtab_dialog->tab_doc, gtab_dialog->x, gtab_dialog->y, gtab_dialog->chord);
    gtk_widget_destroy(gtab_dialog->dialog);
}

static void
on_chrd_btn_cncl_clicked               (GtkButton           *button,
                                        GtabChordBuilder       *gtab_dialog)    
{
    xmlFreeDoc(gtab_dialog->chord);
    gtab_dialog->chord = NULL;
    gtk_widget_destroy(gtab_dialog->dialog);

}

static void
popup_destroyed (GtkObject *obj,  void **popup_pointer)
{
    if (popup_pointer != NULL)
    {
        g_free (*popup_pointer);
        *popup_pointer = NULL;
    }
}

static void
on_popup_deactivate        (GtkMenuShell    *menushell,
                                        GtabChordBuilder     *builder)
{
    if (GNOME_IS_CANVAS_ITEM(builder->highlights)) {
        gtk_object_destroy(GTK_OBJECT(builder->highlights));
        builder->highlights = NULL;
    }
}                    

static void
on_clear_popup_activate                (GtkMenuItem     *menuitem,
                                        GtabChordBuilder    *builder)
{
    GnomeCanvasItem *item;
    xmlNodePtr node;
    
    item = g_object_get_data(G_OBJECT(builder->popup->menu), "chord_obj");
    
    if (! GNOME_IS_CANVAS_ITEM(item)) return;
        
    node = (xmlNodePtr ) g_object_get_data(G_OBJECT(item), "node");
    
    if (node)    {
        xmlSetProp(node, "active", "false");   
    }
                        
    gtk_object_destroy(GTK_OBJECT(item));
    
    
    
}

static GnomeUIInfo right_click_popup_uiinfo[] =
{
    {
        GNOME_APP_UI_ITEM, N_("_Delete"),
        NULL,
        (gpointer) on_clear_popup_activate, NULL, NULL,
        GNOME_APP_PIXMAP_NONE, NULL,
        0, (GdkModifierType) 0, NULL
    },
    GNOMEUIINFO_END
};

static GtabDocPopup*
create_right_click_popup (GtabChordBuilder *builder)
{
    static GtabDocPopup *popup = NULL;

    popup = g_new0(GtabDocPopup, 1);
    
    popup->menu = gtk_menu_new ();
    
    gtk_container_set_border_width (GTK_CONTAINER (popup->menu), 2);
    gnome_app_fill_menu_with_data (GTK_MENU_SHELL (popup->menu), 
                                    right_click_popup_uiinfo,
                                    NULL, FALSE, 0, builder);

    g_signal_connect(G_OBJECT(popup->menu), "deactivate",
                G_CALLBACK(on_popup_deactivate), builder);
    g_signal_connect(G_OBJECT(popup->menu), "destroy",
                          G_CALLBACK(popup_destroyed), &popup);
    
    popup->x = 0;
    popup->y = 0;
    
    return popup;
}

GtabChordBuilder*
create_chord_popup (GtabDoc *doc)
{
    static GtabChordBuilder *dialog = NULL;
    GtkWidget *vbox;
    GtkWidget *hseparator;
    GtkWidget *hbox;
    GtkWidget *label;
    GtkWidget *dialog_action_area;
    GtkWidget *toolbar;
    GtkWidget *chrd_btn_num;
    GtkWidget *chrd_btn_bar;
    GtkWidget *chrd_btn_save;
    GtkWidget *chrd_btn_create;
    GtkWidget *chrd_btn_cancel;
    GtkWidget *tmp_toolbar_icon;
    
    dialog = g_new0(GtabChordBuilder, 1);
    
    dialog->dialog = gtk_dialog_new ();
        
    gtk_window_set_title (GTK_WINDOW (dialog->dialog), _("Create Chord"));
    gtk_window_set_position (GTK_WINDOW (dialog->dialog), GTK_WIN_POS_MOUSE);
    gtk_window_set_modal (GTK_WINDOW (dialog->dialog), TRUE);

    vbox = GTK_DIALOG (dialog->dialog)->vbox;
    gtk_widget_show (vbox);
      
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 4);
    
    gtk_widget_push_colormap (gdk_rgb_get_colormap ());
    dialog->canvas = gnome_canvas_new ();
    gtk_widget_pop_colormap ();
    gtk_widget_show (dialog->canvas);
    gtk_box_pack_start (GTK_BOX (vbox), dialog->canvas, TRUE, TRUE, 0);
    gtk_widget_set_size_request (dialog->canvas, 98, 231);
    gtk_container_set_border_width (GTK_CONTAINER (dialog->canvas), 2);
    gnome_canvas_set_scroll_region (GNOME_CANVAS (dialog->canvas), 0, 0, 98, 72);
    gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (dialog->canvas), 2);
    
    hseparator = gtk_hseparator_new ();
    gtk_widget_show (hseparator);
    gtk_box_pack_start (GTK_BOX (vbox), hseparator, TRUE, TRUE, 0);
    
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 5);
    
    label = gtk_label_new (_("Chord Name: "));
    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    
    dialog->name = gtk_entry_new ();
    gtk_entry_set_max_length(GTK_ENTRY(dialog->name), 10);
      
    gtk_widget_show (dialog->name);
    gtk_box_pack_start (GTK_BOX (hbox), dialog->name, TRUE, TRUE, 0);
    gtk_entry_set_text (GTK_ENTRY (dialog->name), _("C"));
    
    dialog_action_area = GTK_DIALOG (dialog->dialog)->action_area;
    gtk_widget_show (dialog_action_area);
    gtk_container_set_border_width (GTK_CONTAINER (dialog_action_area), 10);
    
    toolbar = gtk_toolbar_new ();
    gtk_toolbar_set_style(GTK_TOOLBAR(toolbar), GTK_TOOLBAR_BOTH);
    
     
    tmp_toolbar_icon = create_pixmap(dialog->dialog, text_icon);
    chrd_btn_num = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
                                    GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                    NULL,
                                    N_("Text"),
                                    N_("Enter Text"), NULL,
                                    tmp_toolbar_icon, NULL, NULL);
          
    gtk_widget_show (chrd_btn_num);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (chrd_btn_num), TRUE);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (chrd_btn_num), FALSE);
    
    tmp_toolbar_icon = create_pixmap(dialog->dialog, bar_icon);
    chrd_btn_bar = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
                                    GTK_TOOLBAR_CHILD_RADIOBUTTON,
                                    chrd_btn_num,
                                    N_("Bar"),
                                    N_("Create Bar"), NULL,
                                    tmp_toolbar_icon, NULL, NULL);
      
    gtk_widget_show (chrd_btn_bar);
    gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (chrd_btn_bar), FALSE);
    
    gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));
    
    tmp_toolbar_icon = gtk_image_new_from_stock(GTK_STOCK_SAVE, GTK_ICON_SIZE_SMALL_TOOLBAR);
    chrd_btn_save = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
                                    GTK_TOOLBAR_CHILD_BUTTON,
                                    NULL,
                                    N_("Save"),
                                    N_("Save this chord in the chord library"), NULL,
                                    tmp_toolbar_icon, NULL, NULL);
        
    gtk_widget_show (chrd_btn_save);
    
    tmp_toolbar_icon = gtk_image_new_from_stock(GTK_STOCK_EXECUTE, GTK_ICON_SIZE_SMALL_TOOLBAR);
    chrd_btn_create = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
                                    GTK_TOOLBAR_CHILD_BUTTON,
                                    NULL,
                                    N_("Create"),
                                    N_("Create this chord on the tablature"), NULL,
                                    tmp_toolbar_icon, NULL, NULL);
      
    gtk_widget_show (chrd_btn_create);
    
    tmp_toolbar_icon = gtk_image_new_from_stock(GTK_STOCK_CANCEL, GTK_ICON_SIZE_SMALL_TOOLBAR);
    chrd_btn_cancel = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
                                    GTK_TOOLBAR_CHILD_BUTTON,
                                    NULL,
                                    N_("Cancel"),
                                    N_("Cancel"), NULL,
                                    tmp_toolbar_icon, NULL, NULL);
    
    gtk_widget_show (chrd_btn_cancel);
    
    gtk_widget_show (toolbar);
    
    gtk_box_pack_start (GTK_BOX (dialog_action_area), toolbar, FALSE, FALSE, 0);
    
    g_signal_connect(G_OBJECT(dialog->dialog), "destroy",
                        G_CALLBACK(builder_destroyed), 
                        &dialog);
                      
    g_signal_connect (G_OBJECT (chrd_btn_num), "clicked",
                        G_CALLBACK (on_chrd_btn_num_clicked),
                        dialog);
    g_signal_connect (G_OBJECT (chrd_btn_bar), "clicked",
                        G_CALLBACK (on_chrd_btn_bar_clicked),
                        dialog);
    g_signal_connect (G_OBJECT (chrd_btn_save), "clicked",
                        G_CALLBACK (on_chrd_btn_save_clicked),
                        dialog);
    g_signal_connect (G_OBJECT (chrd_btn_create), "clicked",
                        G_CALLBACK (on_chrd_btn_create_clicked),
                        dialog);
    g_signal_connect (G_OBJECT (chrd_btn_cancel), "clicked",
                        G_CALLBACK (on_chrd_btn_cncl_clicked),
                        dialog);
    
    gtk_window_set_resizable(GTK_WINDOW(dialog->dialog), FALSE);
    
    dialog->popup = create_right_click_popup(dialog);
    dialog->tab_doc = doc;
    dialog->highlights = NULL;
    
    return dialog;
}
