/******************************************************************************\
 gnofin/ui-record-list.c   $Revision: 1.16.2.1 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkclist.h>
#include <libgnome/gnome-config.h>
#include "ui-record-list.h"
#include "record-stringizer.h"
#include "data-if.h"
#include "dialogs.h"
#include "config-saver.h"
#include "date.h"
#include "row-color.h"
#include "notification-list.h"
#include "data-structure/record-select.h"


static GtkWidget *create_clist (UI_RecordList *list);
static GtkEventBoxClass *parent_class;

enum {
  SELECTION_CHANGED,
  LAST_SIGNAL
};
static gint signals [LAST_SIGNAL] = {0};


#define FIELD_IS_BAL(f)   ( ((f) == RECORD_FIELD_OVERALL_BAL) \
			 || ((f) == RECORD_FIELD_CLEARED_BAL) )


/******************************************************************************
 * Configuration
 */

NotificationList ci_change_listeners = {0};

typedef struct {
  UI_RecordListColumnInfo *cinfo;
  guint                    ncols;
} ListConfig;

static const UI_RecordListColumnInfo default_column_info[] = {
  { RECORD_FIELD_DATE,            N_("Date"),          76, GTK_JUSTIFY_LEFT  },
  { RECORD_FIELD_TYPE,            N_("Type"),          32, GTK_JUSTIFY_LEFT  },
  { RECORD_FIELD_NUMBER,          N_("No"),            21, GTK_JUSTIFY_LEFT  },
  { RECORD_FIELD_LINKED_ACC_NAME, N_("Transfer from"), 74, GTK_JUSTIFY_LEFT  },
  { RECORD_FIELD_PAYEE,           N_("Payee/payor"),  212, GTK_JUSTIFY_LEFT  },
//{ RECORD_FIELD_MEMO,            N_("Memo"),         150, GTK_JUSTIFY_LEFT  },
  { RECORD_FIELD_CATEGORY,        N_("Category"),     150, GTK_JUSTIFY_LEFT  },
  { RECORD_FIELD_CLEARED,         N_("S"),              5, GTK_JUSTIFY_LEFT  },
  { RECORD_FIELD_AMOUNT,          N_("Amount"),        49, GTK_JUSTIFY_RIGHT },
  { RECORD_FIELD_OVERALL_BAL,     N_("Balance"),       49, GTK_JUSTIFY_RIGHT }
};
#define DEFCOLS sizeof_array (default_column_info)

#define CAT  "UI_RecordList"
#define KEY  "/" PACKAGE "/" CAT "/"

static void
load_list_config (ListConfig *config)
{
  gchar path[512];
  guint i;

  trace ("");
  g_return_if_fail (config);

  g_snprintf (path, sizeof path, KEY "num_cols=%d",
  	      sizeof_array (default_column_info));
  config->ncols = gnome_config_get_int (path);

  config->cinfo = g_new0 (UI_RecordListColumnInfo, config->ncols);
  for (i=0; i<config->ncols; ++i)
  {
    g_snprintf (path, sizeof path, KEY "column[%d].field=%d", i,
	        i < DEFCOLS ? default_column_info[i].field : 0);
    config->cinfo[i].field = gnome_config_get_int (path);

    if (config->cinfo[i].field == 0)
    {
      config->ncols = i;
      break;
    }

    g_snprintf (path, sizeof path, KEY "column[%d].label=%s", i,
	        i < DEFCOLS ? _(default_column_info[i].label) : "");
    config->cinfo[i].label = gnome_config_get_string (path);

    g_snprintf (path, sizeof path, KEY "column[%d].width=%d", i,
	        i < DEFCOLS ? default_column_info[i].width : 100);
    config->cinfo[i].width = gnome_config_get_int (path);

    g_snprintf (path, sizeof path, KEY "column[%d].align=%d", i,
	        i < DEFCOLS ? default_column_info[i].align : GTK_JUSTIFY_LEFT);
    config->cinfo[i].align = gnome_config_get_int (path);
  }
}

static void
save_list_config (const ListConfig *config)
{
  gchar path[512];
  guint i;

  trace ("");
  g_return_if_fail (config);

  gnome_config_set_int (KEY "num_cols", config->ncols);
  for (i=0; i<config->ncols; ++i)
  {
    g_snprintf (path, sizeof path, KEY "column[%d].field", i);
    gnome_config_set_int (path, config->cinfo[i].field);

    g_snprintf (path, sizeof path, KEY "column[%d].label", i);
    gnome_config_set_string (path, config->cinfo[i].label);

    g_snprintf (path, sizeof path, KEY "column[%d].width", i);
    gnome_config_set_int (path, config->cinfo[i].width);

    g_snprintf (path, sizeof path, KEY "column[%d].align", i);
    gnome_config_set_int (path, config->cinfo[i].align);
  }
}

static ListConfig *
get_list_config (void)
{
  static ListConfig config = {0};
  static gboolean init = FALSE;

  if (!init)
  {
    load_list_config (&config);
    config_saver_register (CAT, (ConfigSaveFunc) save_list_config, &config);
    init = TRUE;
  }
  return &config;
}


/******************************************************************************
 * Helper functions
 */

static inline GtkCList *
get_clist (UI_RecordList *list)
{
  g_return_val_if_fail (list, NULL);
  g_return_val_if_fail (GTK_BIN (list)->child, NULL);
  g_return_val_if_fail (GTK_BIN (GTK_BIN (list)->child)->child, NULL);

  return GTK_CLIST (GTK_BIN (GTK_BIN (list)->child)->child);
}

static inline GtkContainer *
get_scrolled_win (UI_RecordList *list)
{
  g_return_val_if_fail (list, NULL);

  return GTK_CONTAINER (GTK_BIN (list)->child);
}

static inline void
block_select_row (UI_RecordList *list)
{
  trace ("");
  gtk_signal_handler_block (GTK_OBJECT (get_clist (list)),
  			    list->select_row_id);
  gtk_signal_handler_block (GTK_OBJECT (get_clist (list)),
  			    list->unselect_row_id);
}

static inline void
unblock_select_row (UI_RecordList *list)
{
  trace ("");
  gtk_signal_handler_unblock (GTK_OBJECT (get_clist (list)),
			      list->select_row_id);
  gtk_signal_handler_unblock (GTK_OBJECT (get_clist (list)),
			      list->unselect_row_id);
}

static inline void
set_row_colors (GtkCList *clist, guint r)
{
  trace ("");

  gtk_clist_set_foreground (clist, r, (r&1 ? row_color_dark_fg() : row_color_bright_fg()));
  gtk_clist_set_background (clist, r, (r&1 ? row_color_dark_bg() : row_color_bright_bg()));
}

static void
set_cell_style (UI_RecordList *list, guint r, guint c, guint field, const RecordInfo *rec)
{
  trace ("");

  if (((field == RECORD_FIELD_CLEARED_BAL) && (rec->cleared_bal < 0)) ||
      ((field == RECORD_FIELD_OVERALL_BAL) && (rec->overall_bal < 0)))
  {
    GtkCList *clist = get_clist (list);
    GtkStyle *style;
    GdkColor color;

    color.red = 0xffff;
    color.green = 0;
    color.blue = 0;

    style = gtk_clist_get_cell_style (clist, r, c);
    if (style == NULL)
      style = GTK_WIDGET (clist)->style;

    style = gtk_style_copy (style);
    style->fg [GTK_STATE_NORMAL] = color;
    style->fg [GTK_STATE_SELECTED] = color;
    style->base [GTK_STATE_NORMAL] = *(r&1 ? row_color_dark_bg() : row_color_bright_bg());

    gtk_clist_set_cell_style (clist, r, c, style);
    gtk_style_unref (style);
  }
}

static void
refresh_bal (UI_RecordList *list, guint start_from)
{
  const GList *it;
  GtkCList *clist;
  ListConfig *config = get_list_config ();
  guint k;

  trace ("");
  g_return_if_fail (list);
  g_return_if_fail (list->account);

  clist = get_clist (list);
  gtk_clist_freeze (clist);

  it = if_account_get_records (list->account); 
  it = g_list_nth ((GList *) it, start_from);

  for (k=start_from; it; it=it->next, ++k)
  {
    RecordInfo rec;
    Record *record;
    guint i, field;
    gchar *buf;

    record = LIST_DEREF (Record, it);
    trace ("record=%p (%d)", record, k);

    if_record_get_info (record, 0, &rec);

    set_row_colors (clist, k);
    if (!is_record_view(record))
      continue;

    for (i=0; i < config->ncols; ++i)
    {
      field = config->cinfo[i].field;

      if (FIELD_IS_BAL (field))
      {
	buf = stringize_record_field (NULL, 0, field, &rec);
	gtk_clist_set_text (clist, k, i, buf);
	set_cell_style (list, k, i, field, &rec);
	g_free (buf);
      }
    }
  }
  gtk_clist_thaw (clist);
}


/******************************************************************************
 * Signal handlers
 */

static void
on_select_row (GtkCList *clist, gint row, gint col, GdkEvent *event, UI_RecordList *list)
{
  trace ("row=%d", row);
  g_return_if_fail (clist);
  g_return_if_fail (list);

  list->selected_record = if_account_get_record_by_index (list->account, row);

  gtk_signal_emit (GTK_OBJECT (list), signals [SELECTION_CHANGED]);
}

static void
on_unselect_row (GtkCList *clist, gint row, gint col, GdkEvent *event, UI_RecordList *list)
{
  trace ("row=%d", row);
  g_return_if_fail (clist);
  g_return_if_fail (list);

  /* Normally, since we are using the BROWSE selection mode, we dont have to worry
   * about informing others that our selection has been removed, since it will be
   * set again almost immediately.  However, if this is the last row left, there
   * will be no other row selected, so we have to emit the SELECTION_CHANGED signal.
   */
  if (clist->rows == 1)
  {
    list->selected_record = NULL;
    gtk_signal_emit (GTK_OBJECT (list), signals [SELECTION_CHANGED]);
  }
}

static void
on_resize_column (GtkCList *clist, gint col, gint width, UI_RecordList *list)
{
  ListConfig *config = get_list_config ();

  trace ("");
  g_return_if_fail (list);
  g_return_if_fail (col < config->ncols);

  config->cinfo[col].width = width;
}

static void
on_click_column (GtkCList *clist, gint col, UI_RecordList *list)
{
  ListConfig *config = get_list_config ();
  guint field, rev;

  trace ("col = %d", col);
  g_return_if_fail (list);

  field = config->cinfo[col].field;

  if (list->account)
  {
    AccountInfo acc;
    GtkCList *clist;
    guint index;
    gboolean result;

    if_account_get_info (list->account,
    			 ACCOUNT_FIELD_SORT_FIELD |
    			 ACCOUNT_FIELD_SORT_REV,
			 &acc);

    if (field == acc.sort_field)
      rev = !acc.sort_rev;
    else
      rev = 0;

    /* re-sort records, keeping same record selected */

    clist = get_clist (list);
    gtk_clist_freeze (clist);

    block_select_row (list);
    result = if_account_sort_records ((Account *) list->account, field, rev);
    unblock_select_row (list);

    if (result)
    {
      if (list->selected_record)
      {
	index = if_record_get_index (list->selected_record);
	ui_record_list_select_record (list, index);
      }
    }
    else
    {
      Bankbook *book = if_account_get_parent (list->account);

      dialog_error (GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (list))),
      		    if_bankbook_get_error (book));
    }

    gtk_clist_thaw (clist);
  }
}

static void
on_row_color_changed (UI_RecordList *list)
{
  trace ("");
  ui_record_list_refresh (list, 0);
}

static void
on_date_format_changed (UI_RecordList *list)
{
  trace ("");
  ui_record_list_refresh (list, 0);
}

static void
on_money_format_changed (UI_RecordList *list)
{
  trace ("");
  ui_record_list_refresh (list, 0);
}

static void
on_column_info_changed (UI_RecordList *list)
{
  GtkWidget *old_clist, *new_clist;
  GtkContainer *scrolled_win;

  trace ("");
  g_return_if_fail (list);

  old_clist = GTK_WIDGET (get_clist (list));
  scrolled_win = get_scrolled_win (list);
  new_clist = create_clist (list); 
  gtk_widget_show_all (new_clist);

  gtk_container_remove (scrolled_win, old_clist);
  gtk_container_add (scrolled_win, new_clist);

  ui_record_list_refresh (list, 0);

//if (list->selected_record)
//  ui_record_list_select_record (list, if_record_get_index (list->selected_record));
}


/******************************************************************************
 * Initialization
 */

static GtkWidget *
create_clist (UI_RecordList *list)
{
  ListConfig *config = get_list_config ();
  GtkWidget *clist;
  guint i;

  trace ("");

  clist = gtk_clist_new (config->ncols);
  gtk_clist_column_titles_show (GTK_CLIST (clist));
  for (i=0; i < config->ncols; ++i)
  {
    gtk_clist_set_column_title (
    	GTK_CLIST (clist), i, _(config->cinfo[i].label));
    gtk_clist_set_column_width (
    	GTK_CLIST (clist), i, config->cinfo[i].width);
    gtk_clist_set_column_justification (
    	GTK_CLIST (clist), i, config->cinfo[i].align);
    
    if (FIELD_IS_BAL (config->cinfo[i].field))
      gtk_clist_column_title_passive (GTK_CLIST (clist), i);
  }
  gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_BROWSE);

  list->select_row_id =
    gtk_signal_connect (GTK_OBJECT (clist), "select_row",
			GTK_SIGNAL_FUNC (on_select_row), list);
  list->unselect_row_id = 
    gtk_signal_connect (GTK_OBJECT (clist), "unselect_row",
			GTK_SIGNAL_FUNC (on_unselect_row), list);
  gtk_signal_connect (GTK_OBJECT (clist), "resize_column",
  		      GTK_SIGNAL_FUNC (on_resize_column), list);
  gtk_signal_connect (GTK_OBJECT (clist), "click_column",
  		      GTK_SIGNAL_FUNC (on_click_column), list);
  return clist;
}

static void
ui_record_list_class_init (GtkObjectClass *object_class)
{
  trace ("");

  parent_class = gtk_type_class (gtk_event_box_get_type ());

  signals [SELECTION_CHANGED] = gtk_signal_new (
  	"selection_changed",
	GTK_RUN_FIRST,
	object_class->type,
	GTK_SIGNAL_OFFSET (UI_RecordListClass, selection_changed),
	gtk_marshal_NONE__NONE,
	GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

static void
ui_record_list_init (UI_RecordList *list)
{
  GtkWidget *scrolled_win, *clist;

  trace ("");
  g_return_if_fail (list);

  list->account = NULL;
  list->selected_record = NULL;

  notification_list_add (&date_format_change_listeners,
  			 NOTIFICATION_FUNC (on_date_format_changed), list);
  notification_list_add (&money_format_change_listeners,
  			 NOTIFICATION_FUNC (on_money_format_changed), list);
  notification_list_add (&ci_change_listeners,
  			 NOTIFICATION_FUNC (on_column_info_changed), list);
  notification_list_add (&row_color_change_listeners,
  			 NOTIFICATION_FUNC (on_row_color_changed), list);

  /* Scrolled window */
  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  /* List object */
  clist = create_clist (list);


  gtk_container_add (GTK_CONTAINER (scrolled_win), clist);
  gtk_container_add (GTK_CONTAINER (list), scrolled_win);
  gtk_widget_show_all (scrolled_win);
}


/******************************************************************************
 * Interface
 */

GtkType
ui_record_list_get_type (void)
{
  static GtkType type = 0;

  if (!type)
  {
    GtkTypeInfo info = {
      "UI_RecordList",
      sizeof (UI_RecordList),
      sizeof (UI_RecordListClass),
      (GtkClassInitFunc) ui_record_list_class_init,
      (GtkObjectInitFunc) ui_record_list_init,
      NULL,
      NULL,
      (GtkClassInitFunc) NULL
    };
    type = gtk_type_unique (gtk_event_box_get_type (), &info);
  }
  return type;
}

GtkWidget *
ui_record_list_new (void)
{
  return GTK_WIDGET (gtk_type_new (ui_record_list_get_type ()));
}

void
ui_record_list_set_account (UI_RecordList *list, Account *account)
{
  trace ("");
  g_return_if_fail (list);
  g_return_if_fail (account);

  list->account = account;
  //ui_record_list_refresh (list, 0);
}

void
ui_record_list_refresh (UI_RecordList *list, guint start_from)
{
  const GList *it;
  GtkCList *clist;
  ListConfig *config = get_list_config ();
  guint k;

  trace ("");
  g_return_if_fail (list);
  g_return_if_fail (list->account);

  clist = get_clist (list);
  g_assert (clist);

  gtk_clist_freeze (clist);

  while (clist->rows > start_from)
    gtk_clist_remove (clist, start_from);

  it = if_account_get_records (list->account); 
  it = g_list_nth ((GList *) it, start_from);
  
  for (k=start_from; it; it=it->next, k++)
  {
    RecordInfo rec;
    Record *record;
    gchar **buf;
    guint i;

    record = LIST_DEREF (Record, it);
    trace ("record=%p (%d)", record, k);

    if_record_get_info (record, 0, &rec);
    if (!is_record_view(record))
      continue;

    buf = g_new (gchar *, config->ncols);

    for (i=0; i<config->ncols; ++i)
      buf[i] = stringize_record_field (NULL, 0, config->cinfo[i].field, &rec);

    gtk_clist_insert (clist, k, buf);
    set_row_colors (clist, k);

    /* Apply negative style if balance < 0
     */
    for (i = 0; i < config->ncols; i++)
    {
      set_cell_style (list, k, i, config->cinfo[i].field, &rec);
      g_free (buf[i]);
    }
    g_free (buf);
  }

  if (list->selected_record)
  {
    block_select_row (list);
    ui_record_list_select_record (list, if_record_get_index (list->selected_record));
    unblock_select_row (list);
  }

  gtk_clist_thaw (clist);
}

void
ui_record_list_refresh_record (UI_RecordList *list, const Record *record, guint last_index, guint mask)
{
  ListConfig *config = get_list_config ();
  GtkCList *clist;
  guint index;

  trace ("");
  g_return_if_fail (list);
  g_return_if_fail (list->selected_record);

  /* Check if from equals current index of selected_record, if it
   * doesn't then move the from row to the new index. */

  index = if_record_get_index (record);

  clist = get_clist (list);
  gtk_clist_freeze (clist);

  block_select_row (list);

  if (index != last_index)
    gtk_clist_row_move (clist, last_index, index);

  /* Re-insert record */
  {
    RecordInfo rec;
    guint i;
    gchar *buf;

    if_record_get_info (record, 0, &rec);

    set_row_colors (clist, index);

    for (i=0; i < config->ncols; ++i)
    {
      buf = stringize_record_field (NULL, 0, config->cinfo[i].field, &rec);
      gtk_clist_set_text (clist, index, i, buf);
      g_free (buf);
    }

    /* Apply negative style if balance < 0
     */
    for (i=0; i < config->ncols; ++i)
      set_cell_style (list, index, i, config->cinfo[i].field, &rec);
  }

  if ((last_index != index) || (mask == 0) || (mask & RECORD_FIELD_AMOUNT))
    refresh_bal (list, MIN (index, last_index));

  unblock_select_row (list);

  if (record == list->selected_record)
    ui_record_list_select_record (list, index);

  gtk_clist_thaw (clist);
}

void
ui_record_list_insert_record (UI_RecordList *list, guint index, gboolean select)
{
  ListConfig *config = get_list_config ();
  GtkCList *clist;
  RecordInfo rec;
  Record *record;
  guint i;
  gchar **buf;

  trace ("");
  g_return_if_fail (list);

  clist = get_clist (list);
  gtk_clist_freeze (clist);

  record = if_account_get_record_by_index (list->account, index);
  if_record_get_info (record, 0, &rec);

  buf = g_new (gchar *, config->ncols);
  for (i = 0; i < config->ncols; i++)
    buf[i] = stringize_record_field (NULL, 0, config->cinfo[i].field, &rec);

  gtk_clist_insert (clist, index, buf);
  set_row_colors (clist, index);
  
  /* Apply negative style if balance < 0
   */
  for (i = 0; i < config->ncols; i++)
  {
    set_cell_style (list, index, i, config->cinfo[i].field, &rec);
    g_free (buf[i]);
  }
  g_free (buf);

  if (select)
    ui_record_list_select_record (list, index);

  refresh_bal (list, index + 1);
  gtk_clist_thaw (clist);
}

void
ui_record_list_remove_record (UI_RecordList *list, guint index)
{
  GtkCList *clist;

  trace ("");
  g_return_if_fail (list);

  clist = get_clist (list);
  gtk_clist_freeze (clist);

  gtk_clist_remove (clist, index);
  refresh_bal (list, index);

  gtk_clist_thaw (clist);
}

void
ui_record_list_select_record (UI_RecordList *list, guint index)
{
  GtkCList *clist;

  trace ("row=%d", index);
  g_return_if_fail (list);

  if (index == (guint) -1)
    index = g_list_length ((GList *) if_account_get_records (list->account)) - 1;

  clist = get_clist (list);
  gtk_clist_freeze (clist);
  gtk_clist_select_row (clist, index, 0);
  gtk_clist_moveto (clist, index, -1, 0.5, 0);
  gtk_clist_thaw (clist);
}


/******************************************************************************
 * Column info set/get
 */

guint
ui_record_list_get_column_info (UI_RecordListColumnInfo **cinfo)
{
  ListConfig *config = get_list_config ();
  trace ("");

  if (cinfo) *cinfo = config->cinfo;
  return config->ncols;
}

void
ui_record_list_set_column_info (UI_RecordListColumnInfo *cinfo, guint num)
{
  ListConfig *config = get_list_config ();
  guint i;

  trace ("");
  g_return_if_fail (cinfo);

  for (i=0; i < config->ncols; ++i)
    g_free (config->cinfo[i].label);
  
  if (config->ncols != num)
  {
    g_free (config->cinfo);

    config->ncols = num;
    config->cinfo = g_new0 (UI_RecordListColumnInfo, num);
  } 

  for (i=0; i < num; ++i)
  {
    memcpy (config->cinfo + i, cinfo + i, sizeof (*cinfo));
    config->cinfo[i].label = g_strdup (cinfo[i].label);
  }

  /* Activate changes */
  notification_list_notify (&ci_change_listeners);
}
