/******************************************************************************\
 gnofin/plugin.c   $Revision: 1.6 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

/*
 * Support for dynamically-loaded (plugin) components.
 *
 * Based on code from Gnumeric by:
 *    Tom Dyas (tdyas@romulus.rutgers.edu)
 */

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <unistd.h>
#include <dirent.h>
#include <glib.h>
#include <gmodule.h>
#include <string.h>
#include <libgnome/gnome-util.h>
#include "dialogs.h"
#include "plugin.h"

GSList *plugin_list = NULL;

PluginData *
plugin_load (GtkWindow *win, const gchar *filename)
{
  PluginData *pd;
  const gchar *reason;
  guint res;

  trace ("loading: %s", filename);

  g_assert (filename != NULL);
  
  pd = g_new0 (PluginData, 1);
  if (!pd)
  {
    reason = _("Insufficient Memory");
    goto error;
  }

  pd->handle = g_module_open (filename, 0);

  if (!pd->handle)
  {
    reason = g_module_error ();
    goto error;
  }
  
  if (!g_module_symbol (pd->handle, "init_plugin", (gpointer *) &pd->init_plugin))
  {
    reason = _("Plugin initialization failed: init_plugin() not found");
    goto error;
  }
  
  if ((res = pd->init_plugin (pd)) != 0)
  {
    switch (res)
    {
    case PLUGIN_ERROR_INVALID_API:
      reason = _("Plugin initialization failed: ERROR_INVALID_API");
      break;
    default:
      reason = _("Plugin initialization failed: ERROR_UNKNOWN");
    }
    goto error;
  }

  plugin_list = g_slist_append (plugin_list, pd);
  return pd;

error:
  /* cleanup */
  if (pd && pd->handle)
    g_module_close (pd->handle);
  g_free (pd);

  /* report error */
  dialog_error (win, _("Error loading plugin: %s\n[%s]"), filename, reason);

  return NULL;
}

void
plugin_unload (GtkWindow *win, PluginData *pd)
{
  g_return_if_fail (pd != NULL);
  
  if (pd->can_unload && !pd->can_unload(pd))
  {
    dialog_error (win, _("Unable to unload plugin because it is busy"));
    return;
  }
  
  if (pd->free_plugin)
    pd->free_plugin (pd);

  plugin_list = g_slist_remove (plugin_list, pd);

  g_module_close (pd->handle);
  g_free (pd);
}

static gboolean
plugin_not_in_use (const gchar *plugin_name)
{
  GSList *node;
  PluginData *pd;

  for (node=plugin_list; node; node=node->next)
  {
    pd = LIST_DEREF (PluginData, node);
    if (strcmp (g_basename (plugin_name), g_basename (g_module_name (pd->handle))) == 0)
      return FALSE;
  }
  return TRUE;
}

static void
plugin_load_plugins_in_dir (GtkWindow *win, char *directory)
{
  DIR *d;
  struct dirent *e;
  
  if ((d = opendir (directory)) == NULL)
    return;

  while ((e = readdir (d)) != NULL)
  {
    if (strncmp (e->d_name + strlen (e->d_name) - 3, ".so", 3) == 0)
    {
      char *plugin_name;
      
      plugin_name = g_strconcat (directory, e->d_name, NULL);
      if (plugin_not_in_use (plugin_name))
	plugin_load (win, plugin_name);
      g_free (plugin_name);
    }
  }
  closedir (d);
}

static void
load_all_plugins (GtkWindow *win)
{
  gchar *plugin_dir;
  gchar *home_dir = getenv ("HOME");
  
  /* Load the user plugins */
  plugin_dir = g_strconcat (home_dir ? home_dir : "", "/.gnofin/plugins/", NULL);
  plugin_load_plugins_in_dir (win, plugin_dir);
  g_free (plugin_dir);

  /* Load the system plugins */
#ifdef GNOFIN_LIBDIR
  plugin_dir = g_strdup_printf ("%s/gnofin/plugins/", GNOFIN_LIBDIR);
#else
  plugin_dir = gnome_unconditional_libdir_file ("gnofin/plugins/");
#endif
  plugin_load_plugins_in_dir (win, plugin_dir);
  g_free (plugin_dir);
}

void
plugins_init (GtkWindow *win)
{
  if (!g_module_supported())
    return;

  load_all_plugins (win);
}
