/******************************************************************************\
 gnofin/file-mru.c   $Revision: 1.1 $
 Copyright (C) 2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <string.h>
#include <libgnome/gnome-config.h>
#include "file-mru.h"
#include "file-utils.h"
#include "config-saver.h"


static guint mru_stamp = 0;
static guint mru_limit = 0;
static GSList *mru_list = NULL;


/******************************************************************************
 * Configuration
 */

#define CAT  "MRU"
#define KEY  "/" PACKAGE "/" CAT "/"

static void
load_config (void)
{
  gchar *text, path[256];
  guint i, n;

  trace ("");
  g_return_if_fail (mru_list == NULL);

  mru_limit = gnome_config_get_int (KEY "max_length=8");
  n = gnome_config_get_int (KEY "length=0");

  for (i=0; i < n; ++i)
  {
    g_snprintf (path, sizeof path, KEY "file[%d]", i);

    text = gnome_config_get_string (path);
    mru_list = g_slist_prepend (mru_list, text);
  }
  mru_list = g_slist_reverse (mru_list);
  mru_stamp++;

  file_mru_prune ();
}

static void
save_config (void)
{
  GSList *node;
  gchar path[256];
  guint i;

  trace ("");

  gnome_config_set_int (KEY "max_length", mru_limit);
  gnome_config_set_int (KEY "length", g_slist_length (mru_list));

  node = mru_list;
  for (i=0; node; ++i, node=node->next)
  {
    g_snprintf (path, sizeof path, KEY "file[%d]", i);
    gnome_config_set_string (path, (const gchar *) node->data);
  }
}

static void
init_config (void)
{
  static gboolean init = FALSE;

  if (!init)
  {
    init = TRUE;
    load_config ();
    config_saver_register (CAT, (ConfigSaveFunc) save_config, NULL);
  }
}


/******************************************************************************
 * Interface
 */

void
file_mru_add (const gchar *filename)
{
  GSList *node;

  trace ("%s", filename);
  init_config ();

  node = g_slist_find_custom (mru_list, (gpointer) filename, (GCompareFunc) strcmp);
  if (node)
  {
    /* Remove matching node */
    g_free (node->data);
    mru_list = g_slist_remove_link (mru_list, node);
  }

  mru_list = g_slist_prepend (mru_list, g_strdup (filename));
  mru_stamp++;

  if (g_slist_length (mru_list) > mru_limit)
  {
    node = mru_list;

    g_free (mru_list->data);
    mru_list = mru_list->next;
    g_slist_free_1 (node);
  }
}

void
file_mru_clear (void)
{
  trace ("");
  init_config ();
 
  g_slist_foreach (mru_list, (GFunc) g_free, NULL);
  g_slist_free (mru_list);

  mru_list = NULL;
  mru_stamp++;
}

void
file_mru_prune (void)
{
  GSList *node;
  GSList *new_list = NULL;

  trace ("");
  init_config ();

  for (node=mru_list; node; node=node->next)
  {
    if (!file_exists ((const gchar *) node->data))
      g_free (node->data);
    else
      new_list = g_slist_prepend (new_list, node->data);
  }
  g_slist_free (mru_list);

  mru_list = g_slist_reverse (new_list);
  mru_stamp++;
}

GSList *
file_mru_get (guint *stamp)
{
  trace ("");
  init_config ();

  if (stamp) *stamp = mru_stamp;
  return mru_list;
}

gboolean
file_mru_changed (guint stamp)
{
  init_config ();
  return (stamp < mru_stamp);
}

void
file_mru_set_max_length (guint limit)
{
  trace ("");
  init_config ();

  if ((limit < mru_limit) && (limit < g_slist_length (mru_list)))
  {
    /* Truncate mru list */
    GSList *node;
    guint i = 0;

    for (node=mru_list; node && (i < limit); node=node->next, ++i)
      g_free (node->data);
    g_assert (node);

    node = g_slist_copy (node);
    g_slist_free (mru_list);
    mru_list = node;

    mru_stamp++; 
  }
  mru_limit = limit;
}

guint
file_mru_get_max_length (void)
{
  trace ("");
  init_config ();
  return mru_limit;
}
