/* giFTui
 * Copyright (C) 2003 the giFTui team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "main.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <unistd.h>
#include <libgift/libgift.h>
#include <gtk/gtk.h>

#include "configure.h"
#include "io.h"
#include "event.h"
#include "util.h"

#include "ui_util.h"
#include "ui_connect.h"

void
giftui_connect_free (GiftuiConnect_t *con)
{
	g_return_if_fail (con != NULL);
	
	if (con->host)
		g_free (con->host);
	
	g_free (con);
	
	return;
}

void
giftui_connect (const gchar *host, gint port, gboolean launch)
{
	if (gift_connected ())
		gift_disconnect ();
	
	if (launch)
	{
		gint pid;
		const gchar *bin;
		
		pid = fork ();
		bin = giftui_config_get_str (PREFS_DAEMON_BINARY);
		
		if (pid > 0)
			sleep (1);
		else if (pid == 0)
		{
			execlp (bin, NULL);
			exit (0);
		}
		else
			GIFTUI_PRINT_ERR ((_("Failed to launch %s."), bin));
		
		host = "localhost";
	}
	
	if (gift_connect (host, port))
	{
		giftui_config_set_str (PREFS_DAEMON_HOST, host);
		giftui_config_set_int (PREFS_DAEMON_PORT, port);
		gift_send ("ATTACH", NULL, "client", PACKAGE, "version", VERSION, NULL);
	}
	else
	{
		gchar *message;
		
		message = g_strdup_printf (_("Connection to %s:%i failed."), host, port);
		giftui_connect_create (GTK_STOCK_STOP, message, host, port, launch);
		g_free (message);
	}
	
	return;
}

static void
giftui_connect_launch_clicked (GiftuiConnect_t *con)
{
	const gchar *tmp;
	
	g_return_if_fail (con != NULL);
	
	tmp = gtk_entry_get_text (GTK_ENTRY (con->host_entry));
	con->port = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (con->port_spin));
	
	/* utf-8 -> ascii */
	con->host = str_convert_to_ascii (tmp);
	
	gtk_widget_hide (con->window);
	giftui_connect (con->host, con->port, con->launch);
	gtk_widget_destroy (con->window);
	
	return;
}

static gboolean
giftui_connect_destroy (GiftuiConnect_t *con)
{
	giftui_connect_free (con);
	
	return TRUE;
}

void
giftui_connect_create (const gchar *stock_id, const gchar *text,
		       const gchar *host, gint port, gboolean launch)
{
	GiftuiConnect_t *con = g_new0 (GiftuiConnect_t, 1);
	GtkWidget *vbox, *hbox, *bbox;
	GtkWidget *image, *label, *button;
	
	g_return_if_fail (text != NULL);
	
	/* ascii -> utf-8 */
	if (host)
		con->host = str_convert_to_utf8 (host);
	con->port = port;
	
	con->window = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (con->window), _("Connection"));
	gtk_window_set_modal (GTK_WINDOW (con->window), TRUE);
	g_signal_connect_swapped (con->window, "destroy",
				  G_CALLBACK (giftui_connect_destroy),
				  con);
	
	vbox = GTK_DIALOG (con->window)->vbox;
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
	
	bbox = GTK_DIALOG (con->window)->action_area;
	gtk_container_set_border_width (GTK_CONTAINER (bbox), 5);
	
	hbox = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start_defaults (GTK_BOX (vbox), hbox);
	
	image = gtk_image_new_from_stock (stock_id, GTK_ICON_SIZE_DIALOG);
	gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 5);
	
	label = gtk_label_new (text);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
	
	con->host_entry = widget_entry_add_chars (GTK_BOX (vbox), _("host"), con->host, 256);
	con->port_spin = widget_spin_add_int (GTK_BOX (vbox), _("port"), con->port, 65535, 1, 1);
	
	button = gtk_button_new_from_stock (GTK_STOCK_EXECUTE);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	g_signal_connect_swapped (button, "clicked",
				  G_CALLBACK (giftui_connect_launch_clicked),
				  con);
	gtk_box_pack_start_defaults (GTK_BOX (bbox), button);
	gtk_widget_grab_default (button);
	
	button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	g_signal_connect_swapped (button, "clicked",
				  G_CALLBACK (gtk_widget_destroy),
				  con->window);
	gtk_box_pack_start_defaults (GTK_BOX (bbox), button);
	
	gtk_widget_show_all (con->window);
	
	return;
}
