// VIEWS.H : base classes for "views", to be used with the "docview" class.

// Copyright (C) 1998 Tommi Hassinen, Jarno Huuskonen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef VIEWS_H
#define VIEWS_H

#ifdef ENABLE_GRAPHICS

class view;
class ogl_view;

class graphics_view;

class enlevdiag_view;

class plotting_view;
class plot1d_view;
class plot2d_view;

class treelist_view;

class graphics_class_factory;

/*################################################################################################*/

class camera;		// camera.h

class color_mode;	// model.h
class docview;		// docview.h

#include "mtools.h"
#include "factory.h"

#include "qm1mdl.h"	// energy level diagrams...
#include "mm1mdl.h"	// plot1d and plot2d...

/*################################################################################################*/

/**	The "##view" is a base class for all views. All things dependent on operating 
	system and/or windowing environment are to be wrapped in ctor/dtor or in a set of 
	pure virtual functions.
	
	The "##view"-objects should not be created directly, but a "##view_factory" object 
	should create them instead. Different platforms/operating systems might have different 
	and tricky requirements on how windows should be created and initialized there. It's 
	up to "##view_factory" and/or "##view" itself to meet these requirements, using 
	specific callbacks or whatever is necessary...
*/

class view
{
	protected:
	
	// no relevant common data!??!!?
	// no relevant common data!??!!?
	// no relevant common data!??!!?
	
	public:
	
	view(void);
	virtual ~view(void);
	
	virtual docview * GetDV(void) = 0;
	
	/**	This should update the visible contents of this view. 
		The parameter is a "mode" of update, either "##immediate" (true) or 
		"##as soon as possible" (false, is default).
	*/
	
	virtual void Update(bool = false) = 0;
	
	/// This should set the window title text for this view.
	
	virtual void SetTitle(const char *) = 0;
};

/*################################################################################################*/

/**	The "##ogl_view" is a view that will use OpenGL graphics to present it's contents, 
	and is (optionally) connected to the standard set of mouse tool classes.
*/

class ogl_view : virtual public view
{
	protected:
	
	i32s size[2];	// size of the viewing window, in pixels.
	
	// static data about mouse/keyboard state...
	
	static mouse_tool::state state;
	static mouse_tool::button button;
	
	static bool shift_down;
	static bool ctrl_down;
	
	// all available mouse tools are here...
	
	static draw_tool tool_draw;
	static erase_tool tool_erase;
	
	static select_tool tool_select;
	static select_tool::select_mode select_mode;
	static zoom_tool tool_zoom;
	
	static translate_xy_tool tool_translate_xy;
	static translate_z_tool tool_translate_z;
	
	static orbit_xy_tool tool_orbit_xy;
	static orbit_z_tool tool_orbit_z;
	
	static rotate_xy_tool tool_rotate_xy;
	static rotate_z_tool tool_rotate_z;

	static measure_tool tool_measure;
	static measure_tool::measure_mode measure_mode;
	
	friend class camera;
	friend class gnome_mdi_app;
	
	public:
	
	// pointer to the currently selected mouse tool.
	// OBSOLETE?!?!? SOMETHING ELSE IS NEEDED FOR THIS SOON!!!
	// OBSOLETE?!?!? SOMETHING ELSE IS NEEDED FOR THIS SOON!!!
	// OBSOLETE?!?!? SOMETHING ELSE IS NEEDED FOR THIS SOON!!!
	
	static mouse_tool * current_tool;
	
	public:
	
	ogl_view(void);
	virtual ~ogl_view(void);
	
	void SetSize(i32s, i32s);
	
	i32s GetGlutStringWidth(const char *, void * = NULL);
	void WriteGlutString2D(const char *, fGL, fGL, void * = NULL);
	void WriteGlutString3D(const char *, fGL, fGL, fGL, const obj_loc_data *, void * = NULL);
	
	/**	This should do all what is necessary to make the OpenGL rendering 
		context of this view "##current", so that the following OpenGL 
		commands would affect this view.
	*/
	
	virtual void SetCurrent(void) = 0;
	
	/// This should contain the necessary OpenGL initialization commands.
	
	virtual void InitGL(void) = 0;
	
	/// This should contain the OpenGL commands that create the visible contents of this view.
	
	virtual void Render(void) = 0;
};

/*################################################################################################*/

/// "##graphics_view" presents the molecular models in OpenGL 3D graphics.

class graphics_view : virtual public ogl_view
{
	protected:
	
	camera * cam;
	
	fGL range[2];	// dimensions of the view, in nanometers.
	
	i32s render; i32s label;
	color_mode * colormode;
	
	bool enable_fog;
	bool accumulate;
	
	static bool quick_update;	// some static flags...
	static bool draw_info;		// some static flags...
	
	friend class draw_tool;
	friend class erase_tool;
	friend class select_tool;
	friend class zoom_tool;
	friend class translate_xy_tool;
	friend class translate_z_tool;
	friend class orbit_xy_tool;
	friend class orbit_z_tool;
	friend class rotate_xy_tool;
	friend class rotate_z_tool;
	friend class measure_tool;
	
	friend class camera;
	
	friend class docview;
	
	friend class qm1_docv;
	friend class glut_qm1_docv;
	
	friend class qm2_docv;
	friend class glut_qm2_docv;
	
	friend class mm1_docv;
	friend class glut_mm1_docv;
	
	friend class mm2_docv;
	friend class glut_mm2_docv;
	
	friend class mm3_docv;
	friend class glut_mm3_docv;
	
	friend class gnome_graphics_view;	// attach/detach needs this...
	
	public:
	
	graphics_view(camera *);
	virtual ~graphics_view(void);
	
	void GetCRD(i32s *, fGL *);
	
	void InitGL(void);	// virtual
	void Render(void);	// virtual
};

/*################################################################################################*/

void * mm1_convert_cset_to_plotting_udata(mm1_mdl *, i32s);	// also docv???
void mm1_apply_udata_as_cset(mm1_docv *, void *);

/**	"##plotting_view" is reserved for 1d and 2d plotting...

	It would be also nice to develop some kind of spreadsheet-program interface
	that could be used for plotting. Gnumeric?!?!?! Sci-Graphica?!?!?!
*/

#define PLOT_USERDATA_IGNORE		0x0000
#define PLOT_USERDATA_STRUCTURE		0x0001

class plotting_view : virtual public ogl_view
{
	protected:
	
	model_simple * mdl;
	iGLu * select_buffer;
	
	protected:
	
/**	If the "##plot_userdata1" is PLOT_USERDATA_STRUCTURE, the udata is interpreted to contain
	molecule's 3D coordinates (3N fGL values in an array), and the coordinate sets are updated
	using this data when data is selected.
	
	This same class works for all models, and it requires some tricks and it's against the
	principle of making different derived classes for different tasks. But since the mechanism
	of making view classes is quite complicated, it seems to make sense here...
*/
	i32s plot_userdata1;
	i32s plot_userdata2;
	
	public:
	
	plotting_view(model_simple *, i32s, i32s);
	virtual ~plotting_view(void);
	
	void InitGL(void);	// virtual
	
/**	ZoomEvent() is triggered by zoom_tool::MotionEvent(), passing delta-Y as parameter.
	It is used by plot2d_view and enlevdiag_view to scale the plots.
*/
	virtual void ZoomEvent(i32s) = 0;
	
/**	TransEvent() is triggered by translate_xy_tool::MotionEvent(), passing delta-Y as parameter.
	It is used by enlevdiag_view to scroll the diagram in Y-direction.
*/
	virtual void TransEvent(i32s) = 0;
	
/**	UserEvent() is triggered by all other mouse events except draw/erase, passing X and Y as parameters.
	It is used to handle the selection events of plotting views.
*/
	virtual void UserEvent(i32s, i32s) = 0;
};

/*################################################################################################*/

struct plot1d_data
{
	f64 c1;			// coordinate 1 (say, x-coordinate)
	f64 v;			// value
	
	void * udata;		// user data
};

/// This class stores a set of 1D data, and is able to display it.

class plot1d_view : public plotting_view
{
	protected:
	
	vector<plot1d_data> dv;
	
	f64 min1; f64 max1;
	f64 minv; f64 maxv;
	
	public:
	
	plot1d_view(model_simple *, i32s, i32s);
	virtual ~plot1d_view(void);
	
/**	AddData() is used to add data points.
	The points should be added in correct (increasing) order.
	See mm1_docv::DoEnergyPlot1D() as an example case...
*/
	void AddData(f64, f64, void * = NULL);
	void SetCenterAndScale(void);
	
	void Render(void);		// virtual
	
	void ZoomEvent(i32s);			// virtual
	void TransEvent(i32s);			// virtual
	void UserEvent(i32s, i32s);		// virtual
};

/*################################################################################################*/

struct plot2d_data
{
	f64 c1;			// coordinate 1 (say, x-coordinate)
	f64 c2;			// coordinate 2 (say, y-coordinate)
	f64 v;			// value
	
	void * udata;		// user data
};

/// This class stores a set of 2D data, and is able to display it.

class plot2d_view : public plotting_view
{
	protected:

	vector<plot2d_data> dv;

	f64 min1; f64 max1;
	f64 min2; f64 max2;
	f64 minv; f64 maxv;
	
	void SetColor(f64);
	
	public:
	
	plot2d_view(model_simple *, i32s, i32s);
	virtual ~plot2d_view(void);
	
/**	AddData() is used to add data points.
	Only "##square" data sets can be used, with equal number of points in both axes (x points per axis -> x*x total points).
	The points should be added in correct (increasing) order, using the following rule of ordering : index = (1st * x) + 2nd
	See mm1_docv::DoEnergyPlot2D() as an example case...
*/
	void AddData(f64, f64, f64, void * = NULL);
	void SetCenterAndScale(void);
	
	void Render(void);		// virtual
	
	void ZoomEvent(i32s);			// virtual
	void TransEvent(i32s);			// virtual
	void UserEvent(i32s, i32s);		// virtual
};

/*################################################################################################*/

/// enlevdiag_view is an energy level diagram for QM models.

// make a member of plotting_view ?!?!?!?
// make a member of plotting_view ?!?!?!?
// make a member of plotting_view ?!?!?!?

class enlevdiag_view : public plotting_view
{
	protected:
	
	qm1_mdl * mdl;
	
	fGL center;
	fGL scale;
	
	public:
	
	enlevdiag_view(qm1_mdl *);
	~enlevdiag_view(void);
	
	void SetCenterAndScale(void);
	
	void Render(void);	// virtual
	
	void ZoomEvent(i32s);			// virtual
	void TransEvent(i32s);			// virtual
	void UserEvent(i32s, i32s);		// virtual
};

/*################################################################################################*/

#ifdef ENABLE_TREELIST_VIEW

/// "##treelist_view" is made optional, since some complicated system services are needed...

class treelist_view : virtual public view
{
	public:
	
	treelist_view(void);
	virtual ~treelist_view(void);
	
	virtual void LightAdded(light *) = 0;
	virtual void LightRemoved(light *) = 0;
	
	virtual void ObjectAdded(smart_object *) = 0;
	virtual void ObjectRemoved(smart_object *) = 0;
};

#endif	// ENABLE_TREELIST_VIEW

/*################################################################################################*/

/**	"##graphics_class_factory" is a graphics_class capable to create also "view" objects.

	For each different platform (just like a "##view") a "##graphics_class_factory" should
	be made which produces new "view" objects for this particular platform...
*/

class graphics_class_factory : public class_factory
{
	protected:
	
	public:
	
	graphics_class_factory(void) : class_factory() { };
	virtual ~graphics_class_factory(void) { };
	
	virtual graphics_view * ProduceGraphicsView(docview *, camera *, bool) = 0;
	
	virtual plot1d_view * ProducePlot1DView(docview *, i32s, i32s) = 0;
	virtual plot2d_view * ProducePlot2DView(docview *, i32s, i32s) = 0;
	
	virtual enlevdiag_view * ProduceEnLevDiagView(qm1_mdl *) = 0;
	
#ifdef ENABLE_TREELIST_VIEW

	virtual treelist_view * ProduceTreeListView(docview *) = 0;
	
#endif	// ENABLE_TREELIST_VIEW

};

/*################################################################################################*/

#endif	// ENABLE_GRAPHICS
#endif	// VIEWS_H

// eof
