// MM1MDL.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm1mdl.h"
#include "v3d.h"

#include "mm1eng1.h"
#include "mm1eng9.h"

#include "filetrans.h"

#include <fstream>
#include <strstream>
#include <algorithm>
using namespace std;

/*################################################################################################*/

mm1_sequencebuilder mm1_mdl::amino_builder(mm1_chn_info::amino_acid, MM1_AMINO_BUILDER_FILE);
mm1_sequencebuilder mm1_mdl::nucleic_builder(mm1_chn_info::nucleic_acid, MM1_NUCLEIC_BUILDER_FILE);

mm1_mdl::mm1_mdl(ostream * p1, class_factory & p2) :
	model_extended(p1, p2), model_simple(p1, p2), all_atoms_interface(), trajectory_interface()
{
	nmol = NOT_DEFINED;
	ref_civ = NULL;
	
	default_eng = 0; periodic = false;
	box_hdim[0] = 0.5; box_fdim[0] = 2.0 * box_hdim[0];
	box_hdim[1] = 1.0; box_fdim[1] = 2.0 * box_hdim[1];
	box_hdim[2] = 1.5; box_fdim[2] = 2.0 * box_hdim[2];
	
	ms = this;	// trajectory_interface
}

mm1_mdl::~mm1_mdl(void)
{
}

const char * mm1_mdl::GetProjectFileNameExtension(void)
{
	static const char ext[] = "gpr";
	return ext;
}

/*##############################################*/
/*##############################################*/

const char * mm1_mdl::engtab1[] =
{
	"mm1_eng_exp1 : the default engine",
	"mm1_eng_exp1_mim : the periodic engine (minimum image model)",
//	"mm1_eng_prmfit : this_is_not_ready_yet!!!",
	NULL
};

const i32s mm1_mdl::engtab2[] =
{
	ENG_MM1_NORMAL,
	ENG_MM1_PERIODIC,
//	ENG_MM1_EXPERIMENTAL,
	NOT_DEFINED
};

mm1_eng * mm1_mdl::CreateDefaultEngine(void)
{
	prmfit_tables * tab = NULL; mm1_eng_prmfit * eng = NULL;
	const char forcefield_path[] = "/home/thassine/DATABASE-mm1/organic_molecules/parameters";
	
	switch (engtab2[default_eng])
	{
		case ENG_MM1_PERIODIC:
		periodic = true; GatherGroups();
		return new mm1_eng_exp1_mim(* this);

		case ENG_MM1_EXPERIMENTAL:
		periodic = false;
		tab = new prmfit_tables(forcefield_path);	// experimental!!! for debugging only!!!
		eng = new mm1_eng_prmfit(* this, * tab);	// experimental!!! for debugging only!!!
		delete tab; return eng;				// experimental!!! for debugging only!!!
		
		default:	// ENG_MM1_NORMAL
		periodic = false;
		return new mm1_eng_exp1(* this);
	}
}

/*##############################################*/
/*##############################################*/

void mm1_mdl::PushCRDSets(i32u p1)
{
	for (i32u n1 = 0;n1 < p1;n1++) cs_vector.push_back(new crd_set());
	
	fGL_a3 newcrd = { 0.0, 0.0, 0.0 };
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		for (i32u n1 = 0;n1 < p1;n1++)
		{
			(* it1).crd_vector.push_back(newcrd);
		}
	}
}

void mm1_mdl::PopCRDSets(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void mm1_mdl::CopyCRDSet(i32u p1, i32u p2)
{
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		(* it1).crd_vector[p2][0] = (* it1).crd_vector[p1][0];
		(* it1).crd_vector[p2][1] = (* it1).crd_vector[p1][1];
		(* it1).crd_vector[p2][2] = (* it1).crd_vector[p1][2];
	}
}

void mm1_mdl::SwapCRDSets(i32u, i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

void mm1_mdl::CenterCRDSet(i32u p1)
{
	fGL sum[3] = { 0.0, 0.0, 0.0 };
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		for (i32s n1 = 0;n1 < 3;n1++) sum[n1] += (* it1).crd_vector[p1][n1];
	}
	
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		for (i32s n1 = 0;n1 < 3;n1++)
		{
			fGL tmp1 = sum[n1] / (f64) atom_list.size();
			(* it1).crd_vector[p1][n1] -= tmp1;
		}
	}
}

void mm1_mdl::ReserveCRDSets(i32u)
{
cout << "Oops!!! This function is not yet ready." << endl;
//exit(EXIT_FAILURE);
}

/*##############################################*/
/*##############################################*/

void mm1_mdl::AddAtom(mm1_atom & p1)
{
	InvalidateGroups();
	atom_list.push_back(p1);
}

void mm1_mdl::RemoveAtom(iter_mm1al it1)
{
	InvalidateGroups();
	
	// this strange while-loop is needed because removing bonds
	// will invalidate all those atom::cr_list-iterators...
	
	while ((* it1).cr_list.begin() != (* it1).cr_list.end())
	{
		mm1_cr * ref = & (* it1).cr_list.back();
		iter_mm1bl it2 = find(bond_list.begin(), bond_list.end(), (* ref->bndr));
		
		if (it2 != bond_list.end()) RemoveBond(it2);
		else if (ostr != NULL) (* ostr) << "bad news: find failed in RA" << endl;
	}
	
	atom_list.erase(it1);
}

void mm1_mdl::AddBond(mm1_bond & p1)
{
	InvalidateGroups();
	bond_list.push_back(p1);
	
	mm1_cr info1 = mm1_cr(p1.atmr[1], & bond_list.back());
	p1.atmr[0]->cr_list.push_back(info1);
	
	mm1_cr info2 = mm1_cr(p1.atmr[0], & bond_list.back());
	p1.atmr[1]->cr_list.push_back(info2);
}

void mm1_mdl::RemoveBond(iter_mm1bl it1)
{
	InvalidateGroups();
	mm1_cr tmpinfo = mm1_cr(NULL, & (* it1));
	iter_mm1cl it2;
	
	it2 = find((* it1).atmr[0]->cr_list.begin(), (* it1).atmr[0]->cr_list.end(), tmpinfo);
	if (it2 != (* it1).atmr[0]->cr_list.end()) (* it1).atmr[0]->cr_list.erase(it2);
	else if (ostr != NULL) (* ostr) << "bad news: find #1 failed in RB" << endl;
	
	it2 = find((* it1).atmr[1]->cr_list.begin(), (* it1).atmr[1]->cr_list.end(), tmpinfo);
	if (it2 != (* it1).atmr[1]->cr_list.end()) (* it1).atmr[1]->cr_list.erase(it2);
	else if (ostr != NULL) (* ostr) << "bad news: find #2 failed in RB" << endl;
	
	bond_list.erase(it1);
}

/*##############################################*/
/*##############################################*/

bool mm1_mdl::ReadStream_OLD(istream & p1, bool, bool)
{
	static const char message[] = "Hmmm. It seems that we are trying to read a VERY OLD version of this file.\nIt's not possible to verify the type and version of the file, so this\noperation may fail. Are you sure you wish to try (Y/N) ??? ";
	
	bool result = err->Question(message);
	if (!result) return false;
	
	vector<mm1_atom *> tmp1; i32s tmp2[2]; char tmp3; fGL tmp4[3];
	
#ifdef VERBOSE
cout << "mm1_mdl::ReadStream_OLD() : starting file input." << endl;
#endif	// VERBOSE

	while (p1.peek() != 'E')
	{
		p1 >> tmp2[0] >> tmp2[1];
		p1 >> tmp4[0] >> tmp4[1] >> tmp4[2] >> tmp2[0]; p1.get();
		
		element el = element(tmp2[1]);
		mm1_atom newatom(el, tmp4, cs_vector.size());
		AddAtom(newatom); tmp1.push_back(& atom_list.back());
		
		for (i32s n1 = 0;n1 < tmp2[0];n1++)
		{
			p1 >> tmp2[1] >> tmp3;
			if (tmp2[1] < (i32s) tmp1.size())
			{
				mm1_bond newbond(tmp1[tmp2[1]], tmp1[tmp1.size() - 1], bondtype(tmp3));
				AddBond(newbond);
			}
		}
		
		p1.getline(buffer, sizeof(buffer));
	}
	
	return true;
}

bool mm1_mdl::ReadStream(istream & p1, bool selected, bool skipheader)
{
	if (!skipheader)
	{
		// first we have to identify the file version, and then call the matching function,
		// if necessary. the very first file format did not use the "!Section" idea, so
		// we must skip the test if no such traces are seen...
		
		int version;
		if (p1.peek() != '!') version = -1;
		else
		{
			// now we expect to see "!Header" section.
			// next we will check the file type and version information...
			
			p1 >> buffer;		// read "!Header".
			p1 >> buffer;		// read the file type.
			p1 >> version;		// read the version number;
			
			p1.getline(buffer, sizeof(buffer));
		}
		
		// now look at the version number, and pick a correct function if needed.
		
		if (version < 100) return ReadStream_OLD(p1, selected, true);
		else if (version > 100)
		{
			static const char message[] = "Ooops. It seems that we are trying to read a FUTURE version of this file.\nIt's not sure if we can handle that. Are you sure you wish to try (Y/N) ??? ";
			
			bool result = err->Question(message);
			if (!result) return false;
		}
	}
	
	// the "!Header" section is now processed,
	// and the actual file input begins!!!
	
#ifdef VERBOSE
cout << "mm1_mdl::ReadStream() : starting file input." << endl;
#endif	// VERBOSE

	i32u crd_set_counter = 0;
	vector<mm1_atom *> ar_vector;
	
	// now we just read the sections, until we hit into "!End"-section and leave the loop...
	
	while (true)
	{
		p1 >> buffer;		// read the section string.
		
		if (!strcmp(buffer, "!End")) break;		// handle "!End" section.
		else if (!strcmp(buffer, "!Info"))		// handle "!Info" section.
		{
			int n1; p1 >> n1;
			
			int n2 = n1 - cs_vector.size();
			if (n2 > 0)
			{
				i32u n3 = cs_vector.size(); PushCRDSets((i32u) n2);
				for (i32u n4 = n3;n4 < cs_vector.size();n4++) SetCRDSetVisible(n4, true);
			}
			
#ifdef VERBOSE
cout << "handled \"!Info\" section, added " << n2 << " coordinate sets." << endl;
#endif	// VERBOSE

			p1.getline(buffer, sizeof(buffer));
			
			// TODO : read coordinate set descriptions (if there are some)...
			// TODO : read coordinate set descriptions (if there are some)...
			// TODO : read coordinate set descriptions (if there are some)...
		}
		else if (!strcmp(buffer, "!Atoms"))		// handle "!Atoms" section.
		{
			p1.getline(buffer, sizeof(buffer));
			
#ifdef VERBOSE
cout << "found \"!Atoms\" section..." << endl;
#endif	// VERBOSE

			while (p1.peek() != '!')
			{
				i32u n1; p1 >> n1;	// id number
				i32s n2; p1 >> n2;	// atomic number
				
#ifdef VERBOSE
if (n1 != ar_vector.size()) cout << "WARNING : bad atom id numbers!!!" << endl;
#endif	// VERBOSE

				element tmp_element(n2);
				mm1_atom newatom(tmp_element, NULL, cs_vector.size());
				if (selected) newatom.selected = true;
				AddAtom(newatom);

				ar_vector.push_back(& atom_list.back());
				
				p1.getline(buffer, sizeof(buffer));
			}
		}
		else if (!strcmp(buffer, "!Bonds"))		// handle "!Bonds" section.
		{
			p1.getline(buffer, sizeof(buffer));
			
#ifdef VERBOSE
cout << "found \"!Bonds\" section..." << endl;
#endif	// VERBOSE

			while (p1.peek() != '!')
			{
				i32u id1; p1 >> id1;	// 1st id number
				i32u id2; p1 >> id2;	// 2nd id number
				
				char bt; p1 >> bt;	// bond type
				
#ifdef VERBOSE
if (id1 >= ar_vector.size()) cout << "WARNING : bad atom id numbers!!!" << endl;
if (id2 >= ar_vector.size()) cout << "WARNING : bad atom id numbers!!!" << endl;
#endif	// VERBOSE

				bondtype tmp_bondtype(bt);
				mm1_bond newbond(ar_vector[id1], ar_vector[id2], tmp_bondtype);
				AddBond(newbond);
				
				p1.getline(buffer, sizeof(buffer));
			}
		}
		else if (!strcmp(buffer, "!Coord"))		// handle "!Coord" section.
		{
			p1.getline(buffer, sizeof(buffer));
			
#ifdef VERBOSE
cout << "found \"!Coord\" section..." << endl;
if (crd_set_counter >= cs_vector.size()) cout << "ERROR : too many crd-sets!!!" << endl;
#endif	// VERBOSE

			while (p1.peek() != '!')
			{
				i32u n1; p1 >> n1;	// id number
				
				fGL xcrd; p1 >> xcrd;	// x-coordinate
				fGL ycrd; p1 >> ycrd;	// y-coordinate
				fGL zcrd; p1 >> zcrd;	// z-coordinate
				
#ifdef VERBOSE
if (n1 >= ar_vector.size()) cout << "ERROR : bad atom id numbers!!!" << endl;
#endif	// VERBOSE

				mm1_atom * ar = ar_vector[n1];
				ar->crd_vector[crd_set_counter][0] = xcrd;
				ar->crd_vector[crd_set_counter][1] = ycrd;
				ar->crd_vector[crd_set_counter][2] = zcrd;
				
				p1.getline(buffer, sizeof(buffer));
			}
			
			crd_set_counter++;
		}
		else if (!strcmp(buffer, "!Charges"))		// handle "!Charges" section.
		{
			p1.getline(buffer, sizeof(buffer));
			
#ifdef VERBOSE
cout << "found \"!Charges\" section..." << endl;
#endif	// VERBOSE

			while (p1.peek() != '!')
			{
				i32u n1; p1 >> n1;	// id number
				f64 n2; p1 >> n2;	// charge
				
#ifdef VERBOSE
if (n1 >= ar_vector.size()) cout << "WARNING : bad atom id numbers!!!" << endl;
#endif	// VERBOSE

				mm1_atom * ar = ar_vector[n1];
				ar->charge = n2;
				
				p1.getline(buffer, sizeof(buffer));
			}
		}
		else						// handle an unknown section...
		{
			p1.getline(buffer, sizeof(buffer));
			
			while (p1.peek() != '!')
			{
				p1.getline(buffer, sizeof(buffer));
			}
		}
	}
	
	return true;
}

// obsolete file output functions are also here, but are commented out. just for reference and for emergency cases...
// obsolete file output functions are also here, but are commented out. just for reference and for emergency cases...
// obsolete file output functions are also here, but are commented out. just for reference and for emergency cases...

/*	// obsolete!!!

// this fileformat is not documented yet... there is not even any identifier
// or any record about the version -> it might be difficult to read these files later.

void mm1_mdl::WriteStream_OLD(ostream & p1)
{
	UpdateIndex();
	
	iter_mm1al it1 = atom_list.begin();
	while (it1 != atom_list.end())
	{
		p1 << (* it1).index << " " << (* it1).el.GetAtomicNumber() << " ";
		p1 << (* it1).crd_vector[0][0] << " " << (* it1).crd_vector[0][1] << " " << (* it1).crd_vector[0][2] << " ";
		
		vector<i32s> tmp1;	// this is for atomic numbers...
		vector<char> tmp2;	// this is for bond type symbols...
		
		iter_mm1cl it2;
		for (it2 = (* it1).cr_list.begin();it2 != (* it1).cr_list.end();it2++)
		{
			tmp1.push_back((* it2).atmr->index);
			tmp2.push_back((* it2).bndr->bt.GetSymbol1());
		}
		
		p1 << tmp1.size() << ": "; it1++;
		for (i32u n1 = 0;n1 < tmp1.size();n1++)
		{
			p1 << tmp1[n1] << " ";
			p1 << tmp2[n1] << " ";
		}
		
		p1 << endl;
	}
	
	p1 << "END" << endl;
}

*/	// obsolete!!!

// WHAT WE NEED TO ADD WHEN WE CHANGE THE FORMAT NEXT TIME:
// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
// [just add your ideas here]

void mm1_mdl::WriteStream(ostream & p1)
{
	const int current_version_number = 100;		// version number multiplied by 100 -> int.
	
	p1 << "!Header ";
	p1 << GetProjectFileNameExtension() << " ";
	p1 << current_version_number << endl;
	
	// the "!Header" section is now processed, and the actual file output begins!!!
	// the "!Header" section is now processed, and the actual file output begins!!!
	// the "!Header" section is now processed, and the actual file output begins!!!
	
	// write "!Info" section. write the number of coordinate sets,
	// which should match to the number of "!Coord"-sections later in the file.
	
	p1 << "!Info ";
	p1 << cs_vector.size() << endl;
	
// TODO : write visible-flag and set descriptions (if there are some), for example:
// 0 1 "1st coordinate set is this one, and it's visible"
// 1 1 "2nd set looks like this!, and it's also visible"
// 2 0 "this 3rd set is not visible"

	// write "!Atoms" section. write the number of atoms,
	// which should match to the number of records in this section.
	
	p1 << "!Atoms ";
	p1 << atom_list.size() << endl;
	
	UpdateIndex();		// this will update the "index"-records...
	
	iter_mm1al it1 = atom_list.begin();
	while (it1 != atom_list.end())
	{
		p1 << (* it1).index << " ";
		p1 << (* it1).el.GetAtomicNumber() << " ";
		
		// what else should we print here ?!?!?!?
		// selections?
		
		p1 << endl;
		it1++;
	}
	
	// write "!Bonds" section. write the number of bonds,
	// which should match to the number of records in this section.
	
	p1 << "!Bonds ";
	p1 << bond_list.size() << endl;
	
	iter_mm1bl it2 = bond_list.begin();
	while (it2 != bond_list.end())
	{
		p1 << (* it2).atmr[0]->index << " ";
		p1 << (* it2).atmr[1]->index << " ";
		
		p1 << (* it2).bt.GetSymbol1() << " ";
		
		// what else should we print here ?!?!?!?
		// nothing???
		
		p1 << endl;
		it2++;
	}
	
	// write the "!Coord" sections...
	
	for (i32u n1 = 0;n1 < cs_vector.size();n1++)
	{
		p1 << "!Coord" << endl;
		
		it1 = atom_list.begin();
		while (it1 != atom_list.end())
		{
			p1 << (* it1).index << " ";
			
			p1 << (* it1).crd_vector[n1][0] << " ";
			p1 << (* it1).crd_vector[n1][1] << " ";
			p1 << (* it1).crd_vector[n1][2] << " ";
			
			// what else should we print here ?!?!?!?
			// nothing???
			
			p1 << endl;
			it1++;
		}
	}
	
	// write the !Charges section.
	
	p1 << "!Charges" << endl;
	
	it1 = atom_list.begin();
	while (it1 != atom_list.end())
	{
		p1 << (* it1).index << " ";
		p1 << (* it1).charge;
		
		p1 << endl;
		it1++;
	}
	
	// write "!End" section.
	
	p1 << "!End" << endl;
}

bool mm1_mdl::ImportFile(const char *filename, int index)
{
  ifstream ifile;
  ostrstream intermed;
  file_trans translator;
  
  if (index == 0) // Automatic detection
    {
      if (!translator.CanImport(filename))
	{
	  err->ErrorMessage("Cannot import that file type.");
	  return false;
	}
      
      ifile.open(filename, ios::in);
      translator.Import(filename, ifile, intermed);
      ifile.close();
    }
  else // By type picked by the user
    {
      ifile.open(filename, ios::in);
      translator.Import(filename, index - 1, ifile, intermed);
      ifile.close();
    }

  istrstream interInput(intermed.str());
  return ReadStream(interInput, false);
}

void mm1_mdl::AddHydrogens(void)
{
  // this will use OpenBabel to add hydrogens!!!
  // works fine, but makes Kekule structures out of conjugated bonds!
	
  	file_trans ft;
  	OBMol *obm = ft.CopyAll(this);
	obm->AddHydrogens(false, true);
	// we only want to add atoms and bonds, not change existing ones
  	ft.Synchronize(true);
	
	// The old version can be found in the CVS tree
	// cvs update -r 1.36 mm1mdl.cpp
	
	PrintToLog("Hydrogens added.\n");
}

void mm1_mdl::RemoveHydrogens(void)
{
	iter_mm1bl it1 = bond_list.begin();
	while (it1 != bond_list.end())
	{
		bool flag = false;
		if ((* it1).atmr[0]->el.GetAtomicNumber() == 1) flag = true;
		if ((* it1).atmr[1]->el.GetAtomicNumber() == 1) flag = true;
		iter_mm1bl it2 = it1++; if (flag) RemoveBond(it2);
	}
	
	iter_mm1al it2 = atom_list.begin();
	while (it2 != atom_list.end())
	{
		iter_mm1al it3 = it2++;
		if ((* it3).el.GetAtomicNumber() == 1) RemoveAtom(it3);
	}
	
	PrintToLog("Hydrogens removed.\n");
}

iter_mm1al mm1_mdl::FindAtomByIndex(i32s index)
{
	iter_mm1al end_it = GetAtomsEnd();
	if (index < 0) return end_it;
	
	i32s counter = 0;
	iter_mm1al iter = GetAtomsBegin();
	while (counter != index)
	{
		counter++; iter++;
		if (iter == end_it) return end_it;
	}
	
	return iter;
}

void mm1_mdl::Clear(void)
{
	if (!bond_list.empty())
	  {
	    iter_mm1bl it1 = bond_list.begin();
	    while (it1 != bond_list.end())
	      {
		iter_mm1bl it2 = it1++;
		RemoveBond(it2);
	      }
	  }

	if (!atom_list.empty())
	  {
	    iter_mm1al it2 = atom_list.begin();
	    while (it2 != atom_list.end())
	      {
		iter_mm1al it3 = it2++;
		RemoveAtom(it3);
	      }
	  }
}

/*##############################################*/
/*##############################################*/

void mm1_mdl::GatherGroups(void)
{
	InvalidateGroups(); iter_mm1al it1;
	nmol = 0; ref_civ = new vector<mm1_chn_info>;
	
	for (it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		(* it1).id[0] = (* it1).id[1] = (* it1).id[2] = (* it1).id[3] = NOT_DEFINED;
	}
	
	while (true)
	{
		for (it1 = atom_list.begin();it1 != atom_list.end() && (* it1).id[0] != NOT_DEFINED;it1++);
		if (it1 != atom_list.end()) GatherAtoms(& (* it1), nmol++); else break;
	}
	
	atom_list.sort();
}

void mm1_mdl::GetRange(i32s ind, i32s value, iter_mm1al * result)
{
	iter_mm1al range[2] = { atom_list.begin(), atom_list.end() };
	GetRange(ind, range, value, result);
}

void mm1_mdl::GetRange(i32s ind, iter_mm1al * range, i32s value, iter_mm1al * result)
{
	result[0] = range[0]; while (result[0] != range[1] && (* result[0]).id[ind] != value) result[0]++;
	result[1] = result[0]; while (result[1] != range[1] && (* result[1]).id[ind] == value) result[1]++;
}

void mm1_mdl::InvalidateGroups(void)
{
	nmol = NOT_DEFINED;
	if (ref_civ != NULL)
	{
		delete ref_civ;
		ref_civ = NULL;
	}
}

i32s mm1_mdl::FindPath(mm1_atom * ref1, mm1_atom * ref2, i32s max, i32s flag, i32s dist)
{
	if (ref1 == ref2) return dist;
	if (dist == max) return NOT_FOUND;
	
	i32s tmp1 = NOT_FOUND; iter_mm1cl it1;
	for (it1 = ref1->cr_list.begin();it1 != ref1->cr_list.end();it1++)
	{
		if ((* it1).bndr->flags[flag]) continue;
		
		(* it1).bndr->flags[flag] = true;
		i32s tmp2 = FindPath((* it1).atmr, ref2, max, flag, dist + 1);
		(* it1).bndr->flags[flag] = false; if (tmp2 < tmp1) tmp1 = tmp2;
	}
	
	return tmp1;
}

bool mm1_mdl::FindRing(mm1_atom * ref1, mm1_atom * ref2, signed char * str, i32s size, i32s flag, i32s dist)
{
	static vector<signed char> ring_vector;
		
	if (!dist && str != NULL) ring_vector.resize(0);
	else if (dist && ref1 == ref2)
	{
		if (dist != size) return false;
		
		if (str != NULL)
		{
			for (i32u n1 = 0;n1 < strlen((const char *) str);n1++)
			{
				if (!(n1 % 2) && str[n1] == '?') continue;
				if ((n1 % 2) && str[n1] == NOT_DEFINED) continue;
				if (str[n1] != ring_vector[n1]) return false;
			}
		}
		
		return true;
	}
	
	if (dist == size) return false; iter_mm1cl it1;
	for (it1 = ref1->cr_list.begin();it1 != ref1->cr_list.end();it1++)
	{
		if ((* it1).bndr->flags[flag]) continue;
		
		if (str != NULL)
		{
			ring_vector.push_back((* it1).bndr->bt.GetSymbol2());
			ring_vector.push_back((signed char) (* it1).atmr->el.GetAtomicNumber());
		}
		
		(* it1).bndr->flags[flag] = true;
		bool result = FindRing((* it1).atmr, ref2, str, size, flag, dist + 1);
		(* it1).bndr->flags[flag] = false;
		
		if (str != NULL)
		{
			ring_vector.pop_back();
			ring_vector.pop_back();
		}
		
		if (result) return true;
	}
	
	return false;
}

void mm1_mdl::UpdateIndex(void)
{
	i32s counter = 0;
	iter_mm1al it1 = atom_list.begin();
	while (it1 != atom_list.end())
	{
		(* it1++).index = counter++;
	}
}

void mm1_mdl::GatherAtoms(mm1_atom * ref, i32s id)
{
	if (ref->id[0] != NOT_DEFINED) return;
	ref->id[0] = id; iter_mm1cl it1 = ref->cr_list.begin();
	while (it1 != ref->cr_list.end()) GatherAtoms((* it1++).atmr, id);
}

/*##############################################*/
/*##############################################*/

// what comes to PDB chain id's here, we follow the "pdb-select principle" by converting
// the empty id ' ' with '_' to improve readability...

mm1_readpdb_mdata * mm1_mdl::readpdb_ReadMData(const char * filename)
{
	cout << "reading PDB metadata from file " << filename << endl;
	
	mm1_readpdb_mdata * mdata = new mm1_readpdb_mdata;
	
	ifstream file(filename, ios::in);
	if (file.fail())
	{
		if (ostr != NULL) (* ostr) << "file \"" << filename << "\" not found." << endl;
		
		file.close();
		return mdata;
	}
	
	while (!file.eof())
	{
		char record_id[8];
		for (i32s n1 = 0;n1 < 6;n1++)
		{
			char tchar = (char) file.get();
			if (tchar != ' ') record_id[n1] = tchar;
			else record_id[n1] = 0;
		}
		
		record_id[6] = 0;	// terminate the record string...
		
		if (!strcmp("SEQRES", record_id))
		{
			i32s blocknum; file >> blocknum;
			file.get(); char chn_id = (char) file.get();
			i32s chn_length; file >> chn_length;
			
			if (chn_id == ' ') chn_id = '_';	// fix the empty chain id...
			
			i32s chn_num;
			if (blocknum == 1)	// this is a new record...
			{
				cout << "found a new chain " << mdata->chn_vector.size();
				cout << " '" << chn_id << "' with " << chn_length << " residues." << endl;
				
				mm1_readpdb_mdata_chain * new_data = new mm1_readpdb_mdata_chain;
				new_data->chn_id = chn_id; new_data->seqres = new char[chn_length + 1];
				
				for (i32s n1 = 0;n1 < chn_length;n1++) new_data->seqres[n1] = '?';
				new_data->seqres[chn_length] = 0;	// terminate the string!!!
				
				chn_num = mdata->chn_vector.size();
				mdata->chn_vector.push_back(new_data);
			}
			else		// this is an old record, find it...
			{
				chn_num = 0;
				while (chn_num < ((i32s) mdata->chn_vector.size()))
				{
					if (mdata->chn_vector[chn_num]->chn_id == chn_id) break;
					else chn_num++;
				}
				
				if (chn_num == ((i32s) mdata->chn_vector.size()))
				{
					cout << "readpdb_ReadMData : unknown chain found!!!" << endl;
					exit(EXIT_FAILURE);
				}
			}
			
			chn_length = strlen(mdata->chn_vector[chn_num]->seqres);
			i32s counter = (blocknum - 1) * 13;
			
			for (i32s n1 = 0;n1 < 13;n1++)
			{
				char residue[16];
				file >> residue;
				
				char symbol = '?';
				if (!strcmp("ALA", residue)) symbol = 'A';
				if (!strcmp("ARG", residue)) symbol = 'R';
				if (!strcmp("ASN", residue)) symbol = 'N';
				if (!strcmp("ASP", residue)) symbol = 'D';
				if (!strcmp("CYS", residue)) symbol = 'C';
				if (!strcmp("GLN", residue)) symbol = 'Q';
				if (!strcmp("GLU", residue)) symbol = 'E';
				if (!strcmp("GLY", residue)) symbol = 'G';
				if (!strcmp("HIS", residue)) symbol = 'H';
				if (!strcmp("ILE", residue)) symbol = 'I';
				if (!strcmp("LEU", residue)) symbol = 'L';
				if (!strcmp("LYS", residue)) symbol = 'K';
				if (!strcmp("MET", residue)) symbol = 'M';
				if (!strcmp("PHE", residue)) symbol = 'F';
				if (!strcmp("PRO", residue)) symbol = 'P';
				if (!strcmp("SER", residue)) symbol = 'S';
				if (!strcmp("THR", residue)) symbol = 'T';
				if (!strcmp("TRP", residue)) symbol = 'W';
				if (!strcmp("TYR", residue)) symbol = 'Y';
				if (!strcmp("VAL", residue)) symbol = 'V';
				
				mdata->chn_vector[chn_num]->seqres[counter++] = symbol;
				if (counter == chn_length) break;
			}
		}
		
		file.getline(buffer, sizeof(buffer));		// move to the next line...
	}
	
	file.close();
	
	// ready...
	
	if (mdata->chn_vector.empty()) cout << "WARNING : no chains found!!!" << endl;
	cout << "done." << endl;
	
	return mdata;
}

// chn_num must be either NOT_DEFINED (which is -1 meaning all chains) or the chain index in mdata...
// chn_num must be either NOT_DEFINED (which is -1 meaning all chains) or the chain index in mdata...
// chn_num must be either NOT_DEFINED (which is -1 meaning all chains) or the chain index in mdata...

//#define READPDB_ENABLE_MULTIPLE_CRDSETS	// enable this to read all data from NMR entries...

void mm1_mdl::readpdb_ReadData(const char * filename, mm1_readpdb_mdata * mdata, i32s chn_num)
{
	cout << "reading PDB data from file " << filename << endl;
	
	ifstream file(filename, ios::in);
	if (file.fail())
	{
		if (ostr != NULL) (* ostr) << "file \"" << filename << "\" not found." << endl;
		
		file.close();
		return;
	}
	
	// read the whole file to temporary arrays...
	
	vector<mm1_readpdb_data_atom> atom_data;
	vector<mm1_readpdb_data_ssbond> ssbond_data;
	
	i32s model_counter = 0;
	i32s atom_counter = NOT_DEFINED;
	
	while (!file.eof())
	{
		char record_id[8];
		for (i32s n1 = 0;n1 < 6;n1++)
		{
			char tchar = (char) file.get();
			if (tchar != ' ') record_id[n1] = tchar;
			else record_id[n1] = 0;
		}
		
		record_id[6] = 0;	// terminate the record string...
		
		if (!strcmp("ATOM", record_id))
		{
			i32s serial_number; file >> serial_number;
			
			for (i32s n1 = 0;n1 < 18;n1++) buffer[n1] = (char) file.get();
			buffer[5] = ' '; buffer[15] = ' ';
			
			char chn_id; i32s res_num; char res_name[5]; char atm_name[5]; fGL crd[3];
			
			istrstream str(buffer, sizeof(buffer));
			str >> atm_name >> res_name; str.get(); chn_id = (char) str.get(); str >> res_num;
			file >> crd[0]; crd[0] /= 10.0; file >> crd[1]; crd[1] /= 10.0; file >> crd[2]; crd[2] /= 10.0;
			
			if (chn_id == ' ') chn_id = '_';	// fix the empty chain id...
			
			bool test1 = (chn_num == NOT_DEFINED);
			bool test2 = test1 ? false : (chn_id == mdata->chn_vector[chn_num]->chn_id);
			
			if (test1 || test2)
			{
				if (model_counter == 0)
				{
					mm1_readpdb_data_atom new_data;
					new_data.ref = NULL;
					
					strcpy(new_data.atm_name, atm_name);
					strcpy(new_data.res_name, res_name);
					
					new_data.chn_id = chn_id;
					new_data.res_num = res_num;
					
					new_data.crd[model_counter][0] = crd[0];
					new_data.crd[model_counter][1] = crd[1];
					new_data.crd[model_counter][2] = crd[2];
					
					atom_data.push_back(new_data);
				}
				else
				{
					bool test1 = !strcmp(atom_data[atom_counter].res_name, res_name);
					bool test2 = !strcmp(atom_data[atom_counter].atm_name, atm_name);
					
					if (!test1) { cout << "test1 failed!!!" << endl; exit(EXIT_FAILURE); }
					if (!test2) { cout << "test2 failed!!!" << endl; exit(EXIT_FAILURE); }
					
					atom_data[atom_counter].crd[model_counter][0] = crd[0];
					atom_data[atom_counter].crd[model_counter][1] = crd[1];
					atom_data[atom_counter].crd[model_counter][2] = crd[2];
					
					atom_counter++;
				}
			}
		}
		
		if (!strcmp("SSBOND", record_id))
		{
			i32s serial_number; file >> serial_number;
			mm1_readpdb_data_ssbond new_data; new_data.ref = NULL;
			
			for (i32s n1 = 0;n1 < 2;n1++)
			{
				file >> buffer; file.get();
				char chn_id = (char) file.get();
				
				if (chn_id == ' ') chn_id = '_';	// fix the empty chain id...
				
				new_data.chn_id = chn_id;
				
				for (i32s n2 = 0;n2 < 6;n2++) buffer[n2] = (char) file.get();
				istrstream str(buffer, sizeof(buffer));
				str >> new_data.res_num;
				
				ssbond_data.push_back(new_data);
			}
		}
		
		// how to deal with multiple coordinate sets (often present in NMR entries)???
		// how to deal with multiple coordinate sets (often present in NMR entries)???
		// how to deal with multiple coordinate sets (often present in NMR entries)???
		
#ifdef READPDB_ENABLE_MULTIPLE_CRDSETS

		// in this caseat the moment, we will read multiple records up to READPDB_MAX_CRDSETS.
		// if READPDB_MAX_CRDSETS is exceeded, the program stops; must recompile then...
		
		if (!strcmp("MODEL", record_id))
		{
			i32s model_number; file >> model_number;
			
			if (model_number > 1)
			{
				model_counter++;
				
				if (model_counter == READPDB_MAX_CRDSETS)	// this should never happen...
				{
					cout << "READPDB_MAX_CRDSETS exceeded!!!" << endl;
					exit(EXIT_FAILURE);
				}
				
				atom_counter = 0;
			}
		}
		
#else	// READPDB_ENABLE_MULTIPLE_CRDSETS

		// in this case, we just look for "ENDMDL" record, which says this coordinate set is
		// now ready. so, we read only the first coordinate set. the "MODEL"/"ENDMDL" records
		// are only after "SSBOND", so it should not have big effects...
		
		if (!strcmp("ENDMDL", record_id))
		{
			cout << "ENDMDL record found, skipping the rest of this file..." << endl;
			break;
		}
		
#endif	// READPDB_ENABLE_MULTIPLE_CRDSETS

		file.getline(buffer, sizeof(buffer));		// move to the next line...
	}
	
	file.close();
	
	if (atom_data.empty())
	{
		cout << "no atoms found !!!" << endl;
		return;
	}
		
	// we have now read all the crd-sets; check if we need more space for them...
	
	i32s csets = cs_vector.size(); i32s new_csets = ((model_counter + 1) - csets);
	cout << "there were " << csets << " old crd-sets, creating " << new_csets << " new..." << endl;
	
	PushCRDSets(new_csets);
	
	for (i32u n1 = 0;n1 < cs_vector.size();n1++) SetCRDSetVisible(n1, true);
	
	// count the chains and relate them to mdata, create new records to mdata if necessary...
	
	vector<i32s> chn_index;
	char tmp_chn_id; i32u tmp_index;
	
	tmp_chn_id = atom_data.front().chn_id;
	
	tmp_index = 0;
	while (tmp_index < mdata->chn_vector.size())
	{
		if (tmp_chn_id == mdata->chn_vector[tmp_index]->chn_id) break;
		else tmp_index++;
	}
	
	if (tmp_index == mdata->chn_vector.size())
	{
		mm1_readpdb_mdata_chain * unknown = new mm1_readpdb_mdata_chain;
		unknown->chn_id = tmp_chn_id; unknown->seqres = NULL;
		mdata->chn_vector.push_back(unknown);
	}
	
	chn_index.push_back(tmp_index);
	
	for (i32u n1 = 1;n1 < atom_data.size();n1++)
	{
		tmp_chn_id = atom_data[n1].chn_id;
		if (tmp_chn_id == mdata->chn_vector[chn_index.back()]->chn_id) continue;
		else
		{
			tmp_index = 0;
			while (tmp_index < mdata->chn_vector.size())
			{
				if (tmp_chn_id == mdata->chn_vector[tmp_index]->chn_id) break;
				else tmp_index++;
			}
			
			if (tmp_index == mdata->chn_vector.size())
			{
				mm1_readpdb_mdata_chain * unknown = new mm1_readpdb_mdata_chain;
				unknown->chn_id = tmp_chn_id; unknown->seqres = NULL;
				mdata->chn_vector.push_back(unknown);
			}
			
			chn_index.push_back(tmp_index);
		}
	}
	
	// now create the chains, checking validity of the residues...
	
	for (i32u n1 = 0;n1 < chn_index.size();n1++)
	{
		char current_chn_id = mdata->chn_vector[chn_index[n1]]->chn_id;
		
		i32s current_chn_length;
		if (mdata->chn_vector[chn_index[n1]]->seqres != NULL)
		{
			current_chn_length = strlen(mdata->chn_vector[chn_index[n1]]->seqres);
		}
		else
		{
			current_chn_length = NOT_DEFINED;
		}
		
		i32s previous_residue = 0;
		
		i32s range1[2];
		
		range1[0] = 0;
		while (atom_data[range1[0]].chn_id != current_chn_id) range1[0]++;
		
		range1[1] = range1[0];
		while (range1[1] < (i32s) atom_data.size() && atom_data[range1[1]].chn_id == current_chn_id) range1[1]++;
		
		vector<i32s> res_data;
		
		i32s range2[2];
		range2[0] = range1[0];
		while (range2[0] < range1[1])
		{
			i32s residue = atom_data[range2[0]].res_num;
			const char * res_name = atom_data[range2[0]].res_name;
			
			range2[1] = range2[0];
			while (range2[1] < (i32s) atom_data.size() && atom_data[range2[1]].res_num == residue) range2[1]++;
			
			bool standard = false;
			if (!strcmp("ALA", res_name)) standard = true;
			if (!strcmp("ARG", res_name)) standard = true;
			if (!strcmp("ASN", res_name)) standard = true;
			if (!strcmp("ASP", res_name)) standard = true;
			if (!strcmp("CYS", res_name)) standard = true;
			if (!strcmp("GLN", res_name)) standard = true;
			if (!strcmp("GLU", res_name)) standard = true;
			if (!strcmp("GLY", res_name)) standard = true;
			if (!strcmp("HIS", res_name)) standard = true;
			if (!strcmp("ILE", res_name)) standard = true;
			if (!strcmp("LEU", res_name)) standard = true;
			if (!strcmp("LYS", res_name)) standard = true;
			if (!strcmp("MET", res_name)) standard = true;
			if (!strcmp("PHE", res_name)) standard = true;
			if (!strcmp("PRO", res_name)) standard = true;
			if (!strcmp("SER", res_name)) standard = true;
			if (!strcmp("THR", res_name)) standard = true;
			if (!strcmp("TRP", res_name)) standard = true;
			if (!strcmp("TYR", res_name)) standard = true;
			if (!strcmp("VAL", res_name)) standard = true;
			
			bool skip = false;
			if (standard)
			{
				if (readpdb_ReadData_sub1(atom_data, range2, "N", true) == NOT_DEFINED) skip = true;
				if (readpdb_ReadData_sub1(atom_data, range2, "CA", true) == NOT_DEFINED) skip = true;
				if (readpdb_ReadData_sub1(atom_data, range2, "C", true) == NOT_DEFINED) skip = true;
				if (readpdb_ReadData_sub1(atom_data, range2, "O", true) == NOT_DEFINED) skip = true;
			}
			else
			{
				cout << "could not recognize this residue: " << res_name << endl;
				skip = true;
			}
			
			if (skip)
			{
				cout << "skipping broken residue " << residue << " " << res_name << endl;
				atom_data.erase(atom_data.begin() + range2[0], atom_data.begin() + range2[1]);
				range1[1] += (range2[0] - range2[1]);
			}
			else
			{
				// ok, there seems to be no big problems with this residue.
				// first we make some bookkeeping with the original sequence...
				
				if ((previous_residue + 1) != residue)
				{
					for (i32s n2 = (previous_residue + 1);n2 < residue;n2++)
					{
						// the variables use "pascal/fortran-style" counting, beginning 1,
						// but we will store the values in "c-style" way, beginning 0!!!
						
						mdata->chn_vector[chn_index[n1]]->missing_residues.push_back(n2 - 1);
					}
				}
				
				previous_residue = residue;
				bool alpha_carbon_is_found = false;
				
				// ...and then we just create the atoms.
				
				res_data.push_back(range2[0]);
				for (i32s n2 = range2[0];n2 < range2[1];n2++)
				{
					strcpy(buffer, atom_data[n2].atm_name); buffer[1] = 0;		// here we suppose
					element el = element(buffer);					// single-char elements!!!
					
					if (el.GetAtomicNumber() == 1) continue;
					mm1_atom newatom(el, NULL, cs_vector.size());
					
					for (i32u n3 = 0;n3 < cs_vector.size();n3++)
					{
						newatom.crd_vector[n3][0] = atom_data[n2].crd[n3][0];
						newatom.crd_vector[n3][1] = atom_data[n2].crd[n3][1];
						newatom.crd_vector[n3][2] = atom_data[n2].crd[n3][2];
					}
					
					AddAtom(newatom); atom_data[n2].ref = (& atom_list.back());
					
					// if this atom was an alpha-carbon, then add it to the list... but alternative
					// locations might cause trouble here; make sure that only one c-alpha is added!!!
					
					if (!strcmp(atom_data[n2].atm_name, "CA") && !alpha_carbon_is_found)
					{
						mdata->chn_vector[chn_index[n1]]->alpha_carbons.push_back(atom_data[n2].ref);
						alpha_carbon_is_found = true;
					}
				}
				
				readpdb_ReadData_sub2(atom_data, range2, "N", "CA", 'S');
				readpdb_ReadData_sub2(atom_data, range2, "CA", "C", 'S');
				readpdb_ReadData_sub2(atom_data, range2, "C", "O", 'D');
				
				if (!strcmp("ALA", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
				}
				
				if (!strcmp("ARG", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CD", "NE", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "NE", "CZ", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CZ", "NH1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CZ", "NH2", 'C');
				}
				
				if (!strcmp("ASN", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "OD1", 'D');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "ND2", 'S');
				}
				
				if (!strcmp("ASP", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "OD1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "OD2", 'C');
				}
				
				if (!strcmp("CYS", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "SG", 'S');
				}
				
				if (!strcmp("GLN", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CD", "OE1", 'D');
					readpdb_ReadData_sub2(atom_data, range2, "CD", "NE2", 'S');
				}
				
				if (!strcmp("GLU", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CD", "OE1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CD", "OE2", 'C');
				}
				
				// GLY -> we don't have to do anything in this case...
				// GLY -> we don't have to do anything in this case...
				// GLY -> we don't have to do anything in this case...
				
				if (!strcmp("HIS", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "ND1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "ND1", "CE1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CD2", "NE2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CE1", "NE2", 'C');
				}
				
				if (!strcmp("ILE", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG1", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG2", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG1", "CD1", 'S');
				}
				
				if (!strcmp("LEU", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD1", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD2", 'S');
				}
				
				if (!strcmp("LYS", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CD", "CE", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CE", "NZ", 'S');
				}
				
				if (!strcmp("MET", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "SD", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "SD", "CE", 'S');
				}
				
				if (!strcmp("PHE", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CD1", "CE1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CD2", "CE2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CE1", "CZ", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CE2", "CZ", 'C');
				}
				
				if (!strcmp("PRO", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CD", "N", 'S');
				}
				
				if (!strcmp("SER", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "OG", 'S');
				}
				
				if (!strcmp("THR", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "OG1", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG2", 'S');
				}
				
				if (!strcmp("TRP", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CD1", "NE1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CD2", "CE2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CD2", "CE3", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "NE1", "CE2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CE2", "CZ2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CE3", "CZ3", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CZ2", "CH2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CZ3", "CH2", 'C');
				}
				
				if (!strcmp("TYR", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CG", "CD2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CD1", "CE1", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CD2", "CE2", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CE1", "CZ", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CE2", "CZ", 'C');
					readpdb_ReadData_sub2(atom_data, range2, "CZ", "OH", 'S');
				}
				
				if (!strcmp("VAL", res_name))
				{
					readpdb_ReadData_sub2(atom_data, range2, "CA", "CB", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG1", 'S');
					readpdb_ReadData_sub2(atom_data, range2, "CB", "CG2", 'S');
				}
				
				range2[0] = range2[1];
			}
		}
		
		// the end of sequence bookkeeping : check if there were missing n-terminal residues.
		
		if (previous_residue < current_chn_length)
		{
			for (i32s n2 = previous_residue;n2 < current_chn_length;n2++)
			{
				mdata->chn_vector[chn_index[n1]]->missing_residues.push_back(n2);
			}
		}
		
		if (!mdata->chn_vector[chn_index[n1]]->missing_residues.empty())
		{
			char out1 = mdata->chn_vector[chn_index[n1]]->chn_id;
			i32u out2 = mdata->chn_vector[chn_index[n1]]->missing_residues.size();
			
			cout << "at chain '" << out1 << "' there were " << out2 << " missing residues:" << endl;
			
			for (i32u n2 = 0;n2 < mdata->chn_vector[chn_index[n1]]->missing_residues.size();n2++)
			{
				cout << mdata->chn_vector[chn_index[n1]]->missing_residues[n2] << " ";
			}
			
			cout << endl;
		}
		
		// add connecting bonds between residues...
		
		res_data.push_back(range1[1]);
		for (i32s n2 = 0;n2 < ((i32s) res_data.size()) - 2;n2++)
		{
			i32s r1[2] = { res_data[n2],  res_data[n2 + 1] };
			i32s r2[2] = { res_data[n2 + 1],  res_data[n2 + 2] };
			
			i32s ind1 = readpdb_ReadData_sub1(atom_data, r1, "C", false);
			i32s ind2 = readpdb_ReadData_sub1(atom_data, r2, "N", false);
			
			if (ind1 == NOT_DEFINED || ind2 == NOT_DEFINED)
			{
				(* ostr) << "could not connect ..." << endl;
				exit(EXIT_FAILURE);
			}
			else
			{
				bondtype bt = bondtype('S');
				mm1_bond newbond(atom_data[ind1].ref, atom_data[ind2].ref, bt);
				AddBond(newbond);
			}
		}
		
		// check the c-terminal residue...
		
		if (res_data.size() < 2) continue;
		i32s last[2] = { res_data[res_data.size() - 2], res_data[res_data.size() - 1] };
		
		i32s ind1 = readpdb_ReadData_sub1(atom_data, last, "C", false);
		i32s ind2 = readpdb_ReadData_sub1(atom_data, last, "O", false);
		
		if (ind1 == NOT_DEFINED || ind2 == NOT_DEFINED)
		{
			(* ostr) << "could not find c-term group..." << endl;
			exit(EXIT_FAILURE);
		}
		else
		{
			bondtype bt = bondtype('C');
			mm1_bond tmpbond(atom_data[ind1].ref, atom_data[ind2].ref, bt);
			iter_mm1bl it1 = find(bond_list.begin(), bond_list.end(), tmpbond);
			if (it1 != bond_list.end()) (* it1).bt = bt;
			
			ind2 = readpdb_ReadData_sub1(atom_data, last, "OXT", false);
			if (ind2 != NOT_DEFINED)
			{
				mm1_bond newbond(atom_data[ind1].ref, atom_data[ind2].ref, bt);
				AddBond(newbond);
			}
			else
			{
				cout << "missing terminal oxygen..." << endl;
				
				i32s ind[3];
				ind[0] = readpdb_ReadData_sub1(atom_data, last, "CA", false);
				ind[1] = readpdb_ReadData_sub1(atom_data, last, "C", false);
				ind[2] = readpdb_ReadData_sub1(atom_data, last, "O", false);
				
				fGL * ref1; fGL * ref2;
				
				ref1 = atom_data[ind[1]].ref->crd_vector[0].data;
				ref2 = atom_data[ind[0]].ref->crd_vector[0].data;
				v3d<fGL> v1 = v3d<fGL>(ref1, ref2);
				
				ref1 = atom_data[ind[1]].ref->crd_vector[0].data;
				ref2 = atom_data[ind[2]].ref->crd_vector[0].data;
				v3d<fGL> v2 = v3d<fGL>(ref1, ref2);
				
				fGL tmp2 = v1.len(); fGL tmp3 = v1.spr(v2) / (tmp2 * tmp2);
				v1 = v1 * tmp3; v2 = v2 - v1;
				
				fGL tmp4[3];
				for (i32s n1 = 0;n1 < 3;n1++)
				{
					tmp4[n1] = atom_data[ind[1]].ref->crd_vector[0][n1];
					tmp4[n1] += v1.data[n1] - v2.data[n1];
				}
				
				element el = element(8);
				mm1_atom newatom(el, tmp4, cs_vector.size());
				AddAtom(newatom);
				
				mm1_bond newbond(atom_data[ind1].ref, (& atom_list.back()), bt);
				AddBond(newbond);
			}
		}
	}
	
	// add disulphide bonds...
	
	for (i32u n1 = 0;n1 < ssbond_data.size();n1++)
	{
		i32u counter = 0;
		while (counter < atom_data.size())
		{
			bool test1 = (ssbond_data[n1].chn_id == atom_data[counter].chn_id);
			bool test2 = (ssbond_data[n1].res_num == atom_data[counter].res_num);
			bool test3 = !strcmp(atom_data[counter].atm_name, "SG");
			if (test1 && test2 && test3) break; else counter++;
		}
		
		if (counter == atom_data.size()) continue;
		ssbond_data[n1].ref = atom_data[counter].ref;
	}
	
	for (i32u n1 = 0;n1 < ssbond_data.size();n1 += 2)
	{
		if (ssbond_data[n1 + 0].ref == NULL || ssbond_data[n1 + 1].ref == NULL)
		{
			cout << "could not create bridge ";
			cout << ssbond_data[n1 + 0].chn_id << " " << ssbond_data[n1 + 0].res_num << " -> ";
			cout << ssbond_data[n1 + 1].chn_id << " " << ssbond_data[n1 + 1].res_num << endl;
		}
		else
		{
			bondtype bt = bondtype('S');
			mm1_bond newbond(ssbond_data[n1 + 0].ref, ssbond_data[n1 + 1].ref, bt);
			AddBond(newbond);
		}
	}
	
	// ready...
	
	cout << "done." << endl;
	
	CenterCRDSet(0);
}

i32s mm1_mdl::readpdb_ReadData_sub1(vector<mm1_readpdb_data_atom> & adata, i32s * range, const char * atom_name, bool flag)
{
	for (i32s n1 = range[0];n1 < range[1];n1++)
	{
		if (!strcmp(adata[n1].atm_name, atom_name)) return n1;
	}
	
	if (flag && ostr != NULL) (* ostr) << "atom " << atom_name << " is missing..." << endl;
	return NOT_DEFINED;
}

void mm1_mdl::readpdb_ReadData_sub2(vector<mm1_readpdb_data_atom> & adata, i32s * range, const char * at1, const char * at2, char btype)
{
	i32s ind1 = readpdb_ReadData_sub1(adata, range, at1, false);
	i32s ind2 = readpdb_ReadData_sub1(adata, range, at2, false);
	
	if (ind1 == NOT_DEFINED || ind2 == NOT_DEFINED) return;
	
	bondtype bt = bondtype(btype);
	mm1_bond newbond(adata[ind1].ref, adata[ind2].ref, bt);
	AddBond(newbond);
}

/*##############################################*/
/*##############################################*/

iter_mm1al mm1_mdl::aai_FindA(i32s atmi) //const
{
	if (atmi < 0) return GetAtomsEnd();				// out of valid range...
	if (atmi >= (i32s) aai_GetAtomCount()) return GetAtomsEnd();	// out of valid range...
	
	i32s counter = 0;
	iter_mm1al it1 = GetAtomsBegin();
	
	while (counter < atmi)
	{
		counter++;
		it1++;
	}
	
	return it1;
}

iter_mm1bl mm1_mdl::aai_FindB(i32s bndi) //const
{
	if (bndi < 0) return GetBondsEnd();				// out of valid range...
	if (bndi >= (i32s) aai_GetBondCount()) return GetBondsEnd();	// out of valid range...
	
	i32s counter = 0;
	iter_mm1bl it1 = GetBondsBegin();
	
	while (counter < bndi)
	{
		counter++;
		it1++;
	}
	
	return it1;
}

i32u mm1_mdl::aai_GetAtomCount(void) //const
{
	return atom_list.size();
}

i32u mm1_mdl::aai_GetBondCount(void) //const
{
	return bond_list.size();
}

void * mm1_mdl::aai_FindAtomByInd(i32s atmi) //const
{
	iter_mm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return NULL;
	else return (void *) & (* it1);
}

i32s mm1_mdl::aai_FindAtomByPtr(void * atmr) //const
{
	i32s counter = 0;
	iter_mm1al it1 = GetAtomsBegin();
	
	while (it1 != GetAtomsEnd())
	{
		void * tmpr = & (* it1);
		if (atmr == tmpr) return counter;
		
		counter++;
		it1++;
	}
	
	return -1;	// not found...
}

i32s mm1_mdl::aai_FindBond(i32s atmi1, i32s atmi2) //const
{
	void * atmr1 = aai_FindAtomByInd(atmi1);
	if (atmr1 == NULL) return -1;
	
	void * atmr2 = aai_FindAtomByInd(atmi2);
	if (atmr2 == NULL) return -1;
	
	i32s counter = 0;
	iter_mm1bl it1 = GetBondsBegin();
	
	while (it1 != GetBondsEnd())
	{
		void * tmpr1 = (* it1).atmr[0];
		void * tmpr2 = (* it1).atmr[1];
		
		if (atmr1 == tmpr1 && atmr2 == tmpr2) return counter;
		if (atmr1 == tmpr2 && atmr2 == tmpr1) return counter;
		
		counter++;
		it1++;
	}
	
	return -1;	// not found...
}

void * mm1_mdl::aai_AddAtom(element el, fGL * xyz)
{
	mm1_atom newatom(el, xyz, cs_vector.size());
	AddAtom(newatom);
	
	return (void *) & atom_list.back();
}

bool mm1_mdl::aai_RemoveAtom(i32s atmi)
{
	iter_mm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return false;
	
	RemoveAtom(it1);
	
	return true;
}

bool mm1_mdl::aai_AddBond(i32s atmi1, i32s atmi2, bondtype bt)
{
	void * atmr1 = aai_FindAtomByInd(atmi1);
	if (atmr1 == NULL) return false;
	
	void * atmr2 = aai_FindAtomByInd(atmi2);
	if (atmr2 == NULL) return false;
	
	mm1_bond newbond((mm1_atom *) atmr1, (mm1_atom *) atmr2, bt);
	AddBond(newbond);
	
	return true;
}

bool mm1_mdl::aai_RemoveBond(i32s bndi)
{
	iter_mm1bl it1 = aai_FindB(bndi);
	if (it1 == GetBondsEnd()) return false;
	
	RemoveBond(it1);
	
	return true;
}

bool mm1_mdl::aai_AtomGetElement(i32s atmi, element & el) //const
{
	iter_mm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return false;
	
	el = (* it1).el;
	
	return true;
}

bool mm1_mdl::aai_AtomSetElement(i32s atmi, element el)
{
	iter_mm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return false;
	
	(* it1).el = el;
	
	return true;
}

bool mm1_mdl::aai_AtomGetXYZ(i32s atmi, fGL * xyz) //const
{
	iter_mm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return false;
	
	xyz[0] = (* it1).crd_vector[0].data[0];
	xyz[1] = (* it1).crd_vector[0].data[1];
	xyz[2] = (* it1).crd_vector[0].data[2];
	
	return true;
}

bool mm1_mdl::aai_AtomSetXYZ(i32s atmi, const fGL * xyz)
{
	iter_mm1al it1 = aai_FindA(atmi);
	if (it1 == GetAtomsEnd()) return false;
	
//	for (i32u n1 = 0;n1 < cs_vector.size();n1++)
	i32s n1 = 0;	// how to handle this properly?!?!?!
	{
		(* it1).crd_vector[n1].data[0] = xyz[0];
		(* it1).crd_vector[n1].data[1] = xyz[1];
		(* it1).crd_vector[n1].data[2] = xyz[2];
	}
	
	return true;
}

bool mm1_mdl::aai_BondGetBondType(i32s bndi, bondtype & bt) //const
{
	iter_mm1bl it1 = aai_FindB(bndi);
	if (it1 == GetBondsEnd()) return false;
	
	bt = (* it1).bt;
	
	return true;
}

bool mm1_mdl::aai_BondSetBondType(i32s bndi, bondtype bt)
{
	iter_mm1bl it1 = aai_FindB(bndi);
	if (it1 == GetBondsEnd()) return false;
	
	(* it1).bt = bt;
	
	return true;
}

bool mm1_mdl::aai_BondGetAtomIndices(i32s bndi, i32s * iii) //const
{
	iter_mm1bl it1 = aai_FindB(bndi);
	if (it1 == GetBondsEnd()) return false;
	
	iii[0] = aai_FindAtomByPtr((void *) (* it1).atmr[0]);
	iii[1] = aai_FindAtomByPtr((void *) (* it1).atmr[1]);
	
	return true;
}

/*##############################################*/
/*##############################################*/

void mm1_mdl::DoEnergy(void)
{
	mm1_eng * eng = CreateDefaultEngine();
	
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "Calculating Energy ";
	str1 << "(molecular mechanics";
	str1 << ", engine = " << engtab1[default_eng];
	str1 << ")." << endl << ends;
	PrintToLog(mbuff1);
	
	CopyCRD(this, eng, 0);
	eng->Compute(0);
	
	char buffer[128];
	ostrstream str(buffer, sizeof(buffer)); str.setf(ios::fixed); str.precision(8);
	str << "Energy = " << eng->energy << " kJ/mol" << endl << ends;
	
	PrintToLog(buffer);
	delete eng;
}

void mm1_mdl::DoGeomOpt(mm1_geomopt_param & param)
{
	GeomOptGetParam(param);
	if (!param.confirm) return;
	
	mm1_eng * eng = CreateDefaultEngine();
	
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "Starting Geometry Optimization ";
	str1 << "(molecular mechanics";
	str1 << ", engine = " << engtab1[default_eng];
	str1 << ")." << endl << ends;
	PrintToLog(mbuff1);
	
	CopyCRD(this, eng, 0);
	
	mm1_geomopt * opt = new mm1_geomopt(eng, 100, 0.025);		// optimal settings?!?!?
	
	char buffer[1024];
	f64  last_energy = 0.0;		// this is for output and delta_e test...
	
	PrintToLog("Cycle    Energy    Gradient       Step        Delta E\n");
	
	i32s n1 = 0;	// n1 counts the number of steps...
	while (true)
	{
		opt->TakeCGStep(conjugate_gradient::Newton2An);
		
// problem: the gradient information is in fact not precise in this stage. the current gradient
// is the one that was last calculated in the search, and it is not necessarily the best one.
// to update the gradient, we need to Compute(1) here...
	eng->Compute(1);	// this is not vital, but will update the gradient vector length...
		
		if (n1 != 0)
		  {
		    sprintf(buffer, "%4d %12.5f %10.4e %10.4e %10.4e \n", n1,
			    opt->optval, eng->GetGradientVectorLength(),
			    opt->optstp, last_energy - opt->optval);
		  }
		else
		  {
		    sprintf(buffer, "%4d %12.5f %10.4e %10.4e ********** \n", n1,
			    opt->optval, eng->GetGradientVectorLength(),
			    opt->optstp);
		  }
		PrintToLog(buffer);
		
		bool terminate = false;
		
		if (param.enable_nsteps)	// the nsteps test...
		{
			if (n1 >= param.treshold_nsteps)
			{
				terminate = true;
				PrintToLog("the nsteps termination test was passed.\n");
			}
		}
		
		if (param.enable_grad)		// the grad test...
		{
			if (eng->GetGradientVectorLength() < param.treshold_grad)
			{
				terminate = true;
				PrintToLog("the grad termination test was passed.\n");
			}
		}
		
		if (param.enable_delta_e)	// the delta_e test...
		{
			bool flag = false; const f64 treshold_step = 1.0e-12;		// can we keep this as a constant???
			if (n1 != 0 && (last_energy - opt->optval) != 0.0 && fabs(last_energy - opt->optval) < param.treshold_delta_e) flag = true;
			if ((opt->optstp != 0.0) && (opt->optstp < treshold_step)) flag = true;
			
			if (flag)
			{
				terminate = true;
				PrintToLog("the delta_e termination test was passed.\n");
			}
		}
		
		last_energy = opt->optval;
		
		if (!(n1 % 10) || terminate)
		{
			CopyCRD(eng, this, 0); CenterCRDSet(0);
			UpdateAllGraphicsViews(true);
		}
		
		if (terminate) break;		// exit the loop here!!!
		
		n1++;	// update the number of steps...
	}
	
	delete opt;
	delete eng;
}

void mm1_mdl::GeomOptGetParam(mm1_geomopt_param & param)
{
	param.confirm = true;
}

#define CFRQ 1
#define CSIZE 100

#define START 10000

void mm1_mdl::DoMolDyn(mm1_moldyn_param & param)
{
	MolDynGetParam(param);
	if (!param.confirm) return;
	
	mm1_eng * eng = CreateDefaultEngine();
	
	mm1_eng_exp1b3 * yyy = NULL;		// temporary...
	mm1_eng_pbc * zzz = NULL;		// temporary...
	
	mm1_eng_exp1_mim * xxx = dynamic_cast<mm1_eng_exp1_mim *>(eng);
	if (xxx != NULL) zzz = yyy = xxx;
	
	char mbuff1[256];
	ostrstream str1(mbuff1, sizeof(mbuff1));
	str1 << "Starting Molecular Dynamics ";
	str1 << "(molecular mechanics";
	str1 << ", engine = " << engtab1[default_eng];
	str1 << ")." << endl << ends;
	PrintToLog(mbuff1);
	
	CopyCRD(this, eng, 0);
	
	mm1_moldyn * dyn = new mm1_moldyn(eng, param.temperature, param.timestep);
	
	ofstream ofile;
	ofile.open(param.filename, ios::out | ios::binary);
	
	const char file_id[10] = "traj_v10";
	const int number_of_atoms = atom_list.size();
	const int total_frames = param.nsteps / 100;
	
	ofile.write((char *) file_id, 8);					// file id, 8 chars.
	ofile.write((char *) & number_of_atoms, sizeof(number_of_atoms));	// number of atoms, int.
	ofile.write((char *) & total_frames, sizeof(total_frames));		// total number of frames, int.
	
	i32s ccnt = 0; f64 control[CSIZE];
	
	for (i32s n1 = 0;n1 < START + param.nsteps;n1++)
	{
		if (!(n1 % 1000) && zzz != NULL) zzz->CheckLocations();
		if (!(n1 % 100) && yyy != NULL) yyy->update = true;
		
		dyn->TakeMDStep();
		if (!(n1 % 50))
		{
			ostrstream str2(mbuff1, sizeof(mbuff1));
			str2 << "step " << n1 << " T = " << dyn->ConvEKinTemp(dyn->kin);
			str2 << " Epot = " << dyn->pot << " Etot = " << (dyn->kin + dyn->pot) << endl << ends;
			PrintToLog(mbuff1);
		}
		
		if (n1 < START)
		{
			if (!(n1 % CFRQ))
			{
				control[ccnt++] = dyn->kin;
			}
			
			if (ccnt == CSIZE)
			{
				f64 sum = 0.0;
				for (i32s n9 = 0;n9 < ccnt;n9++)
				{
					sum += control[n9];
				}
				
				sum /= (f64) ccnt; ccnt = 0;
				f64 rtemp = (f64) (n1 + CFRQ) / (f64) START;
				
				ostrstream str2(mbuff1, sizeof(mbuff1));
				str2 << "average Ekin = " << sum << " kJ/mol = " << dyn->ConvEKinTemp(sum) << " K." << endl << ends;
				PrintToLog(mbuff1);
				
				dyn->SetEKin(dyn->kin + (dyn->ConvTempEKin(dyn->temp * rtemp) - sum));
			}
		}
		else if (!(n1 % 100))
		{
			CopyCRD(eng, this, 0);
			
			const float ekin = dyn->kin;
			const float epot = dyn->pot;
			
			ofile.write((char *) & ekin, sizeof(ekin));	// kinetic energy, float.
			ofile.write((char *) & epot, sizeof(epot));	// potential energy, float.
			
			for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
			{
				for (i32s t4 = 0;t4 < 3;t4++)		// all coordinates, float.
				{
					float t1a = (* it1).crd_vector[0][t4];
					ofile.write((char *) & t1a, sizeof(t1a));
				}
			}
		}
		
		if (!(n1 % 100))
		{
			CopyCRD(eng, this, 0);
			UpdateAllGraphicsViews(true);
		}
	}
	
	delete dyn;
	delete eng;
	
	UpdateAccumValues();	// is this needed !??!?!!?
}

void mm1_mdl::MolDynGetParam(mm1_moldyn_param & param)
{
	param.confirm = true;
}

void mm1_mdl::DoRandomSearch(i32s cycles, i32s optsteps)
{
	if (cs_vector.size() < 2)
	{
		PushCRDSets(1);
		SetCRDSetVisible(1, false);
	}
	
	mm1_random_search rs(this, 0, 0, 1, cycles, optsteps);
	
	while (true)
	{
		int zzz = rs.TakeStep();
	//	cout << zzz << endl;
		
		UpdateAllGraphicsViews(true);
		if (zzz < 0) break;
	}
	
	CopyCRDSet(1, 0);		// get the best structure!!!
	UpdateAllGraphicsViews(true);
	
	char mbuff1[256]; strstream str1(mbuff1, sizeof(mbuff1));
	str1 << "lowest energy found = " << rs.GetMinEnergy() << " kJ/mol" << endl << ends;
	PrintToLog(mbuff1);
	
	PrintToLog("RANDOM SEARCH IS READY.\n");
}

void mm1_mdl::DoSystematicSearch(i32s divisions, i32s optsteps)
{
	if (cs_vector.size() < 2)
	{
		PushCRDSets(1);
		SetCRDSetVisible(1, false);
	}
	
	mm1_systematic_search ss(this, 0, 0, 1, divisions, optsteps);
	
	while (true)
	{
		int zzz = ss.TakeStep();
	//	cout << zzz << endl;
		
		UpdateAllGraphicsViews(true);
		if (zzz < 0) break;
	}
	
	CopyCRDSet(1, 0);		// get the best structure!!!
	UpdateAllGraphicsViews(true);
	
	char mbuff1[256]; strstream str1(mbuff1, sizeof(mbuff1));
	str1 << "lowest energy found = " << ss.GetMinEnergy() << " kJ/mol" << endl << ends;
	PrintToLog(mbuff1);
	
	PrintToLog("SYSTEMATIC SEARCH IS READY.\n");
}

void mm1_mdl::DoMonteCarloSearch(i32s n_init_steps, i32s n_simul_steps, i32s optsteps)
{
	
	if (cs_vector.size() < 2)
	{
		PushCRDSets(1);
		//PushCRDSets(1 + n_simul_steps);	//DEBUG - this works...
		SetCRDSetVisible(1, false);
	}
	
	mm1_monte_carlo_search mcs(this, 0, 0, 1, n_init_steps, n_simul_steps, optsteps);
	
	while (true)
	{
		int zzz = mcs.TakeStep();
	//	cout << zzz << endl;
		
		UpdateAllGraphicsViews(true);
		if (zzz < 0) break;
	}
	
	CopyCRDSet(1, 0);		// get the best structure!!!
	UpdateAllGraphicsViews(true);
	
	char mbuff1[256]; strstream str1(mbuff1, sizeof(mbuff1));
	str1 << "lowest energy found = " << mcs.GetMinEnergy() << " kJ/mol" << endl << ends;
	PrintToLog(mbuff1);
	
	PrintToLog("MONTE CARLO SEARCH IS READY.\n");
	
//DEBUG
//for (i32s n9 = 0;n9 < n_simul_steps;n9++)
//{
//	mm1_superimpose si(this, 1, n9 + 2);
//	cout << "superimposing structure " << n9 << ":   ";
//	for (i32s n2 = 0;n2 < 100;n2++) si.TakeCGStep(conjugate_gradient::Newton2An);
//	cout << "rms = " << si.GetRMS() << endl;
//}
}

void mm1_mdl::DoFormula()
{
  double molweight = 0.0;
  int i;
  char buffer[1024];
  ostrstream str(buffer, sizeof(buffer));

  int count[ELEMENT_SYMBOLS];
  // These are the atomic numbers of the elements in alphabetical order.
  const int alphabetical[ELEMENT_SYMBOLS] = {
   89, 47, 13, 95, 18, 33, 85, 79, 5, 56, 4, 107, 83, 97, 35, 6, 20, 48,
   58, 98, 17, 96, 27, 24, 55, 29, 105, 66, 68, 99, 63, 9, 26, 100, 87, 31,
   64, 32, 1, 2, 72, 80, 67, 108, 53, 49, 77, 19, 36, 57, 3, 103, 71, 101,
   12, 25, 42, 109, 7, 11, 41, 60, 10, 28, 102, 93, 8, 76, 15, 91, 82, 46, 
   61, 84, 59, 78, 94, 88, 37, 75, 104, 45, 86, 44, 16, 51, 21, 34, 106, 14, 
   62, 50, 38, 73, 65, 43, 52, 90, 22, 81, 69, 92, 110, 23, 74, 54, 39, 70, 
   30, 40 };
  int index;

  for (i = 0; i < ELEMENT_SYMBOLS; i++)
    {
      count[i] = 0;
    }

  iter_mm1al it2 = atom_list.begin();
  while (it2 != atom_list.end())
    {
      iter_mm1al it3 = it2++;
      count[(* it3).el.GetAtomicNumber() - 1]++;
      molweight += (* it3).el.GetAtomicMass();
    }
  
  for (i = 0; i < ELEMENT_SYMBOLS; i++)
    {
      index = alphabetical[i] - 1;
      if (count[index] > 1)
	{
	  str << (element(index + 1)).GetSymbol() << count[index] << " ";
	}
      else if (count[index] == 1)
	{
	  str << (element(index + 1)).GetSymbol();
	}
    }

  str << "MW: " << molweight << ends;
  err->Message(buffer);
}

/*##############################################*/
/*##############################################*/

void mm1_mdl::OpenTrajectory(const char * fn)
{
	if (!trajfile)
	{
		trajfile = new ifstream(fn, ios::in | ios::binary);
		
		trajfile->seekg(8, ios::beg);	// skip the file id...
		
		int natoms;
		trajfile->read((char *) & natoms, sizeof(natoms));
		
		if (natoms != (int) atom_list.size())
		{
			PrintToLog("incompatible file : different number of atoms!\n");
			CloseTrajectory(); return;
		}
		
		trajfile->read((char *) & total_traj_frames, sizeof(total_traj_frames));
		
		char mbuff1[256]; strstream str1(mbuff1, sizeof(mbuff1));
		str1 << "the trajectory file contains " << total_traj_frames << " frames." << endl << ends;
		PrintToLog(mbuff1);
		
		current_traj_frame = 0;
	}
	else PrintToLog("trajectory already exists!\n");
}

void mm1_mdl::CloseTrajectory(void)
{
	if (trajfile != NULL)
	{
		trajfile->close();
		delete trajfile;
		
		trajfile = NULL;
	}
}

void mm1_mdl::ReadFrame(void)
{
	i32s atoms = atom_list.size();
	
	i32s place = 8 + 2 * sizeof(int);					// skip the header...
	place += (2 + 3 * atoms) * sizeof(float) * current_traj_frame;		// get the correct frame...
	place += 2 * sizeof(float);						// skip epot and ekin...
	
	trajfile->seekg(place, ios::beg);
	
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		for (i32s t4 = 0;t4 < 3;t4++)
		{
			float t1a; trajfile->read((char *) & t1a, sizeof(t1a));
			(* it1).crd_vector[0][t4] = t1a;
		}
	}
}

/*################################################################################################*/
/*################################################################################################*/

// f = sum[Q/r] = sum[Q * r^-1]
// df/dr = -1 * sum[Q * r^-2]

fGL mm1_GetESPValue(fGL * crd, mm1_mdl * ref, fGL * grad)
{
	fGL espv = 0.0;
	if (grad != NULL) grad[0] = grad[1] = grad[2] = 0.0;
	
	for (iter_mm1al it1 = ref->atom_list.begin();it1 != ref->atom_list.end();it1++)
	{
		fGL tmp1[3]; fGL r2 = 0.0;
		for (i32s n1 = 0;n1 < 3;n1++)
		{
			tmp1[n1] = crd[n1] - (* it1).crd_vector[0][n1];
			r2 += tmp1[n1] * tmp1[n1];
		}
		
		if (r2 == 0.0) return +1.0e+35;		// numeric_limits<fGL>::max()?!?!?!
		fGL r1 = sqrt(r2);
		
		// do we have a correct constant here??? I think so, if we define
		// electrostatic potential as potential energy of a unit positive charge.
		
		fGL tmp2 = 4.1868 * 33.20716 * (* it1).charge / r1;
		espv += tmp2;
		
		if (grad != NULL)	// sign ??? constant ???
		{
			for (i32s n1 = 0;n1 < 3;n1++)
			{
				fGL tmp3 = tmp1[n1] / r1;
				fGL tmp4 = tmp3 * tmp2 / r1;
				grad[n1] += tmp4;
			}
		}
	}
	
	return espv;
}

/*################################################################################################*/

// f = sum[(r/a)^-3] = sum[a^3 * r^-3]		// now seems to be r^-12
// df/dr = -3 * sum[a^3 * r^-4]

fGL mm1_GetVDWSValue(fGL * crd, mm1_mdl * ref, fGL * grad)
{
	fGL vdwsv = 0.0;
	if (grad != NULL) grad[0] = grad[1] = grad[2] = 0.0;
	
	for (iter_mm1al it1 = ref->atom_list.begin();it1 != ref->atom_list.end();it1++)
	{
		fGL tmp1[3]; fGL r2 = 0.0;
		for (i32s n1 = 0;n1 < 3;n1++)
		{
			tmp1[n1] = crd[n1] - (* it1).crd_vector[0][n1];
			r2 += tmp1[n1] * tmp1[n1];
		}
		
		if (r2 == 0.0) return +1.0e+35;		// numeric_limits<fGL>::max()?!?!?!
		fGL r1 = sqrt(r2);
		
		fGL tmp2 = r1 / (* it1).el.GetVDWRadius();
		fGL qqq = tmp2 * tmp2 * tmp2 * tmp2;
		
		fGL tmp3 = 1.0 / (qqq * qqq * qqq);
		vdwsv += tmp3;
		
		if (grad != NULL)	// sign ??? constant ???
		{
			for (i32s n1 = 0;n1 < 3;n1++)
			{
				fGL tmp4 = tmp1[n1] / r1;
				fGL tmp5 = tmp4 * tmp3 / tmp2;
				grad[n1] += tmp5;
			}
		}
	}
	
	return vdwsv;
}

/*################################################################################################*/
/*################################################################################################*/

// this is taken form mm2_superimpose...
// this is taken form mm2_superimpose...
// this is taken form mm2_superimpose...

mm1_superimpose::mm1_superimpose(mm1_mdl * p1, i32s p2, i32s p3) : superimpose(p2, p3)
{
	mdl = p1;
}

mm1_superimpose::~mm1_superimpose(void)
{
}

f64 mm1_superimpose::GetValue(void)
{
	value = 0.0; counter = 0;
	
	for (iter_mm1al it1 = mdl->GetAtomsBegin();it1 != mdl->GetAtomsEnd();it1++)
	{
		f64 d1[3];
		d1[0] = (* it1).crd_vector[index[0]].data[0];
		d1[1] = (* it1).crd_vector[index[0]].data[1];
		d1[2] = (* it1).crd_vector[index[0]].data[2];
		
		f64 d2[3];
		d2[0] = (* it1).crd_vector[index[1]].data[0];
		d2[1] = (* it1).crd_vector[index[1]].data[1];
		d2[2] = (* it1).crd_vector[index[1]].data[2];
		
		Compare(d1, d2, false);
	}
	
	return value;
}

f64 mm1_superimpose::GetGradient(void)
{
	value = 0.0; counter = 0;
	dloc[0] = dloc[1] = dloc[2] = 0.0;
	drot[0] = drot[1] = drot[2] = 0.0;
	
	for (iter_mm1al it1 = mdl->GetAtomsBegin();it1 != mdl->GetAtomsEnd();it1++)
	{
		f64 d1[3];
		d1[0] = (* it1).crd_vector[index[0]].data[0];
		d1[1] = (* it1).crd_vector[index[0]].data[1];
		d1[2] = (* it1).crd_vector[index[0]].data[2];
		
		f64 d2[3];
		d2[0] = (* it1).crd_vector[index[1]].data[0];
		d2[1] = (* it1).crd_vector[index[1]].data[1];
		d2[2] = (* it1).crd_vector[index[1]].data[2];
		
		Compare(d1, d2, true);
	}
	
	return value;
}

void mm1_superimpose::Transform(void)
{
	value = 0.0; counter = 0;
	
	for (iter_mm1al it1 = mdl->GetAtomsBegin();it1 != mdl->GetAtomsEnd();it1++)
	{
		f64 d1[3];
		d1[0] = (* it1).crd_vector[index[0]].data[0];
		d1[1] = (* it1).crd_vector[index[0]].data[1];
		d1[2] = (* it1).crd_vector[index[0]].data[2];
		
		f64 d2[3];
		d2[0] = (* it1).crd_vector[index[1]].data[0];
		d2[1] = (* it1).crd_vector[index[1]].data[1];
		d2[2] = (* it1).crd_vector[index[1]].data[2];
		
		f64 d3[3];
		Compare(d1, d2, false, d3);
		
		(* it1).crd_vector[index[1]].data[0] = d3[0];
		(* it1).crd_vector[index[1]].data[1] = d3[1];
		(* it1).crd_vector[index[1]].data[2] = d3[2];
	}
}

/*################################################################################################*/

// eof
