/* 

                          Firewall Builder

                 Copyright (C) 2002 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: OSConfigurator_linux24.cpp,v 1.20 2004/09/28 05:39:07 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"

#include "OSConfigurator_linux24.h"

#include "fwbuilder/Firewall.h"
#include "fwbuilder/FWOptions.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/IPv4.h"
#include "fwbuilder/Network.h"
#include "fwbuilder/Address.h"
#include "fwbuilder/Resources.h"

#ifndef _WIN32
#  include <unistd.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>

#include <algorithm>
#include <functional>
#include <iostream>

#include <assert.h>

using namespace libfwbuilder;
using namespace fwcompiler;
using namespace std;

string OSConfigurator_linux24::myPlatformName() { return "Linux24"; }

OSConfigurator_linux24::OSConfigurator_linux24(FWObjectDatabase *_db,
                                               const string &fwname) : 
    OSConfigurator(_db,fwname) , os_data()
{
}

void OSConfigurator_linux24::processFirewallOptions() 
{
    FWOptions* options=fw->getOptionsObject();
    string s;
    int    i;

/*
 * check if all interfaces configured for the firewall are present
 */
    if (options->getBool("verify_interfaces")) 
    {
        list<FWObject*> l2=fw->getByType(Interface::TYPENAME);
        if ( ! l2.empty() )
        {
            output << endl;
            output << "INTERFACES=\"";
            for (list<FWObject*>::iterator i=l2.begin(); i!=l2.end(); ++i)
            {
                Interface *iface=Interface::cast(*i);
                if (iface->getName().find("*")==string::npos)
                    output << iface->getName() << " ";
            }
            output << "\"" << endl;
            output << "for i in $INTERFACES ; do" << endl;
            output << "  $IP link show \"$i\" > /dev/null 2>&1 || {" << endl;
            output << "    log \"Interface $i does not exist\"" << endl;
            output << "    exit 1" << endl;
            output << "  }" << endl;
            output << "done" << endl;
            output << endl;
        }
    }

/*
 *    Turn off packet forwarding for now. We'll turn it on if needed in the end
 *
 * turned this off. This seems to be an overkill as we set default
 * policy to DROP in all chains before we purge the current firewall policy.

    output << "\n\n";
    output << "FWD=`cat /proc/sys/net/ipv4/ip_forward`\n";
    output << "echo \"0\" > /proc/sys/net/ipv4/ip_forward\n\n";
*/

    s=options->getStr("linux24_ip_dynaddr");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/ip_dynaddr\n\n";


    s=options->getStr("linux24_rp_filter");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/conf/all/rp_filter\n\n";

    s=options->getStr("linux24_accept_source_route");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/conf/all/accept_source_route\n\n";

    s=options->getStr("linux24_accept_redirects");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/conf/all/accept_redirects\n\n";

    s=options->getStr("linux24_log_martians");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/conf/all/log_martians\n\n";


    
    s=options->getStr("linux24_icmp_echo_ignore_broadcasts");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts\n\n";

    s=options->getStr("linux24_icmp_echo_ignore_all");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/icmp_echo_ignore_all\n\n";

    s=options->getStr("linux24_icmp_ignore_bogus_error_responses");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/icmp_ignore_bogus_error_responses\n\n";


    
    if ( (i=options->getInt("linux24_tcp_fin_timeout"))>0 )
	output << "echo " << i << " > /proc/sys/net/ipv4/tcp_fin_timeout\n\n";

    if ( (i=options->getInt("linux24_tcp_keepalive_interval"))>0 )
	output << "echo " << i << " > /proc/sys/net/ipv4/tcp_keepalive_intvl\n\n";

    s=options->getStr("linux24_tcp_window_scaling");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/tcp_window_scaling\n\n";

    s=options->getStr("linux24_tcp_sack");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/tcp_sack\n\n";

    s=options->getStr("linux24_tcp_fack");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/tcp_fack\n\n";

    s=options->getStr("linux24_tcp_syncookies");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/tcp_syncookies\n\n";

    s=options->getStr("linux24_tcp_ecn");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/tcp_ecn\n\n";

    s=options->getStr("linux24_tcp_timestamps");
    if (!s.empty())
	output << "echo " << s << " > /proc/sys/net/ipv4/tcp_timestamps\n\n";


    output << endl;
}

void OSConfigurator_linux24::addVirtualAddressForNAT(const Network *nw)
{
    FWOptions* options=fw->getOptionsObject();
    if ( options->getBool("manage_virtual_addr") ) 
    {
        if (virtual_addresses.empty() || 
            find(virtual_addresses.begin(),virtual_addresses.end(),nw->getAddress())==virtual_addresses.end()) 
        {
            Interface *iface=findInterfaceFor( nw, fw );
            if (iface!=NULL)
            {
                IPNetwork n( nw->getAddress() , nw->getNetmask() );

                IPAddress a;
                string str, subnet, first, last;

                a=nw->getAddress() +1;
                first  = a.toString();
                        
                a      = n.getBroadcastAddress() -1;
                last   = a.toString();

                output                                       << endl;
                output << "incaddr()"                        << endl;
                output << "{"                                << endl;
                output << "  n1=$4"                          << endl;
                output << "  n2=$3"                          << endl;
                output << "  n3=$2"                          << endl;
                output << "  n4=$1"                          << endl;
                output                                       << endl;
                output << "  vn1=`eval  \"echo \\\\$$n1\"`"  << endl;
                output                                       << endl;
                output << "  R=`expr $vn1 \\< 255`"          << endl;
                output << "  if test $R = \"1\"; then"       << endl;
                output << "    eval \"$n1=`expr $vn1 + 1`\"" << endl;
                output << "  else"                           << endl;
                output << "    eval \"$n1=0\""               << endl;
                output << "    incaddr XX $n4 $n3 $n2"       << endl;
                output << "  fi"                             << endl;
                output << "}"                                << endl;
                output                                       << endl;

                output << "a=\"" << first << "\""            << endl;
                output << "while test \"$a\" != \"" << last << "\"; do" << endl;

                output << "  add_addr ${a} " << nw->getNetmask().getLength() << " "
                       << iface->getName() << endl;


                output                                       << endl;
                output << "  OIFS=$IFS"                      << endl;
                output << "  IFS=\".\""                      << endl;
                output << "  set $a"                         << endl;
                output << "  a4=$1"                          << endl;
                output << "  a3=$2"                          << endl;
                output << "  a2=$3"                          << endl;
                output << "  a1=$4"                          << endl;
                output << "  IFS=$OIFS"                      << endl;
                output << "  incaddr a4 a3 a2 a1"            << endl;
                output << "  a=$a4\".\"$a3\".\"$a2\".\"$a1"  << endl;

                output << "done" << endl << endl;
                virtual_addresses.push_back(nw->getAddress());

            } else
                warning(_("Can not add virtual address ") + nw->getAddress().toString() +
                        _(" (object ") + nw->getName() + ")" );
        }
    }
}

void OSConfigurator_linux24::addVirtualAddressForNAT(const Address *addr)
{
    FWOptions* options=fw->getOptionsObject();
    if ( options->getBool("manage_virtual_addr") ) 
    {
        if (virtual_addresses.empty() || 
            find(virtual_addresses.begin(),virtual_addresses.end(),addr->getAddress())==virtual_addresses.end()) 
        {
            IPv4 *iaddr=IPv4::cast( findAddressFor(addr, fw ) );
            if (iaddr!=NULL)
            {
                Interface *iface=Interface::cast(iaddr->getParent());
                assert(iface!=NULL);

                output << "add_addr " << addr->getAddress().toString() << " "
                       << iaddr->getNetmask().getLength() <<  " "
                       << iface->getName() << endl;
        
                virtual_addresses.push_back(addr->getAddress());
            } else
                warning(_("Can not add virtual address ") + addr->getAddress().toString() +
                        _(" (object ") + addr->getName() + ")" );
        }
        return;
    }
}

void  OSConfigurator_linux24::configureInterfaces()
{
    FWOptions* options=fw->getOptionsObject();

/*
 *  Remove all host static routes and "pub" ARP entries if we are going to 
 *  create new ones
 */
    if ( options->getBool("manage_virtual_addr") ) 
    {
	list<FWObject*> l2=fw->getByType(Interface::TYPENAME);
	for (list<FWObject*>::iterator i=l2.begin(); i!=l2.end(); ++i) 
        {
	    Interface *interface_=Interface::cast(*i);
	    if ( interface_->isDyn() ) continue;
	    if ( interface_->isUnnumbered() ) continue;
	    if ( interface_->isLoopback() ) continue;

            output << "$IP -4 neigh flush dev " 
                   << interface_->getName() << " >/dev/null 2>&1" << endl;

            output << "$IP -4 addr flush dev " 
                   << interface_->getName() 
                   << " secondary label \"" << interface_->getName() << ":FWB*\""
                   <<  " >/dev/null 2>&1" << endl;
	}
	output << endl;
    }

    if ( options->getBool("configure_interfaces") ) 
    {

        output << endl;

        FWObjectTypedChildIterator i=fw->findByType(Interface::TYPENAME);
        for ( ; i!=i.end(); ++i ) 
        {
            Interface *iface=dynamic_cast<Interface*>(*i);
            assert(iface);

            if (iface->isDyn()) continue;
            if (iface->isUnnumbered()) continue;

            FWObjectTypedChildIterator j=iface->findByType(IPv4::TYPENAME);
            for ( ; j!=j.end(); ++j ) 
            {
                IPv4 *iaddr=IPv4::cast(*j);

                output << "add_addr " << iaddr->getAddress().toString() << " "
                       << iaddr->getNetmask().getLength() << " "
                       << iface->getName() << endl;
        
                virtual_addresses.push_back(iaddr->getAddress());
            }
            output << "$IP link set " << iface->getName() << " up" << endl;
        }
        output << endl;
    }

/*
 * get addresses of dynamic interfaces
 */
    FWObjectTypedChildIterator j=fw->findByType(Interface::TYPENAME);
    for ( ; j!=j.end(); ++j ) 
    {
        Interface *iface=Interface::cast(*j);
        
        if ( iface->isDyn() )
        {
/* if interface name ends with '*', this is a wildcard interface. Do
 * not get its address at this time. */
            if (iface->getName().find("*")==string::npos)
                output << "getaddr " << iface->getName() << "  i_" << iface->getName() << endl;
        }
    }
    output << endl;
}


int  OSConfigurator_linux24::prolog()
{
/*
 * print definitions for variables IPTABLES, IP, LOGGER. Some day we may
 * add a choice of distro in the GUI. Right now paths are either default
 * for a given distro, or custom strings entered by user in the GUI and stored
 * in firewall options.
 */
    printPathForAllTools(DISTRO);

/*
 * Process firewall options, build OS network configuration script
 */
    processFirewallOptions();

    output << endl;

    output << "$IPTABLES -P OUTPUT  DROP" << endl;
    output << "$IPTABLES -P INPUT   DROP" << endl;
    output << "$IPTABLES -P FORWARD DROP" << endl;
    output << endl;

/*
 *    flush all the rules which existed before
 */

    output << "\n\
\n\
cat /proc/net/ip_tables_names | while read table; do\n\
  test \"X$table\" = \"Xmangle\" && continue\n\
  $IPTABLES -t $table -L -n | while read c chain rest; do\n\
      if test \"X$c\" = \"XChain\" ; then\n\
        $IPTABLES -t $table -F $chain\n\
      fi\n\
  done\n\
  $IPTABLES -t $table -X\n\
done\n\
\n";

    output << endl;

    configureInterfaces();

    return 0;
}

void  OSConfigurator_linux24::printPathForAllTools(const string &os)
{
    FWOptions* options=fw->getOptionsObject();
    
    string s, path_lsmod, path_modprobe, path_iptables, path_ip, path_logger;

    s=options->getStr("linux24_path_lsmod");
    if (!s.empty()) path_lsmod=s;
    else            path_lsmod=os_data.getPathForTool(os,OSData::LSMOD);

    s=options->getStr("linux24_path_modprobe");
    if (!s.empty()) path_modprobe=s;
    else            path_modprobe=os_data.getPathForTool(os,OSData::MODPROBE);

    s=options->getStr("linux24_path_iptables");
    if (!s.empty()) path_iptables=s;
    else            path_iptables=os_data.getPathForTool(os,OSData::IPTABLES);

    s=options->getStr("linux24_path_ip");
    if (!s.empty()) path_ip=s;
    else            path_ip=os_data.getPathForTool(os,OSData::IP);

    s=options->getStr("linux24_path_logger");
    if (!s.empty()) path_logger=s;
    else            path_logger=os_data.getPathForTool(os,OSData::LOGGER);

    output                                                      << endl;
    output << "log() {"                                         << endl;
    output << "  echo \"$1\""                                   << endl;
    output << "  test -x \"$LOGGER\" && $LOGGER -p info \"$1\"" << endl;
    output << "}"                                               << endl;
    output                                                      << endl;

    output << "va_num=1"                                   << endl;

    output << "add_addr() {"                               << endl;
    output << "  addr=$1"                                  << endl;
    output << "  nm=$2"                                    << endl;
    output << "  dev=$3"                                   << endl;
    output << ""                                           << endl;
    output << "  type=\"\""                                << endl;
    output << "  aadd=\"\""                                << endl;
    output << ""                                           << endl;
    output << "  L=`$IP -4 link ls $dev | head -n1`"       << endl;
    output << "  if test -n \"$L\"; then"                  << endl;
    output << "    OIFS=$IFS"                              << endl;
    output << "    IFS=\" /:,<\""                          << endl;
    output << "    set $L"                                 << endl;
    output << "    type=$4"                                << endl;
    output << "    IFS=$OIFS"                              << endl;
    output << ""                                           << endl;
    output << "    L=`$IP -4 addr ls $dev to $addr | tail -n1`" << endl;
    output << "    if test -n \"$L\"; then"                << endl;
    output << "      OIFS=$IFS"                            << endl;
    output << "      IFS=\" /\""                           << endl;
    output << "      set $L"                               << endl;
    output << "      aadd=$2"                              << endl;
    output << "      IFS=$OIFS"                            << endl;
    output << "    fi"                                     << endl;
    output << "  fi"                                       << endl;
    output << "  if test -z \"$aadd\"; then"               << endl;
    output << "    if test \"$type\" = \"POINTOPOINT\"; then"                                        << endl;
    output << "      $IP -4 addr add $addr dev $dev scope global label $dev:FWB${va_num}"            << endl;
    output << "      va_num=`expr $va_num + 1`"            << endl;
    output << "    fi"                                     << endl;
    output << "    if test \"$type\" = \"BROADCAST\"; then"<< endl;
    output << "      $IP -4 addr add $addr/$nm dev $dev brd + scope global label $dev:FWB${va_num}"  << endl;
    output << "      va_num=`expr $va_num + 1`"            << endl;
    output << "    fi"                                     << endl;
    output << "  fi"                                       << endl;
    output << "}"                                          << endl;

    output << "getaddr() {"                                 << endl;
    output << "  dev=$1"                                    << endl;
    output << "  name=$2"                                   << endl;
    output << "  L=`$IP -4 addr show dev $dev | tail -n1`"   << endl;
    output << "  test -z \"$L\" && { "                      << endl;
    output << "    eval \"$name=''\""                       << endl;
    output << "    return"                                  << endl;
    output << "  }"                                         << endl;
    output << "  OIFS=$IFS"                                 << endl;
    output << "  IFS=\" /\""                                << endl;
    output << "  set $L"                                    << endl;
    output << "  eval \"$name=$2\""                         << endl;
    output << "  IFS=$OIFS"                                 << endl;
    output << "}"                                           << endl;
    output << endl;
    output << endl;

/* we use function getinterfaces to process wildcard interfaces */

    output << "getinterfaces() {"                           << endl;
    output << "  NAME=$1"                                   << endl;
    output << "  $IP link show | grep -E \"$NAME[^ ]*: \"| while read L; do" << endl;
    output << "    OIFS=$IFS"                               << endl;
    output << "    IFS=\" :\""                              << endl;
    output << "    set $L"                                  << endl;
    output << "    IFS=$OIFS"                               << endl;
    output << "    echo $2"                                 << endl;
    output << "  done"                                      << endl;
    output << "}"                                           << endl;
    output << endl;
    output << endl;



    output << "LSMOD=\""   +path_lsmod+"\"\n";
    output << "MODPROBE=\""+path_modprobe+"\"\n";
    output << "IPTABLES=\""+path_iptables+"\"\n";
    output << "IP=\""      +path_ip+"\"\n";
    output << "LOGGER=\""  +path_logger+"\"\n";
    output << endl;

/* check if package iproute2 is installed, but do this only if
 * we really need /usr/sbin/ip 
 */
    if (options->getBool("verify_interfaces") || 
        options->getBool("manage_virtual_addr") ||
        options->getBool("configure_interfaces") )
    {
        output << "if $IP link ls >/dev/null 2>&1; then" << endl;
        output << "  echo;"                              << endl;
        output << "else"                                 << endl;
        output << "  echo \"iproute not found\""         << endl;
        output << "  exit 1"                             << endl;
        output << "fi"                                   << endl;
    }

#if 0
    output << "check lsmod    $LSMOD"    << endl;
    output << "check modprobe $MODPROBE" << endl;
    output << "check iptables $IPTABLES" << endl;
    output << "check ip       $IP"       << endl;
#endif

    output << endl;
}

void OSConfigurator_linux24::generateCodeForProtocolHandlers(bool have_nat)
{
    FWOptions* options=fw->getOptionsObject();

/* there is no need to load modules on linksys */
    if (options->getBool("load_modules") &&
        fw->getStr("host_OS")!="linksys")
    {
	output << endl;
	output << "MODULE_DIR=\"/lib/modules/`uname -r`/kernel/net/ipv4/netfilter/\" " << endl;
/*
 * TODO: make better version comparison
 */
        string version=fw->getStr("version");
        if (have_nat)
        {
            output << "MODULES=`(cd $MODULE_DIR; ls *_conntrack_* *_nat_* | sed -n -e 's/\\.ko$//p' -e 's/\\.o$//p' -e 's/\\.ko\\.gz$//p' -e 's/\\.o\\.gz$//p')`"
                   << endl;
        } else {
            output << "MODULES=`(cd $MODULE_DIR; ls *_conntrack_* | sed -n -e 's/\\.ko$//p' -e 's/\\.o$//p' -e 's/\\.ko\\.gz$//p' -e 's/\\.o\\.gz$//p')`"
                   << endl;
        }

	output << "for module in $MODULES; do " << endl;
        output << "  if $LSMOD | grep ${module} >/dev/null; then continue; fi" << endl;
	output << "  $MODPROBE ${module} ||  exit 1 " << endl;
	output << "done" << endl;
	output << endl;
	output << endl;
    }
}

void OSConfigurator_linux24::epilog()
{
    FWOptions* options=fw->getOptionsObject();

    try {
	output << "#" << endl;
	output << "#" << endl;

/* Turn on packet forwarding if we have to */

        string s=options->getStr("linux24_ip_forward");
        if (!s.empty())
        {
            if (s=="1" || s=="On" || s=="on") s="1";
            else                              s="0";
            output << "echo " << s << " > /proc/sys/net/ipv4/ip_forward\n\n";
        }
//        else
//            output << "echo \"$FWD\" > /proc/sys/net/ipv4/ip_forward\n\n";

    } catch (FWException ex) {
	error(ex.toString());
	exit(1);
    }
}
