;#
;# Copyright (c) 1996, Ikuo Nakagawa.
;# All rights reserved.
;#
;# $Id: dirinfo.pl,v 1.2 1997/05/22 08:37:37 ikuo Exp $
;#
;# Last updated:
;#	1996/09/20 by Ikuo Nakagawa
;# Description:
;#	dirinfo.pl - generate/load/store directory information.
;#
package dirinfo;
;#
require "log.pl";
;# prototypes
;# sub load_dirinfo($);
;# sub store_dirinfo($$);
;# sub make_dirinfo($;$);
;# sub update_dirinfo($);
;# sub md5checksum($);

;#
$md5_program
 = -x "/sbin/md5" ? "/sbin/md5"
 : -x "/usr/bin/md5" ? "/usr/bin/md5"
 : -x "/usr/local/bin/md5" ? "/usr/local/bin/md5"
 : undef;

;#
@{$required{'directory'}} = qw(begin-update end-update);
@{$required{'symlink'}} = qw(linkto);
@{$required{'file'}} = qw(md5checksum modified size owner group mode);

;#
sub load_dirinfo {
 my($dir) = @_;
 local(*FILE, $_);
 my $dirinfo = -d $dir ? "$dir/.dirinfo" : $dir;
 my $assoc = {};
 my $pp = undef;

 # get directory info if exists
 open(FILE, $dirinfo) || return $assoc;

 #
 while (<FILE>) {
  s/^\s+//; s/\s+$//; next if /^$/ || /^#/;
  /\s*=\s*/ || next;
  my($key, $val) = ($`, $');
  if (exists($required{$key})) {
   $pp = \%{$assoc->{$val}};
   $pp->{'type'} = $key;
   $pp->{'file'} = $val;
  } else {
   $pp->{$key} = $val;
  }
 }
 close(FILE);

 #
 $assoc;
}

;# write directory information
sub store_dirinfo {
 my($dir, $assoc) = @_;
 local(*FILE, *TEMP, $_, $f);
 my $dirinfo = -d $dir ? "$dir/.dirinfo" : $dir;
 my $tmpinfo = "$dirinfo.$$";

 # next, we open the file with write mode
 open(TEMP, ">$tmpinfo")
  || do { log::putl("WARNING", "open($tmpinfo): $!"), return undef };

 # current directory information
 if (defined($assoc->{'.'})) {
  my $pp = \%{$assoc->{'.'}};
  my $key;

  print TEMP "$pp->{'type'} = .\n";
  for $key (@{$required{'directory'}}) {
   print TEMP " $key = $pp->{$key}\n" if exists($pp->{$key});
  }
 }

 #
 for $f (sort keys %{$assoc}) {

  # make reference to assoc
  my $pp = $assoc->{$f};
  my $key;

  # check filename
  # we hate illegal filenames, but '.' is a special filename
  # which contains current directory information.
  # Information for '.' is stored first.
  next if $f eq '' || $f eq '.' || $f eq '..';
  next if $f =~ /^\.dirinfo/;

  print TEMP "$pp->{'type'} = $f\n";
  for $key (@{$required{$pp->{'type'}}}) {
   print TEMP " $key = $pp->{$key}\n" if exists($pp->{$key});
  }
 }
 # close temporary info file once.
 close(TEMP);

 #
 if (open(FILE, $dirinfo) && open(TEMP, $tmpinfo)) {
  my($x, $y, $comp);
  for ($comp = 0; $comp == 0; $comp = $x cmp $y) {
   ($x = <FILE>) =~ s/\s+$//;
   ($y = <TEMP>) =~ s/\s+$//;
   last if !defined($x) && !defined($y);
  }
  close(FILE); close(TEMP);
  if ($comp == 0) {
#  log::putl("DEBUG", "$dirinfo: no change");
   unlink($tmpinfo), return 0;
# OUTPUT DIFF?
# } else {
#  log::putl("DEBUG", "$dirinfo: - $x");
#  log::putl("DEBUG", "$dirinfo: + $y");
  }
 }

 #
 if (!rename($tmpinfo, $dirinfo)) {
  unlink($tmpinfo);
  log::putl("WARNING", "rename($tmpinfo, $dirinfo): $!"), return undef;
 }

 # We have changed!
#log::putl("DEBUG", "$dirinfo: updated");
 return 1;
}

;#
;# make directory information
;# make_dirinfo sets values of begin-update / end-update for directories
;# if and only if old dirinfo contains them.
;# Other values are re-generated by make_dirinfo.
;#
sub make_dirinfo {
 my($dir, $need_md5checksum) = @_;
 local(*DIR, $[, $_);
 local($old, $new, @stat, @list, $file);
 my $dirinfo = "$dir/.dirinfo";

 # get directory info if exists
 $old = &load_dirinfo($dir);
 $new = {};

 #
# if (defined($old)) {
#  log::putl("DEBUG", "$dir... loading dirinfo ok");
# } else {
#  log::putl("DEBUG", "$dir... loading dirinfo failed");
# }

 # start-time
 my $begin_cur = time;

 # get directory entries
 opendir(DIR, $dir)
  || do { log::putl("NOTICE", "opendir($dir): $!"), return undef };
 @list = sort readdir(DIR);
 closedir(DIR);

 # checking all entries
 for $file (@list) {

  next if $file eq '' || $file eq '.' || $file eq '..';
  next if $file =~ /^\.dirinfo/;
  next if $file =~ m!/!;	# must not contain '/'

  if (!(@stat = lstat("$dir/$file"))) {
   log::putl("NOTICE", "stat($dir/$file): $!"), next;
  }

  my $pp = \%{$new->{$file}};
  my $po = exists($old->{$file}) ? $old->{$file} : undef;

  $pp->{'file'} = $file;
  $pp->{'path'} = "$dir/$file";

  if (-l _) {				# symlink?
   $pp->{'type'} = 'symlink';
   $pp->{'linkto'} = readlink("$dir/$file");
  } elsif (-f _) {			# normal file
   $pp->{'type'} = 'file';
   $pp->{'size'} = $stat[7];
   $pp->{'mode'} = sprintf("%04o", ($stat[2] & 0777));
   $pp->{'owner'} = getpwuid($stat[4]);
   $pp->{'group'} = getgrgid($stat[5]);
   $pp->{'modified'} = $stat[9];
   if ($need_md5checksum) {
    $pp->{'md5checksum'} = $po->{'md5checksum'}
     if exists($po->{'md5checksum'});
    $pp->{'md5checksum'} = &md5checksum("$dir/$file")
     if $pp->{'md5checksum'} !~ /^[a-f0-9]{32}$/;
   }
  } elsif (-d _) {			# directory
   $pp->{'type'} = 'directory';
   if (defined($po) &&
       defined($po->{'begin-update'}) &&
       defined($po->{'end-update'})) {
    $pp->{'begin-update'} = $po->{'begin-update'};
    $pp->{'end-update'} = $po->{'end-update'};
   }
  } else {				# unknown file type
   log::putl("DEBUG", "$dir/$file ... ignored");
  }
 } # end of for $file (@list) { ... }

 ;#
# my $pp = \%{$new->{'.'}};
# my $po = exists($old->{'.'}) ? $old->{'.'} : undef;
# $pp->{'type'} = 'directory';
# $pp->{'file'} = '.';
# if (defined($po) &&
#     defined($po->{'begin-update'}) &&
#     defined($po->{'end-update'})) {
#  $pp->{'begin-update'} = $po->{'begin-update'};
#  $pp->{'end-update'} = $po->{'end-update'};
# }

 ;#
 $new;
}

;#
;# update dirinfo recursively
;# update_dirinfo returns 1 iff any change was found.
;#
sub update_dirinfo {
 my($dir) = @_;
 local(*DIR, $[, $_);
 local($old, $new, @stat, @list, $file);
 my $dirinfo = "$dir/.dirinfo";

 # get directory info if exists
 $new = &make_dirinfo($dir, 1);
 $old = &load_dirinfo($dir);

 # get start-time
 my $begin_cur = time;

 #
 for $file (keys %{$new}) {
  next if $file eq '' || $file eq '.' || $file eq '..';
  next if $file =~ /^\.dirinfo/;
# next if $file =~ m!/!;	# must not contain '/'
  my $pp = $new->{$file};
  next if $pp->{'type'} ne 'directory';

  my $t = time;

  if (!defined($r = &update_dirinfo("$dir/$file"))) {
   log::putl("WARNING", "$dir/$file/ ... error in update_dirinfo, ignored");
   next;
  }

  if ($r || !defined($pp->{'begin-update'})
	 || !defined($pp->{'end-update'})) {
#  log::putl("DEBUG", "update time has been changed");
   $pp->{'begin-update'} = $t;
   $pp->{'end-update'} = time;
# } else {
#  log::putl("DEBUG", "update time has no change");
  }
 } # end of for $file (@list) { ... }

 ;#
# my $po = exists($old->{'.'}) ? $old->{'.'} : undef;
# my $pp = \%{$new->{'.'}};
# $pp->{'type'} = 'directory';
# $pp->{'file'} = '.';
# if (defined($po)
#	 && defined($po->{'begin-update'})
#	 && defined($po->{'end-update'})) {
#  $pp->{'begin-update'} = $po->{'begin-update'};
#  $pp->{'end-update'} = $po->{'end-update'};
#  } else {
#  $pp->{'begin-update'} = $begin_cur;
#  $pp->{'end-update'} = time;
# }

 ;#
 &store_dirinfo($dir, $new);
}

;#
sub md5checksum {
 local($file) = @_;
 local($_);

 -f $file || return undef;
 defined($md5_program) || return undef;
 chomp($_ = `$md5_program '$file'`) || return undef;
 return undef if $?;
 $_ = $' if /\s*=\s*/;
 /^[a-f0-9]{32}$/ ? $_ : undef;
}

;#
1;
