/* C Mode */

/* fdscript.c
   The top level FDScript interpreter for FramerD
   Originally implemented by Ken Haase in the Machine Understanding Group
     at the MIT Media Laboratory.

   Copyright (C) 1994-2001 Massachusetts Institute of Technology
   Copyright (C) 2001-2002 beingmeta, inc. (A Delaware Corporation)

   This program comes with absolutely NO WARRANTY, including implied
   warranties of merchantability or fitness for any particular purpose.

    Use, modification, and redistribution of this program is permitted
    under the terms of either (at the developer's discretion) the GNU
    General Public License (GPL) Version 2, the GNU Lesser General Public
    License.

    This program is based on the FramerD library released in Fall 2001 by
    MIT under both the GPL and the LGPL licenses, both of which accompany
    this distribution.  Subsequent modifications by beingmeta, inc. are
    also released under both the GPL and LGPL licenses (at the developer's
    discretion).
*/

static char vcid[] = "$Id: fdscript.c,v 1.13 2002/06/01 21:11:50 haase Exp $";

#include "framerd/fdscript.h"
#include "framerd/plugins.h"
#include "time.h"

/* Readline stubs */

extern char *fd_readline(char *prompt);
extern fd_lisp fd_read_exprs(char *prompt);
extern fd_lisp fd_console_loop(fd_lispenv env);
extern fd_lisp fd_busy_console_loop(fd_lispenv env);
extern fd_lisp fd_promptless_loop(fd_lispenv env);
FDSCRIPT_EXPORT void fd_initialize_fdtext(void);

FDSCRIPT_EXPORT fd_lispenv fd_osprims_env;
FRAMERD_EXPORT fd_lispenv fd_enabled_env;
IMPORTED fd_lispenv fd_texttools_env;
IMPORTED fd_lispenv fd_internals_env;
IMPORTED fd_lispenv fd_fdmaint_env;

#ifndef CLOCKS_PER_SEC
#define CLOCKS_PER_SEC 1000
#endif

#ifndef SPECIAL_INITS
#define SPECIAL_INITS
#endif

#if DO_CGI_INIT
extern void fd_cgi_init(void);
#define MAYBE_INIT_CGI() fd_cgi_init()
#else
#define MAYBE_INIT_CGI() 
#endif

extern int fd_force_object_names;
void fd_interactive_loop(FILE *in,FILE *out);


/* Special input handling procedures */

static fd_u8char *read_stdin_as_string()
{
  struct FD_STRING_STREAM ss; int c;
  FD_INITIALIZE_STRING_STREAM(&ss,1024);
  while ((c=fd_fgetc(stdin)) >= 0) fd_sputc(&ss,c);
  return ss.ptr;
}

static fd_lisp read_stdin_as_lisp()
{
  fd_lisp answer=FD_EMPTY_CHOICE;
  while (1) {
    fd_lisp elt=fd_parse_lisp_from_stream(stdin);
    if (FD_EOF_OBJECTP(elt)) return answer;
    else {FD_ADD_TO_CHOICE(answer,elt);}}
}

static fd_lisp stdin_args;
static char *stdin_string;

fd_lisp listref(fd_lisp lst,int i)
{
  FD_DOLIST(elt,lst)
    if (i == 0) return elt; else i--;
  return FD_VOID;
}

void init_variable(char *name,fd_lisp value)
{
  fd_lisp symbol=fd_make_symbol(name);
  fd_set_symbol_value(symbol,value);
}

static void set_session_mnemonic(char *filename)
{
  /* Sets the session mnemonic to filename with its .x prefix stripped */
  char *copy=fd_strdup(filename), *scan=copy, *slash=NULL, *dot=NULL;
  while (*scan)
    if ((*scan == '/') || (*scan == '\\')) {dot=NULL; slash=++scan;}
    else if (*scan == '.') dot=scan++;
    else scan++;
  if (dot) *dot=0;
  if ((slash) && ((dot==NULL) | (slash<dot)))
    fd_set_session_mnemonic(slash);
  else fd_set_session_mnemonic(copy);
  free(copy);
}

static fd_lisp string_arg(char *string)
{
  fd_u8char *s=fd_convert_os_string(string);
  fd_u8char *result=fd_interpret_unicode_escapes(s);
  free(s); return fd_init_string(result,-1);
}


/* Starting new REPs */

static fd_lisp push_rep_lexpr(fd_lisp args)
{
  fd_lisp env=fd_get_arg(args,0,FD_FALSE);
  if (FD_PRIM_TYPEP(env,env_type)) {
    fd_lisp v=fd_console_loop(FD_CPTR_DATA(env));
    fd_decref(v);
    return fd_incref(env);}
  else {
    fd_lispenv m=fd_make_module(); fd_lisp v;
    fd_module_uses(m,fd_enabled_env);
    fd_module_uses(m,fd_osprims_env);
    fd_module_uses(m,fd_texttools_env);
    fd_module_uses(m,fd_global_modules);
    fd_module_uses(m,fd_restricted_modules);
    fd_module_uses(m,fd_fdinternals_env);
    fd_module_uses(m,fd_fdmaint_env);
    v=fd_console_loop(m); fd_decref(v);
    return fd_make_cptr(env_type,m);}
}

static fd_lisp push_busy_rep_lexpr(fd_lisp args)
{
  fd_lisp env=fd_get_arg(args,0,FD_FALSE);
  if (FD_PRIM_TYPEP(env,env_type)) {
    fd_lisp v=fd_busy_console_loop(FD_CPTR_DATA(env));
    fd_decref(v);
    return fd_incref(env);}
  else {
    fd_lispenv m=fd_make_module(); fd_lisp v;
    fd_module_uses(m,fd_enabled_env);
    fd_module_uses(m,fd_global_modules);
    fd_module_uses(m,fd_restricted_modules);
    fd_module_uses(m,fd_fdinternals_env);
    fd_module_uses(m,fd_osprims_env);
    fd_module_uses(m,fd_texttools_env);
    fd_module_uses(m,fd_fdmaint_env);
    v=fd_busy_console_loop(m); fd_decref(v);
    return fd_make_cptr(env_type,m);}
}

static void process_config_assignment(char *start)
{
  fd_lisp symbol, value; int add=0;
  char *equals=strchr(start,'=');
  char *buf=fd_xmalloc(equals-start+1);
  strncpy(buf,start,equals-start); buf[equals-start]=0;
  symbol=fd_make_symbol(buf);
  if (equals[1]=='+') {add=1; equals++;}
  if ((equals[1]==' ') || (equals[1]=='\0')) value=FD_VOID;
  else value=fd_parse_arg(equals+1);
  if (add) {
    fd_lisp val=fd_symbol_value(symbol);
    if (FD_VOIDP(val))
      fd_set_symbol_value(symbol,value);
    else {
      FD_ADD_TO_CHOICE(val,fd_incref(value));
      fd_set_symbol_value(symbol,val);
      fd_decref(val);}}
  else fd_set_symbol_value(symbol,value);      
  fd_decref(value);
}


/* The Main Event */

int main(int argc,char *argv[])
{
  char *load_filename=NULL, *extra_config=NULL;
  int interactive=0, i=1, argn=0, busy=0, promptless=0, no_profile=0, load_pos;
  fd_lisp args=FD_EMPTY_LIST, *tail=&args; fd_lispenv default_env;

  fd_cmd_args(&argc,&argv);

  stdin_args=FD_VOID; stdin_string=NULL;

  /* Look ahead for special args.  This will stop at
     the load filename or the end of argv.  We do not actual
     process any args that require the FramerD libraries
     to be initialized. */
  while (i < argc) {
    if (strcmp(argv[i],"--noprofile") == 0) {
      no_profile=1; i++;}
    else if (strcmp(argv[i],"--noconfig") == 0) {
      fd_suppress_config(); i++;}
    else if (strcmp(argv[i],"--noherald") == 0) {
      fd_inhibit_herald(1); i++;}
    else if (strchr(argv[i],'=')) i++;
    else if (i+1 >= argc) break;
    else {
      char *argname=argv[i], *argval=argv[i+1];
      if ((strcmp(argv[i],"-m") == 0) ||
	  (strcmp(argv[i],"-c") == 0) ||
	  (strcmp(argv[i],"-f") == 0) ||
	  (strcmp(argv[i],"-x") == 0) ||
	  (strcmp(argv[i],"--module") == 0) ||
	  (strcmp(argv[i],"--config") == 0) ||
	  (strcmp(argv[i],"--file") == 0) ||
	  (strcmp(argname,"--charset") == 0))
	i=i+2;
      else if (argv[i][0] == '-')
	if (fd_file_existsp(argv[i])) break;
	else {
	  fprintf(stderr,_("Invalid flag %s"),argv[i]);
	  exit(1);}
      else break;}}
  /* This is where the load filename is */
  load_pos=i;
    
  /* Library initializations */
  fd_set_build_date(__DATE__);
  fd_initialize_fdscript();

  /* Set up the default module */
  default_env=fd_make_module();
  fd_module_uses(default_env,fd_enabled_env);
  fd_module_uses(default_env,fd_osprims_env);
  fd_module_uses(default_env,fd_fdinternals_env);
  fd_module_uses(default_env,fd_fdmaint_env);
  fd_module_uses(default_env,fd_global_modules);
  fd_module_uses(default_env,fd_restricted_modules);

  /* Initialize the console functions and access
     the fdtext module */
  fd_initialize_console_c();
  fd_initialize_fdtext();
  fd_module_uses(default_env,fd_texttools_env);


  /* Determine how you are running: interactive, promptless,
     interactive with args, etc.  Also find out the load filename. */
  if (load_pos == argc) { /* No filename arg */
    interactive=1;
    set_session_mnemonic(argv[0]); load_filename=NULL;
    if (no_profile == 0) fd_use_profile();
    fd_show_poolids(1); fd_control_frame_printing(2);}
  else if (strcmp(argv[load_pos],"-") == 0) { /* filename is `stdin' */
    /* Running promptless */
    interactive=1; promptless=1;
    fd_show_poolids(0); load_filename=NULL;
    fd_disable_notifications();}
  else if (strcmp(argv[load_pos],".") == 0) {
    /* Running interactive with args */
    interactive=1; promptless=0;
    fd_show_poolids(0); load_filename=NULL;}
  else {
    /* Running like a script */
    fd_disable_notifications();
    fd_show_poolids(0);
    set_session_mnemonic(argv[load_pos]);
    load_filename=fd_strdup(argv[load_pos]);}
  
  /* Now, process the arguments which fdscript handles */
  i=1; while (i < argc) {
    if (strchr(argv[i],'='))
      process_config_assignment(argv[i++]);
    else if (strncmp(argv[i],"-D",2) == 0)
      process_config_assignment(argv[i++]+2);
    else if ((strcmp(argv[i],"--noprofile") == 0) ||
	     (strcmp(argv[i],"--noconfig") == 0) ||
	     (strcmp(argv[i],"--noherald") == 0))
      /* These have already been processed */
      i++;
    else if (i < load_pos) {
      char *argname=argv[i++], *argval=argv[i++];
      if ((strcmp(argname,"--charset") == 0) ||
	  (strcmp(argname,"-x") == 0)) {
	fd_set_default_encoding(argval);
	fd_set_file_encoding(stdin,argval);
	fd_set_file_encoding(stdout,argval);
	fd_set_file_encoding(stderr,argval);}
      else if ((strcmp(argname,"-m") == 0) ||
	  (strcmp(argname,"--module") == 0)) {
	fd_u8char *module_name=fd_convert_os_string(argv[i]+2);
	fd_lispenv module=fd_get_module(module_name,NULL,1);
	if (module == NULL) fd_warn("Couldn't locate module %s",module_name);
	else fd_module_uses(default_env,module);
	fd_xfree(module_name);}
      else if ((strcmp(argname,"-c") == 0) ||
	       (strcmp(argname,"--config") == 0)) {
	fd_load_config(argval);}
      else if ((strcmp(argname,"-f") == 0) ||
	       (strcmp(argname,"--file") == 0)) {
	fd_u8char *fname=fd_convert_os_string(argval);
	fd_load_file(fname,NULL,default_env);
	fd_xfree(fname); i++;}
      else {
	fprintf(stderr,"Unknown option %s\n",argname);
	exit(1);}}
    else if (i == load_pos) {
      if (load_filename)
	*tail=FD_MAKE_LIST1(fd_make_string(load_filename));
      else *tail=FD_MAKE_LIST1(fd_make_string(argv[load_pos]));
      tail=&(FD_CDR(*tail));
      argn++;
      i++;}
    else {
      fd_lisp arg=FD_VOID;
      if ((argn) && (strcmp(argv[i],"-") == 0))
	/* Read STDIN as your argument */
	if (FD_VOIDP(stdin_args))
	  if (stdin_string)
	    fd_warn(_("Can't use stdin as both lisp and string"));
	  else arg=stdin_args=read_stdin_as_lisp();
	else arg=fd_incref(stdin_args);
      else if (strcmp(argv[i],"$") == 0) {
	if (stdin_string == NULL)
	  if (FD_VOIDP(stdin_args))
	    stdin_string=read_stdin_as_string();
	  else fd_warn(_("Can't use stdin as both lisp and string"));
	else arg=fd_make_string(stdin_string);}
      else arg=string_arg(argv[i]);
      if (!(FD_VOIDP(arg))) {
	/* Add it to the list of arguments */
	*tail=FD_MAKE_LIST1(arg); tail=&(FD_CDR(*tail));
	argn++;}
      i++;}}
  
  /* Other inits */
  SPECIAL_INITS;

  fd_add_restricted_lexpr("%PUSH",FD_ND_LEXPR,push_rep_lexpr);
  fd_add_restricted_lexpr("%BUSY",FD_ND_LEXPR,push_busy_rep_lexpr);

  /* Initialize argument variables */
  init_variable("ARGS",args);
  init_variable("NARGS",FD_LISPFIX(argn));
  init_variable("ARG0",listref(args,0));
  init_variable("ARG1",listref(args,1));
  init_variable("ARG2",listref(args,2));
  init_variable("ARG3",listref(args,3));
  init_variable("ARG4",listref(args,4));

  if (fd_string_getenv("ALWAYS_BUSY")) busy=1;

  if (interactive) {
    char *init_name=fd_string_getenv("FD_INTERACTIVE_INIT");
    fd_lisp value;
    if (init_name) {
      value=fd_load_file(init_name,NULL,default_env);
      fd_xfree(init_name);}
    if (promptless) value=fd_promptless_loop(default_env);
    else if (busy) value=fd_busy_console_loop(default_env);
    else value=fd_console_loop(default_env);
    fd_decref(value);}
  else {
    fd_lisp main_symbol=fd_make_symbol("MAIN"), main_proc, value, v;
    char *absolute=fd_absolute_pathname(load_filename);
    FD_WITH_HANDLING {
      v=fd_load_file(absolute,NULL,default_env);
      fd_decref(v);}
    FD_ON_EXCEPTION {
      fd_fprintf(stderr,_("Error while loading %s: %s (%s) %q\n"),
		 load_filename,fd_theException(),
		 fd_exception_details(),fd_exception_object());
      fd_reraise();}
    FD_END_HANDLING;
    fd_xfree(absolute);
    main_proc=fd_symeval(main_symbol,default_env);
    if (FD_VOIDP(main_proc)) {}
    else if (FD_XPROCP(main_proc)) {
      v=fd_apply(main_proc,FD_CDR(args));}
    else {
      v=fd_apply(main_proc,FD_EMPTY_LIST);}
    fd_decref(main_proc); fd_decref(v);}
  if (interactive | ((clock()/CLOCKS_PER_SEC) > 1))
    fd_report_framerd_stats(stderr);
  if (busy) fd_describe_mallocation();
#if WIN32 /* atexit doesn't seem to work under MINGW */
  fd_commit_pools();
  fd_commit_indices();
#endif
  fd_exit(0);
  return 0;
}





/* File specific stuff */

/* The CVS log for this file
   $Log: fdscript.c,v $
   Revision 1.13  2002/06/01 21:11:50  haase
   Add handling of command line var configs which reset vars

   Revision 1.12  2002/04/02 21:39:32  haase
   Added log and emacs init entries to C source files

*/

/* Emacs local variables
;;;  Local variables: ***
;;;  compile-command: "cd ../..; make" ***
;;;  End: ***
*/
