/* -*- C -*-

  Copyright (c) 1994-2001 by the Massachusetts Institute of Technology.
  Copyright (c) 2001-2002 by beingmeta, inc. (A Delaware Corporation)
  All rights reserved.

  $Id: index.h,v 1.14 2002/06/29 01:25:58 haase Exp $

  This file is part of FramerD, a representation language and semantic
  database developed by Kenneth B. Haase and his students at the Media
  Laboratory at the Massachusetts Institute of Technology in Cambridge,
  Massachusetts.  Research at the Media Lab is supported by funds and
  equipment from a variety of corporations and government sponsors whose
  contributions are gratefully acknowledged.

    Use, modification, and redistribution of this program is permitted
    under the terms of either (at the developer's discretion) the GNU
    General Public License (GPL) Version 2, the GNU Lesser General Public
    License.

    This program is based on the FramerD library released in Fall 2001 by
    MIT under both the GPL and the LGPL licenses, both of which accompany
    this distribution.  Subsequent modifications by beingmeta, inc. are
    also released under both the GPL and LGPL licenses (at the developer's
    discretion).

*************************************************************************/

#ifndef FRAMERD_INDEX_H
#define FRAMERD_INDEX_H

#include "framerd/common.h"
#include "framerd/cons.h"
#include "framerd/os.h"

/* Exceptions */

FRAMERD_EXPORT fd_exception
  fd_NotFileIndex,
  fd_ReadOnlyIndex,
  fd_NoKeysMethod,
  fd_NoHashMethod,
  fd_BadIndexSpec;

#define FD_FILE_INDEX_MAGIC_NUMBER 151913496
#define FD_FILE_INDEX_SNAPSHOT_MAGIC_NUMBER 152243728

#if FD_USING_THREADS 
#define FD_INDEX_FIELDS() \
  fd_u8char *id; enum FD_INDEX_TYPE type; \
  struct FD_INDEX_HANDLER *handler; \
  int read_only; int zipf_threshold; \
  struct FD_HASHTABLE cache, adds, drops, sizes; \
  struct FD_HASHSET interned_values; \
  fd_lisp for_slotid; \
  int cache_size, adds_size, drops_size, sizes_size; \
  struct FD_INDEX *next; \
  fd_mutex lock
#else
#define FD_INDEX_FIELDS() \
  fd_u8char *id; enum FD_INDEX_TYPE type; \
  struct FD_INDEX_HANDLER *handler;  \
  int read_only; int zipf_threshold; \
  struct FD_HASHTABLE cache, adds, drops, sizes; \
  struct FD_HASHSET interned_values; \
  fd_lisp for_slotid; \
  int cache_size, adds_size, drops_size, sizes_size; \
  struct FD_INDEX *next
#endif

/* All index structures start like this */
typedef struct FD_INDEX {
  FD_INDEX_FIELDS();} *fd_index;

struct FD_FILE_INDEX {
  FD_INDEX_FIELDS();
  char *filename; char *buf; FILE *store;
  unsigned int size; unsigned int *offsets;
  int at_end; int preloaded;};
typedef struct FD_FILE_INDEX *fd_file_index;

struct FD_NETWORK_INDEX {
  FD_INDEX_FIELDS();
  fd_server conn;};
typedef struct FD_NETWORK_INDEX *fd_network_index;

struct FD_INDEX_HANDLER {
  fd_lisp (*ix_get)(struct FD_INDEX *,fd_lisp key);
  int  (*ix_get_size)(struct FD_INDEX *,fd_lisp key);
  void  (*ix_add)(struct FD_INDEX *,fd_lisp key,fd_lisp values);
  void  (*ix_drop)(struct FD_INDEX *,fd_lisp key,fd_lisp values);
  fd_lisp (*ix_fetch)(struct FD_INDEX *,fd_lisp key);
  int  (*ix_fetch_size)(struct FD_INDEX *,fd_lisp key);
  void (*ix_commit)(struct FD_INDEX *);
  void (*ix_prefetch)(struct FD_INDEX *,fd_lisp keys);
  fd_lisp (*ix_dir)(struct FD_INDEX *);
  void (*ix_spend_memory)(struct FD_INDEX *);
  void (*ix_close)(struct FD_INDEX *);};

FRAMERD_EXPORT fd_index fd_all_indices;

FRAMERD_EXPORT void fd_register_index(fd_index ix);

FRAMERD_EXPORT
void fd_register_file_index_opener
 (int prefix_code,fd_index (*opener)(fd_u8char *c));

struct FD_FILE_INDEX_OPENER {
  int magic_number; fd_index (*opener)(fd_u8char *);
  struct FD_FILE_INDEX_OPENER *next;};

/* These structures are used when searching for a key in a file index.
   <probe> is the slot in the hash table found and <point> is the beginning of
   the key entry to which it points. */

struct INDEX_LOC {unsigned int probe, point, data, n_elts;};
struct INDEX_LOC_CACHE {fd_lisp key; unsigned int point, data, n_elts;};


/* Prototype definitions */

FRAMERD_EXPORT fd_index fd_open_index(fd_u8char *name);
FRAMERD_EXPORT fd_index fd_find_index(fd_u8char *name);
FRAMERD_EXPORT fd_index fd_interpret_index(fd_lisp spec);

FRAMERD_EXPORT fd_index fd_open_file_index(fd_u8char *name);
FRAMERD_EXPORT fd_network_index fd_open_network_index(char *,int,fd_u8char *);

FRAMERD_EXPORT int fd_set_search_max(int new_value);

FRAMERD_EXPORT void fd_intern_index_values(fd_index x);
FRAMERD_EXPORT void fd_cache_file_index(fd_file_index x);

FRAMERD_EXPORT fd_lisp fd_index_get(fd_index x,fd_lisp key,fd_lisp dflt);
FRAMERD_EXPORT unsigned int fd_index_get_size(fd_index x,fd_lisp key);
FRAMERD_EXPORT void fd_index_add(fd_index x,fd_lisp key,fd_lisp value);
FRAMERD_EXPORT void fd_index_drop(fd_index x,fd_lisp key,fd_lisp value);
FRAMERD_EXPORT void fd_index_set(fd_index x,fd_lisp key,fd_lisp value);
FRAMERD_EXPORT void fd_index_zap(fd_index x,fd_lisp key);
FRAMERD_EXPORT void fd_index_remove(fd_index x,fd_lisp key,fd_lisp to_remove);

FRAMERD_EXPORT void fd_index_prefetch(fd_index ix,fd_lisp oids);

FRAMERD_EXPORT fd_lisp fd_index_keys(fd_index idx);
FRAMERD_EXPORT void fd_commit_index(fd_index x);
FRAMERD_EXPORT void fd_index_set_sizes
  (fd_index x,int cache_size,int adds_size,int drops_size,int sizes_size);
FRAMERD_EXPORT void fd_revert_index(fd_index x);
FRAMERD_EXPORT void fd_swap_out_index(fd_index idx);
FRAMERD_EXPORT
void fd_set_index_zipf_threshold(fd_index ix,int threshold);

FRAMERD_EXPORT void fd_auto_cache_file_indices(void);
FRAMERD_EXPORT void fd_commit_indices(void);
FRAMERD_EXPORT void fd_revert_indices(void);
FRAMERD_EXPORT void fd_swap_out_indices(void);

FRAMERD_EXPORT void fd_close_index(fd_index x);

FRAMERD_EXPORT void fd_for_indices
   (void (*fcn)(fd_index p,void *arg),void *arg);
FRAMERD_EXPORT int fd_get_index_count(void);
FRAMERD_EXPORT void fd_index_report_stats(FILE *stream);

/* Internal functions for external use */

FRAMERD_EXPORT unsigned int fd_hash_dtype(fd_lisp x); 
FRAMERD_EXPORT int fd_dtype_compare (fd_lisp key,FILE *stream);

/* File index specific functions */

FRAMERD_EXPORT fd_lisp fd_file_index_collect_values
  (fd_file_index ix,unsigned int loc,unsigned int size);
FRAMERD_EXPORT void fd_repack_file_index
  (char *ofile,char *dfile,unsigned int nsize);
FRAMERD_EXPORT void fd_preload_file_index(fd_index idx_arg);
FRAMERD_EXPORT void fd_unpreload_file_index(fd_index idx_arg);
FRAMERD_EXPORT fd_lisp fd_read_file_index_metadata
   (FILE *f,int *revnum,int *size,time_t *,time_t *,time_t *);

#endif /* ndef FRAMERD_INDEX_H */


/* File specific stuff */

/* The CVS log for this file
   $Log: index.h,v $
   Revision 1.14  2002/06/29 01:25:58  haase
   Made dbtest relocatable

   Revision 1.13  2002/06/26 18:56:39  haase
   Remove leftover declaration for fd_unzipf_file_index

   Revision 1.12  2002/05/01 21:46:31  haase
   Renamed mutex/condvar/rwlock types to have fd_ prefixes

   Revision 1.11  2002/04/22 14:23:07  haase
   Added extended metadata to file pools and indices

   Revision 1.10  2002/04/10 03:01:50  haase
   Added version information to file pools and indices

   Revision 1.9  2002/04/02 21:41:08  haase
   Added log and emacs init entries to C source files

*/

/* Emacs local variables
;;;  Local variables: ***
;;;  compile-command: "cd ../..; make" ***
;;;  End: ***
*/
