/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-viewer-common.c,v 1.12 2004/11/12 12:13:27 nyoxi Exp $
 */

#include "etpan-viewer-common.h"
#include "etpan-viewer-common-types.h"

#include <ctype.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <ncurses.h>
#include <stdlib.h>
#include <string.h>
#include "etpan-msg-renderer.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"

void etpan_viewer_common_handle_key(struct etpan_subapp * app,
    struct etpan_viewer_common_app_state * state, int key)
{
  unsigned int list_lines;
  
  if (app->title != NULL)
    list_lines = app->height - 2;
  else
    list_lines = app->height - 1;
  
  switch (key) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
  case 'G':
    break;
  default:
    state->chosen = 0;
    break;
  }

  switch (key) {
  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
    state->chosen *= 10;
    state->chosen += key - '0';
    break;
  case 'G':
    if (state->count > 0) {
      if (state->chosen == 0)
        state->line = state->count - 1;
      else {
        state->line = state->chosen;
      }
    }
    
    state->chosen = 0;
    break;

  case 'h':
    if (state->cursor) {
      if (state->col > 0)
        state->col --;
    }
    else {
      state->col = state->left;
      if (state->col > (unsigned int) (app->width / 2))
        state->col -= app->width / 2;
      else
        state->col = 0;
    }
    break;
  case 'l':
    if (state->cursor)
      state->col ++;
    else {
      state->col = state->left + app->width - 1;
      state->col += app->width / 2;
    }
    break;
  case 'j':
  case KEY_DOWN:
    if (!state->cursor)
      state->line = state->top + list_lines - 1;
    
    state->line ++;
    break;
  case 'k':
  case KEY_UP:
    if (!state->cursor)
      state->line = state->top;
    
    if (state->line > 0)
      state->line --;
    break;
  case ' ':
  case KEY_NPAGE:
    if (!state->cursor)
      state->line = state->top + list_lines - 1;
    
    state->line += list_lines - 1;
    break;
  case '\b':
  case KEY_PPAGE:
    if (!state->cursor)
      state->line = state->top;
    
    if (state->line > list_lines - 1)
      state->line -= list_lines - 1;
    else
      state->line = 0;
    break;
  case KEY_HOME:
    state->line = 0;
    break;
  case KEY_END:
    if (state->count > 0)
      state->line = state->count - 1;
    break;
    
  case KEY_CTRL('R'):
    state->rot13_enabled = !state->rot13_enabled;
    break;
  }
}


void etpan_viewer_common_set_color(struct etpan_subapp * app,
    struct etpan_viewer_common_app_state * state)
{
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
  
  etpan_app_set_color(app->app, "viewer-normal",
      &state->viewer_normal_attr, A_NORMAL);
  etpan_app_set_color(app->app, "viewer-header",
      &state->viewer_header_attr, A_NORMAL);
  etpan_app_set_color(app->app, "viewer-signature",
      &state->viewer_signature_attr, A_NORMAL);
  etpan_app_set_color(app->app, "viewer-link",
      &state->viewer_link_attr, A_NORMAL);
  etpan_app_set_color(app->app, "viewer-description",
      &state->viewer_description_attr, A_NORMAL);
  etpan_app_set_color(app->app, "viewer-security",
      &state->viewer_security_attr, A_NORMAL);
  etpan_app_set_color(app->app, "viewer-error",
      &state->viewer_error_attr, A_NORMAL);
}

int etpan_viewer_common_init(struct etpan_subapp * app,
    struct etpan_viewer_common_app_state * state)
{
  /* colors */
  state->viewer_normal_attr = A_NORMAL;
  state->viewer_header_attr = A_NORMAL;
  state->viewer_signature_attr = A_NORMAL;
  state->viewer_link_attr = A_NORMAL;
  state->viewer_description_attr = A_NORMAL;
  state->viewer_security_attr = A_NORMAL;
  state->viewer_error_attr = A_NORMAL;
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;
  
  /* text to view */
  state->text = NULL;
  state->size = 0;
  state->attr = NULL;
  state->filename = NULL;
  state->f = NULL;
  state->free_flags = 0;

  /* cursor */
  state->count = 0;
  state->chosen = 0;
  state->col = 0;
  state->left = 0;
  state->line = 0;
  state->top = 0;
  state->current_offset = 0;

  /* options */
  state->rot13_enabled = 0;
  state->cursor = 0;
  
  return NO_ERROR;
}

void etpan_viewer_common_done(struct etpan_subapp * app,
    struct etpan_viewer_common_app_state * state)
{
  etpan_viewer_common_flush(state);
}

static void flush_buffer(struct etpan_subapp * app,
    WINDOW * w,
    unsigned int y, unsigned int x, char * output)
{
  mvwaddstr(w, y, x, output);
}

static int get_attr(struct etpan_viewer_common_app_state * state,
    char * text, size_t size, unsigned int * current_attr)
{
  unsigned int i;
  
  if (state->attr == NULL)
    return state->viewer_normal_attr;

  for(i = * current_attr ; i < carray_count(state->attr) ; i ++) {
    struct etpan_text_prop * prop;
    size_t offset;

    prop = carray_get(state->attr, i);

    offset = text - state->text;
    if ((offset >= prop->start) && (offset < prop->end)) {
      * current_attr = i;
      
      switch (prop->zone_type) {
      case ETPAN_VIEWER_ZONE_NORMAL:
        return state->viewer_normal_attr;
        
      case ETPAN_VIEWER_ZONE_HEADER:
        return state->viewer_header_attr;
        
      case ETPAN_VIEWER_ZONE_SIGNATURE:
        return state->viewer_signature_attr;
        
      case ETPAN_VIEWER_ZONE_LINK:
        return state->viewer_link_attr;
        
      case ETPAN_VIEWER_ZONE_DESCRIPTION:
        return state->viewer_description_attr;

      case ETPAN_VIEWER_ZONE_SECURITY:
        return state->viewer_security_attr;

      case ETPAN_VIEWER_ZONE_ERROR:
        return state->viewer_error_attr;
      }
    }
  }

  return state->viewer_normal_attr;
}

#define UNVISIBLE ' '

#define etpan_is_print(a) (isprint((unsigned char) (a)) || \
  ((unsigned char) (a) >= 0xa0))

static void draw_text(struct etpan_subapp * app,
    struct etpan_viewer_common_app_state * state,
    WINDOW * w, char * output, char * fill,
    char * text, size_t size, unsigned int left_offset, int rot13_enabled,
    char * help_text)
{
  unsigned int x;
  unsigned int y;
  unsigned int buflen;
  int cur_attr;
  unsigned int col;
  unsigned int percent;
  unsigned int list_lines;
  unsigned int current_attr;

  list_lines = app->display_height - 1;

  /* text */
  
  x = 0;
  y = 0;
  current_attr = 0;
  cur_attr = get_attr(state, text, size, &current_attr);
  wattron(w, cur_attr);
  col = 0;
  buflen = 0;

  while (size > 0) {
    int new_attr;
    
    new_attr = get_attr(state, text, size, &current_attr);
    if (new_attr != cur_attr) {
      output[buflen] = '\0';
      flush_buffer(app, w, y, x, output);
      x += buflen;
      buflen = 0;
      wattroff(w, cur_attr);
      cur_attr = new_attr;
      wattron(w, cur_attr);
    }
    
    if (x + buflen >= (unsigned int) app->display_width) {
      if (buflen != 0) {
        output[buflen] = '\0';
        flush_buffer(app, w, y, x, output);
        x += buflen;
        buflen = 0;
      }
      
      if (* text == '\n') {
        buflen = 0;
        
        y ++;
        x = 0;
        col = 0;
        
        if (y >= list_lines)
          break;
      }
    }
    else {
      if (* text == '\n') {
        while (x + buflen < (unsigned int) app->display_width) {
          output[buflen] = UNVISIBLE;
          buflen ++;
        }
        
        output[buflen] = '\0';
        flush_buffer(app, w, y, x, output);
        
        buflen = 0;
        
        y ++;
        x = 0;
        col = 0;
        
        if (y >= list_lines)
          break;
      }
      else {
        if (col >= left_offset) {
          if (!etpan_is_print(* text)) {
            output[buflen] = UNVISIBLE;
            buflen ++;
          }
          else {
            if (rot13_enabled) {
              if ((* text >= 'A') && (* text <= 'Z'))
              output[buflen] = ((* text - 'A') + 13) % 26 + 'A';
              else if ((* text >= 'a') && (* text <= 'z'))
                output[buflen] = ((* text - 'a') + 13) % 26 + 'a';
              else
              output[buflen] = * text;
            }
            else {
              output[buflen] = * text;
            }
            buflen ++;
          }
        }
        else {
          col ++;
        }
      }
    }
    text ++;
    size --;
  }

  while (y < list_lines) {
    while (x + buflen < (unsigned int) app->display_width) {
      output[buflen] = UNVISIBLE;
      buflen ++;
    }
    
    output[buflen] = '\0';
    flush_buffer(app, w, y, x, output);
    
    buflen = 0;
    y ++;
    x = 0;
  }

  attroff(cur_attr);

  /* status bar */

  /*
  if (state->count != 0)
    percent = state->line * 100 / state->count;
  else
    percent = 0;
  */
 
  if (state->count == 0)
    percent = 0;
  else if (state->count <= list_lines)
	percent = 100;
  else
    percent = (state->top+list_lines) * 100 / state->count;
  
  wattron(w, state->status_attr);
  if (help_text != NULL)
    snprintf(output, app->display_width + 1, " %3i %% | %i lines | %s%s",
        percent, state->count, help_text, fill);
  else
    snprintf(output, app->display_width + 1, " %3i %% | %i lines%s",
        percent, state->count, fill);
  mvwaddstr(w, app->display_height - 1, 0, output);
  wattroff(w, state->status_attr);
}

static void find_next_line(char * data, size_t size, size_t * pos);
static void find_previous_line(char * data, size_t size, size_t * pos);

static void etpan_viewer_update_view(struct etpan_subapp * app,
    struct etpan_viewer_common_app_state * state)
{
  int diff;
  unsigned int old_top;
  int i;
  unsigned int list_lines;
  
  list_lines = app->display_height - 1;
  
  old_top = state->top;
  
#if 0
  if (state->col < 0)
    state->col = 0;
#endif
  if (state->col < state->left)
    state->left = state->col;

  if (state->col > state->left + app->display_width - 1)
    state->left = state->col - (app->display_width - 1);

#if 0
  if (state->line < 0)
    state->line = 0;
#endif
  if (state->line > state->count - 1)
    state->line = state->count - 1;

  if (state->line > state->top + list_lines - 1)
    state->top = state->line - (list_lines - 1);
  
  if (state->line < state->top)
    state->top = state->line;
  
  diff = (int) state->top - (int) old_top;
  
  if (diff > 0) {
    for (i = 0 ; i < diff ; i ++)
      find_next_line(state->text, state->size, &state->current_offset);
  }
  else if (diff != 0) {
    for (i = 0 ; i < -diff ; i ++)
      find_previous_line(state->text, state->size, &state->current_offset);
  }
}


int etpan_viewer_common_display(struct etpan_subapp * app,
    struct etpan_viewer_common_app_state * state, WINDOW * w,
    char * help_text)
{
  char * output;
  char * fill;
  char * data;
  size_t size;
  
  output = app->app->output;
  fill = app->app->fill;
  
  etpan_viewer_update_view(app, state);

  data = state->text + state->current_offset;
  size = state->size - state->current_offset;
  
  draw_text(app, state,
      w, output, fill,
      data, size, state->left, state->rot13_enabled,
      help_text);
  
  if (state->cursor)
    wmove(w, state->line - state->top + 1,
        state->col - state->left);
  else
    wmove(w, app->display_height - 1, 0);
  
  return NO_ERROR;
}

static void find_next_line(char * data, size_t size, size_t * pos)
{
  size_t current;

  current = * pos;

  while (current < size) {
    if (* (data + current) == '\n') {
      current ++;
      * pos = current;
      return;
    }

    current ++;
  }
}

static void find_previous_line(char * data, size_t size, size_t * pos)
{
  size_t current;

  current = * pos;

  if (current <= 0)
    return;

  current --;

  while (current > 0) {
    current --;

    if (* (data + current) == '\n') {
      current ++;
      break;
    }
  }

  * pos = current;
}

static unsigned int lines_count(char * data, size_t size)
{
  char * cur;
  unsigned int count;

  count = 1;

  cur = data;
  while (size > 0) {
    if (* cur == '\n')
      count ++;
    cur ++;
    size --;
  }

  return count;
}

int
etpan_viewer_common_set_message(struct etpan_viewer_common_app_state * state,
    char * filename, FILE * f, char * text, size_t size,
    carray * attr, int free_flags)
{
  etpan_viewer_common_flush(state);
  
  if (filename != NULL) {
    state->filename = strdup(filename);
    if (state->filename == NULL)
      goto err;
  }
  else {
    state->filename = NULL;
  }
  state->free_flags = free_flags;
  state->f = f;
  
  state->text = text;
  state->size = size;
  state->attr = attr;
  
  state->count = lines_count(text, size);
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

void
etpan_viewer_common_flush(struct etpan_viewer_common_app_state * state)
{
  if (state->text == NULL)
    return;
  
  if ((state->free_flags & ETPAN_VIEWER_FREE_FLAGS_FREE) != 0) {
    if (state->text != NULL) {
      free(state->text);
      state->text = NULL;
      state->size = 0;
    }
  }
  if ((state->free_flags & ETPAN_VIEWER_FREE_FLAGS_UNMAP) != 0) {
    if (state->text != NULL) {
      munmap(state->text, state->size);
      state->text = NULL;
      state->size = 0;
    }
  }
  if ((state->free_flags & ETPAN_VIEWER_FREE_FLAGS_MMAPFREE) != 0) {
    if (state->text != NULL) {
      mmap_string_unref(state->text);
      state->text = NULL;
      state->size = 0;
    }
  }
  if ((state->free_flags & ETPAN_VIEWER_FREE_FLAGS_CLOSE) != 0) {
    if (state->f != NULL) {
      fclose(state->f);
      state->f = NULL;
    }
  }
  if ((state->free_flags & ETPAN_VIEWER_FREE_FLAGS_REMOVE) != 0) {
    if (state->filename != NULL)
      unlink(state->filename);
  }
  
  if (state->attr != NULL) {
    if (state->attr != NULL)
      etpan_text_prop_array_free(state->attr);
    state->attr = NULL;
  }
  if (state->filename != NULL) {
    free(state->filename);
    state->filename = NULL;
  }
  state->f = NULL;
  state->text = NULL;
  state->size = 0;
  
  state->free_flags = 0;
  
  state->count = 0;
  state->chosen = 0;
  state->col = 0;
  state->left = 0;
  state->line = 0;
  state->top = 0;
  state->current_offset = 0;
}


int etpan_viewer_common_set_file(struct etpan_viewer_common_app_state * state,
    char * filename, FILE * f, carray * attr, int free_flags)
{
  int fd;
  struct stat buf;
  char * text;
  size_t size;
  int res;
  int r;
  
  if (f != NULL)
    fflush(f);
  
  fd = fileno(f);
  
  r = fstat(fd, &buf);
  if (r < 0) {
    res = ERROR_FILE;
    goto err;
  }
  
  size = buf.st_size;
  
  text = mmap(NULL, size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (text == MAP_FAILED) {
    res = ERROR_FILE;
    goto err;
  }
  
  r = etpan_viewer_common_set_message(state,
      filename, f, text, size,
      attr, free_flags | ETPAN_VIEWER_FREE_FLAGS_UNMAP);
  if (r != NO_ERROR) {
    res = r;
    goto unmap;
  }
  
  /*
    we don't munmap() the file because the flag ETPAN_VIEWER_FREE_FLAGS_UNMAP
    is set and when the viewer will flush the data, the munmap()
    will be done
  */
  
  return NO_ERROR;
  
 unmap:
  munmap(text, size);
 err:
  return ERROR_FILE;
}

int
etpan_viewer_common_set_filename(struct etpan_viewer_common_app_state * state,
    char * filename, carray * attr, int free_flags)
{
  int res;
  int r;
  FILE * f;
  
  f = fopen(filename, "r");
  if (f == NULL) {
    res = ERROR_FILE;
    goto err;
  }
  
  r = etpan_viewer_common_set_file(state,
      filename, f, attr, free_flags | ETPAN_VIEWER_FREE_FLAGS_CLOSE);
  if (r != NO_ERROR) {
    res = r;
    goto close;
  }
  
  return NO_ERROR;
  
 close:
  fclose(f);
 err:
  return ERROR_FILE;
}
