/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-mime.c,v 1.9 2003/12/10 23:16:45 hoa Exp $
 */

#include "etpan-cfg-mime.h"

#include <fnmatch.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "etpan-errors.h"
#include "etpan-cfg-common.h"
#include "etpan-db-read.h"
#include "etpan-tools.h"
#include "etpan-imf-helper.h"
#include "etpan-mime-tools.h"

static struct etpan_mime_info * etpan_mime_info_new(char * type,
    char * type_match, int renderer_type, char * renderer,
    int viewer_type, char * viewer,
    char * name, char * encoding, char * ext)
{
  struct etpan_mime_info * mime_info;

  mime_info = malloc(sizeof(struct etpan_mime_info));
  if (mime_info == NULL)
    return NULL;

  mime_info->type = type;
  mime_info->type_match = type_match;
  mime_info->renderer_type = renderer_type;
  mime_info->renderer = renderer;
  mime_info->viewer_type = viewer_type;
  mime_info->viewer = viewer;
  mime_info->name = name;
  mime_info->encoding = encoding;
  mime_info->ext = ext;
  
  return mime_info;
}

static void etpan_mime_info_free(struct etpan_mime_info * mime)
{
  if (mime->type != NULL)
    free(mime->type);
  if (mime->type_match != NULL)
    free(mime->type_match);
  if (mime->renderer != NULL)
    free(mime->renderer);
  if (mime->viewer != NULL)
    free(mime->viewer);
  if (mime->name != NULL)
    free(mime->name);
  if (mime->encoding != NULL)
    free(mime->encoding);
  if (mime->ext != NULL)
    free(mime->ext);
  free(mime);
}

struct etpan_mime_config * etpan_mime_config_new(clist * list)
{
  struct etpan_mime_config * mime_config;

  mime_config = malloc(sizeof(struct etpan_mime_config));
  if (mime_config == NULL)
    return NULL;

  mime_config->list = list;

  return mime_config;
}

void etpan_mime_config_free(struct etpan_mime_config * config)
{
  clist_foreach(config->list, (clist_func) etpan_mime_info_free, NULL);
  clist_free(config->list);
  free(config);
}

static struct etpan_mime_info *
etpan_get_mime_by_content_type(struct etpan_mime_config * config,
    char * mime_type);


struct etpan_mime_info * etpan_get_mime_info(struct etpan_mime_config * config,
    struct mailmime * mime_part)
{
  struct mailmime_single_fields fields;
  char * default_filename;
  struct etpan_mime_info * info;
  char * mime_str;
  
  mime_str = etpan_get_content_type_str(mime_part->mm_content_type);
  if (mime_str == NULL)
    return NULL;
  
  info = etpan_get_mime_by_content_type(config, mime_str);
  free(mime_str);
  
  if (info != NULL)
    return info;
  
  mailmime_single_fields_init(&fields, mime_part->mm_mime_fields,
      mime_part->mm_content_type);
  
  default_filename = fields.fld_disposition_filename;
  if (default_filename == NULL)
    default_filename = fields.fld_content_name;

  if (default_filename == NULL)
    return NULL;

  info = etpan_get_mime_by_name(config, default_filename);
  
  return info;
}

static struct etpan_mime_info *
etpan_get_mime_by_content_type(struct etpan_mime_config * config,
    char * mime_type)
{
  clistiter * cur;

  for(cur = clist_begin(config->list) ; cur != NULL ; cur = cur->next) {
    struct etpan_mime_info * info;

    info = cur->data;

    if (info->type_match != NULL) {
      if (etpan_regexp_match(info->type_match, mime_type)) {
        return info;
      }
    }
  }

  return NULL;
}

struct etpan_mime_info *
etpan_get_mime_by_name(struct etpan_mime_config * config,
    char * name)
{
  clistiter * cur;
  
  for(cur = clist_begin(config->list) ; cur != NULL ; cur = cur->next) {
    struct etpan_mime_info * info;

    info = cur->data;

    if (info->name != NULL) {
      if (etpan_regexp_match(info->name, name)) {
        return info;
      }
    }
  }

  return NULL;
}



int etpan_mime_config_read(char * filename,
    struct etpan_mime_config ** result)
{
  int res;
  struct etpan_mime_config * mime_config;
  struct etpan_db * db;
  int r;
  clist * list;
  unsigned int i;

  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  list = clist_new();
  if (list == NULL) {
    res = ERROR_MEMORY;
    goto free_db;
  }

  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    chash * entry;
    struct etpan_mime_info * mime;
    char * type;
    char * type_match;
    char * viewer;
    char * renderer;
    int renderer_type;
    int viewer_type;
    char * name;
    char * encoding;
    char * ext;

    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    type = etpan_db_entry_get_value(entry, "type");
    type_match = etpan_db_entry_get_value(entry, "type-match");
    renderer = etpan_db_entry_get_value(entry, "renderer");
    viewer = etpan_db_entry_get_value(entry, "viewer");
    name = etpan_db_entry_get_value(entry, "name");
    encoding = etpan_db_entry_get_value(entry, "encoding");
    ext = etpan_db_entry_get_value(entry, "ext");

    if (name != NULL) {
      char * new_name;
      
      new_name = strdup(name);
      if (new_name == NULL) {
        res = ERROR_MEMORY;
        goto free_list;
      }
      name = new_name;
    }
    
    if (type != NULL) {
      char * new_type;
      
      new_type = strdup(type);
      if (new_type == NULL) {
        res = ERROR_MEMORY;
        if (name != NULL)
          free(name);
        goto free_list;
      }
      type = new_type;
    }

    if (type_match != NULL) {
      char * new_type_match;
      
      new_type_match = strdup(type_match);
      if (new_type_match == NULL) {
        res = ERROR_MEMORY;
        if (type != NULL)
          free(type);
        if (name != NULL)
          free(name);
        goto free_list;
      }
      type_match = new_type_match;
    }
    
    renderer_type = COMMAND_TYPE_PARAM;
    if (renderer != NULL) {
      char * new_renderer;
      
      if (renderer[0] == '|') {
        new_renderer = strdup(renderer + 1);
        renderer_type = COMMAND_TYPE_PIPE;
      }
      else {
        new_renderer = strdup(renderer);
        renderer_type = COMMAND_TYPE_PARAM;
      }
      if (new_renderer == NULL) {
        if (type != NULL)
          free(type);
        if (type_match != NULL)
          free(type_match);
        if (name != NULL)
          free(name);
        res = ERROR_MEMORY;
        goto free_list;
      }
      renderer = new_renderer;
    }

    viewer_type = COMMAND_TYPE_PARAM;
    if (viewer != NULL) {
      char * new_viewer;
      
      if (viewer[0] == '|') {
        new_viewer = strdup(viewer + 1);
        viewer_type = COMMAND_TYPE_PIPE;
      }
      else {
        new_viewer = strdup(viewer);
        viewer_type = COMMAND_TYPE_PARAM;
      }
      if (new_viewer == NULL) {
        if (renderer != NULL)
          free(renderer);
        if (type != NULL)
          free(type);
        if (type_match != NULL)
          free(type_match);
        if (name != NULL)
          free(name);
        res = ERROR_MEMORY;
        goto free_list;
      }
      viewer = new_viewer;
    }

    if (encoding != NULL) {
      char * new_encoding;
      
      new_encoding = strdup(encoding);
      if (new_encoding == NULL) {
        if (type != NULL)
          free(type);
        if (renderer != NULL)
          free(renderer);
        if (type != NULL)
          free(type);
        if (type_match != NULL)
          free(type_match);
        if (name != NULL)
          free(name);
        res = ERROR_MEMORY;
        goto free_list;
      }
      encoding = new_encoding;
    }

    if (ext != NULL) {
      char * new_ext;
      
      new_ext = strdup(ext);
      if (new_ext == NULL) {
        if (encoding != NULL)
          free(encoding);
        if (type != NULL)
          free(type);
        if (renderer != NULL)
          free(renderer);
        if (type != NULL)
          free(type);
        if (type_match != NULL)
          free(type_match);
        if (name != NULL)
          free(name);
        res = ERROR_MEMORY;
        goto free_list;
      }
      ext = new_ext;
    }
    
    mime = etpan_mime_info_new(type, type_match, renderer_type, renderer,
        viewer_type, viewer, name, encoding, ext);
    if (mime == NULL) {
      if (ext != NULL)
        free(ext);
      if (encoding != NULL)
        free(encoding);
      if (type != NULL)
        free(type);
      if (renderer != NULL)
        free(renderer);
      if (type != NULL)
        free(type);
      if (type_match != NULL)
        free(type_match);
      if (name != NULL)
        free(name);
      res = ERROR_MEMORY;
      goto free_list;
    }

    r = clist_append(list, mime);
    if (r < 0) {
      etpan_mime_info_free(mime);
      res = ERROR_MEMORY;
      goto free_list;
    }
  }
  
  mime_config = etpan_mime_config_new(list);
  if (mime_config == NULL) {
    res = ERROR_MEMORY;
    goto free_list;
  }

  * result = mime_config;

  etpan_db_free(db);

  return NO_ERROR;

 free_list:
  clist_foreach(list, (clist_func) etpan_mime_info_free, NULL);
  clist_free(list);
 free_db:
  etpan_db_free(db);
 err:
  return res;
}
