# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the writer class for writing an XML project file..
"""

import os
import time

from qt import qApp

from Utilities import escape
from XMLWriterBase import XMLWriterBase
from Config import projectFileFormatVersion

class ProjectWriter(XMLWriterBase):
    """
    Class implementing the writer class for writing an XML project file.
    """
    def __init__(self, file, projectName):
        """
        Constructor
        
        @param file open file (like) object for writing
        @param projectName name of the project (string)
        """
        XMLWriterBase.__init__(self, file)
        
        self.pdata = qApp.mainWidget().getProject().pdata
        self.name = projectName
        
    def writeXML(self):
        """
        Public method to write the XML to the file.
        """
        XMLWriterBase.writeXML(self)
        
        self._write('<!DOCTYPE Project SYSTEM "Project-%s.dtd">' % projectFileFormatVersion)
        
        # add some generation comments
        self._write("<!-- Project file for project %s -->" % self.name)
        self._write("<!-- Saved: %s -->" % time.strftime('%Y-%m-%d, %H:%M:%S'))
        self._write("<!-- Copyright (C) %s %s, %s -->" % \
                (time.strftime('%Y'), 
                 escape(self.pdata["AUTHOR"][0]), 
                 escape(self.pdata["EMAIL"][0])))
        
        # add the main tag
        self._write('<Project version="%s">' % projectFileFormatVersion)
        
        # do description
        if self.pdata["DESCRIPTION"]:
            self._write("  <Description>", 0)
            for d in self.pdata["DESCRIPTION"][:-1]:
                self._write(escape("%s" % d.strip()))
            self._write("%s</Description>" % escape(self.pdata["DESCRIPTION"][-1].strip()))
        
        #do version, author and email
        for key in ["VERSION", "AUTHOR", "EMAIL"]:
            element = key.capitalize()
            if self.pdata[key]:
                self._write("  <%s>%s</%s>" % (element, escape(self.pdata[key][0]), element))
            
        # do the sources
        self._write("  <Sources>")
        for name in self.pdata["SOURCES"]:
            self._write("    <Source>")
            parts = name.split(os.sep)
            for p in parts[:-1]:
                self._write("      <Dir>%s</Dir>" % p)
            self._write("      <Name>%s</Name>" % parts[-1])
            self._write("    </Source>")
        self._write("  </Sources>")
        
        # do the forms
        self._write("  <Forms>")
        for name in self.pdata["FORMS"]:
            self._write("    <Form>")
            parts = name.split(os.sep)
            for p in parts[:-1]:
                self._write("      <Dir>%s</Dir>" % p)
            self._write("      <Name>%s</Name>" % parts[-1])
            self._write("    </Form>")
        self._write("  </Forms>")
        
        # do the translations
        self._write("  <Translations>")
        for name in self.pdata["TRANSLATIONS"]:
            self._write("    <Translation>")
            parts = name.split(os.sep)
            for p in parts[:-1]:
                self._write("      <Dir>%s</Dir>" % p)
            self._write("      <Name>%s</Name>" % parts[-1])
            self._write("    </Translation>")
        self._write("  </Translations>")
        
        # do the interfaces (IDL)
        self._write("  <Interfaces>")
        for name in self.pdata["INTERFACES"]:
            self._write("    <Interface>")
            parts = name.split(os.sep)
            for p in parts[:-1]:
                self._write("      <Dir>%s</Dir>" % p)
            self._write("      <Name>%s</Name>" % parts[-1])
            self._write("    </Interface>")
        self._write("  </Interfaces>")
        
        # do the others
        self._write("  <Others>")
        for name in self.pdata["OTHERS"]:
            self._write("    <Other>")
            parts = name.split(os.sep)
            for p in parts[:-1]:
                self._write("      <Dir>%s</Dir>" % p)
            self._write("      <Name>%s</Name>" % parts[-1])
            self._write("    </Other>")
        self._write("  </Others>")
        
        # do the main script
        if self.pdata["MAINSCRIPT"]:
            self._write("  <MainScript>")
            parts = self.pdata["MAINSCRIPT"][0].split(os.sep)
            for p in parts[:-1]:
                self._write("    <Dir>%s</Dir>" % p)
            self._write("    <Name>%s</Name>" % parts[-1])
            self._write("  </MainScript>")
        
        # do the vcs stuff
        self._write("  <Vcs>")
        if self.pdata["VCS"]:
            self._write("    <VcsType>%s</VcsType>" % self.pdata["VCS"][0])
        if self.pdata["VCSOPTIONS"]:
            self._write("    <VcsOptions>%s</VcsOptions>" % \
                escape(self.pdata["VCSOPTIONS"][0]))
        if self.pdata["VCSOTHERDATA"]:
            self._write("    <VcsOtherData>%s</VcsOtherData>" % \
                escape(self.pdata["VCSOTHERDATA"][0]))
        self._write("  </Vcs>")
        
        # do the eric3-doc stuff
        if len(self.pdata["ERIC3DOCPARMS"]):
            self._write("  <Eric3Doc>")
            if self.pdata["ERIC3DOCPARMS"]:
                self._write("    <Eric3DocParams>%s</Eric3DocParams>" % \
                    escape(self.pdata["ERIC3DOCPARMS"][0]))
            self._write("  </Eric3Doc>")
        
        # do the eric3-api stuff
        if len(self.pdata["ERIC3APIPARMS"]):
            self._write("  <Eric3Api>")
            if self.pdata["ERIC3APIPARMS"]:
                self._write("    <Eric3ApiParams>%s</Eric3ApiParams>" % \
                    escape(self.pdata["ERIC3APIPARMS"][0]))
            self._write("  </Eric3Api>")
        
        # do the happydoc stuff
        if len(self.pdata["HAPPYDOCPARMS"]):
            self._write("  <HappyDoc>")
            if self.pdata["HAPPYDOCPARMS"]:
                self._write("    <HappyDocParams>%s</HappyDocParams>" % \
                    escape(self.pdata["HAPPYDOCPARMS"][0]))
            self._write("  </HappyDoc>")
        
        self._write("</Project>")
