/*
		       Endeavour Mark II API Context
 */

#ifndef EDVCONTEXT_H
#define EDVCONTEXT_H

#include <glib.h>
#include "cfg.h"
#include "edvdevices.h"
#include "edvmimetypes.h"

/*
 *	Endeavour Context:
 *
 *	All members are private, do not access members directly.
 *
 *	This structure is the fundimental "connection" to Endeavour,
 *	it is used by most functions that interact with Endeavour.
 */
typedef struct _edv_context_struct		edv_context_struct;
struct _edv_context_struct {

	/* Configuration list */
	cfg_item_struct	*cfg_list;

        /* Devices list */
        edv_device_struct	**device;
        gint			total_devices;

	/* MIME Types list */
	edv_mimetype_struct	**mimetype;
	gint			total_mimetypes;

	/* List of queued interprocess commands to be sent, these
	 * will be sent when EDVNotifyFlush() is called
	 */
	gchar		**queued_command;
	gint		total_queued_commands;

	/* Quick references */
	gchar		*recycled_index_file,
			*prog_file,
			*prog_full_path;

};
#define EDV_CONTEXT(p)	((edv_context_struct *)(p))


/*
 *	Creates a new Endeavour Context.
 *
 *	The returned pointer should be deallocated by passing it to
 *	EDVContextDelete().
 */
extern edv_context_struct *EDVContextNew(void);

/*
 *	Loads the configuration from the specified Endeavour
 *	configuration file.
 *
 *	If path is NULL then the default configuration file found in the
 *	user's home directory will be used:
 *
 *		$HOME/.endeavour2/endeavour2.ini
 *
 *	Additional configuration file loading and set up will also be
 *	performed during this call.
 *
 *	This should be called right after EDVContextNew() to ensure
 *	that the context ctx is set up properly before passing to any
 *	other function.
 */
extern void EDVContextLoadConfigurationFile(
	edv_context_struct *ctx,
	const gchar *path		/* Can be NULL for default file */
);

/*
 *	Returns the number of queued commands on the context ctx.
 */
extern gint EDVContextCommandsPending(edv_context_struct *ctx);

/*
 *	Appends the given cmd to the list of queued commands on the
 *	context ctx.
 *
 *	The command is queued and will not actually be sent until
 *	EDVContextFlush() is called.
 */
extern void EDVContextQueueCommand(
	edv_context_struct *ctx, const gchar *cmd
);

/*
 *	Processes and flushes all pending operations and resources on
 *	the given context ctx.
 *
 *	All queued commands will be sent (regardless if Endeavour is
 *	running or not).
 *
 *	This call will not block/wait for Endeavour to acknowlage
 *	the request, for that use EDVContextWait();
 */
extern void EDVContextFlush(edv_context_struct *ctx);

/*
 *	Waits for a response from Endeavour (if it is running) to
 *	indicate that it has processed all the commands sent to it
 *	from a prior call to EDVContextFlush() (if any).
 *
 *	If Endeavour is not running then this call returns immediately.
 */
extern void EDVContextWait(edv_context_struct *ctx);

/*
 *	Macro to flush all pending operations and wait for them to
 *	be completed.
 */
#define EDVContextSync(c)	\
{				\
 EDVContextFlush(c);		\
 EDVContextWait(c);		\
}

/*
 *	Deletes the Endeavour Context and all it's resources.
 *
 *	Any pending operations will not be performed and will be
 *	discarded.
 */
extern void EDVContextDelete(edv_context_struct *ctx);


#endif	/* EDVCONTEXT_H */
