/* This file is part of EdiTeX, an editor of mathematical
 * expressions based on TeX syntax.
 * 
 * Copyright (C) 2002-2003 Luca Padovani <lpadovan@cs.unibo.it>,
 *                    2003 Paolo Marinelli <pmarinel@cs.unibo.it>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, please visit the project's home page
 * http://helm.cs.unibo.it/editex/
 * or send an email to <lpadovan@cs.unibo.it>
 */

#include <cassert>

#include "dom.hh"
#include "TPushParser.hh"
#include "timer.hh"

#include "ILPushLexer.hh"
#include "TDictionary.hh"
#include "CLoggerConsole.hh"
#include "CMathMLFactoryXSLT.hh"
#include "CMathMLFactoryXSLTDiff.hh"
#include "AMathMLConsumer.hh"

#include "guiGTK.h"

extern void *parseMathMLFile(char *);

struct Context
{
  /*
  Context(const std::string& s, TPushLexer& l, TPushParser& p) : buffer(s), i(0), lexer(l), parser(p) { };
  */
  Context(const std::string& s, APushLexer& l, TPushParser& p, DOMX::XSLTStylesheet& ts) 
    : buffer(s), i(0), lexer(l), parser(p), texStyle(ts) { };

  void send(void)
  {
    if (i < buffer.length()) lexer.push(buffer[i++]);
  }

  std::string buffer;
  unsigned i;
  APushLexer& lexer;
  TPushParser& parser;
  DOMX::XSLTStylesheet& texStyle;
};

extern "C" void
edit_output_tex(Context* data)
{
  assert(data);
  DOM::Document res = data->texStyle.apply(data->parser.document());
#if 0
  res.normalize();
  DOM::Node c = res.get_firstChild();
  if (c) std::cout << "HEY, there is a child! " << c.get_nodeName() << " " << c.get_nodeValue() << std::endl;
#endif
  //data->texStyle.save(res, stdout);
}

extern "C" int
edit_timeout(Context* data)
{
  assert(data);
  GUI_freeze();
  data->send();
  GUI_thaw();
  return 1;
}

extern "C" void
edit_push_char(Context* context, gchar ch)
{
  assert(context != NULL);
  long t0 = getTimer();
  GUI_freeze();
  std::cout << "*** SENDING " << ch << std::endl;
  context->lexer.push(ch);
  GUI_thaw();
  long t1 = getTimer();
  std::cout << "=== OVERALL TIME = " << (t1 - t0) / 1000 << std::endl;
}

#include <unistd.h>

extern "C" void
edit_push_string(Context* context, const gchar* s)
{
  assert(context != NULL);
  assert(s != NULL);
#if 0
//   GUI_freeze();
//   context->parser.freeze();
  for (unsigned i = 0; s[i]; i++)
    {
      GUI_freeze();
      context->lexer.push(s[i]);
      GUI_thaw();
      usleep(100000);
      usleep(100000);
    }
//   context->parser.thaw();
//   GUI_thaw();
#endif
  context->buffer = s;
}

extern "C" void
edit_drop(Context* context, gboolean alt, gboolean control)
{
  // At the moment, the last parameter is not used, but it will
  // be useful when we will handle the "fast" deletion
  assert(context != NULL);
  GUI_freeze();
  context->lexer.drop(alt);
  GUI_thaw();
}

extern "C" void
edit_reset_tex(Context* context)
{
  assert(context != NULL);
  GUI_freeze();
  context->lexer.reset();
  context->parser.reset();
  GUI_thaw();
}

extern "C" void
edit_complete(Context* context)
{
  assert(context != NULL);
  GUI_freeze();
  if (!context->lexer.complete()) context->lexer.push('\t');
  GUI_thaw();
}

int
main(int argc, char* argv[])
{
  CLoggerConsole logger;
  logger.verbosity(ALogger::Debug);

  TDictionary dictionary(logger);
  logger.info("loading the dictionary...");
  dictionary.load("./dict/dictionary-tex.xml");

  logger.info("loading the stylesheet...");
  DOM::DOMImplementation di;
  DOM::Document mmlStyleDoc = di.createDocumentFromURI("./xsl/tml-mmlp.xsl");
  DOMX::XSLTStylesheet mmlStyle(mmlStyleDoc);

  DOM::Document texStyleDoc = di.createDocumentFromURI("./xsl/tml-texid.xsl");
  DOMX::XSLTStylesheet texStyle(texStyleDoc);

  CMathMLFactoryXSLT factory(logger, mmlStyle);
  TPushParser parser(logger, factory, dictionary);
  ILPushLexer lexer(logger, parser, dictionary);

#if 0
  lexer.push('$');
  lexer.push(' ');
  assert(result);
#endif

#if 0
  DOM::Document doc = parser.document().document();
  std::vector< std::pair<DOM::GdomeString, DOM::GdomeString> > np;
  result = style.apply(doc, np);
  style.save(result, stdout);
#endif

  Context context("", lexer, parser, texStyle);

  GUI_init(&argc, &argv, "EditTeX", 500, 600, &context);
  GUI_load_document(gdome_cast_doc(static_cast<GdomeNode*>(factory.document())));
  GUI_run();
  GUI_uninit();
  GUI_unload_document();

}
