/* This file is part of EdiTeX, an editor of mathematical
 * expressions based on TeX syntax.
 * 
 * Copyright (C) 2002-2003 Luca Padovani <lpadovan@cs.unibo.it>,
 *                    2003 Paolo Marinelli <pmarinel@cs.unibo.it>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, please visit the project's home page
 * http://helm.cs.unibo.it/editex/
 * or send an email to <lpadovan@cs.unibo.it>
 */

#include <string>
#include <cctype>
#include <cassert>

#include "TToken.hh"
#include "TPushLexer.hh"
#include "APushParser.hh"

TPushLexer::TPushLexer(ALogger& l, APushParser& p) : APushLexer(l, p)
{
  state = ACCEPT;
}

void
TPushLexer::reset()
{
  buffer.erase();
  state = ACCEPT;

  displayCursor();
}

void
TPushLexer::flush()
{
  push(-1);
}

void
TPushLexer::transaction(char ch, State newState)
{
  switch (ch)
    {
    case '{': parser.push(TToken(TToken::BEGIN)); break;
    case '}': parser.push(TToken(TToken::END)); break;
    case '$': parser.push(TToken(TToken::SHIFT)); break;
    case '&': parser.push(TToken(TToken::ALIGN)); break;
    case '\n':
    case '\r': parser.push(TToken(TToken::EOL, ch)); break;
    case '^': parser.push(TToken(TToken::SUPERSCRIPT)); break;
    case '_': parser.push(TToken(TToken::SUBSCRIPT)); break;
    case '\t':
    case ' ': parser.push(TToken(TToken::IGNORABLE_SPACE, ch)); break;
    case '~': parser.push(TToken(TToken::ACTIVE, ch)); break;
    case '%': parser.push(TToken(TToken::COMMENT)); break;     
    default:
      if (isalpha(ch)) parser.push(TToken(TToken::LETTER, ch));
      else if (isdigit(ch)) parser.push(TToken(TToken::DIGIT, ch));
      else parser.push(TToken(TToken::OTHER, ch));
      break;
    }
  state = newState;
}

void
TPushLexer::push(char ch)
{
  switch (state)
    {
    case ACCEPT:
      if (ch == '\\') state = ESCAPE;
      else if (ch == '#') state = PARAMETER;
      else if (ch == -1) ;
      else transaction(ch, ACCEPT);
      break;
    case ESCAPE:
      if (isalpha(ch))
	{
	  buffer.push_back(ch);
	  state = MACRO;
	}
      else if (ch == -1) error();
      else
	{
	  parser.push(TToken(TToken::CONTROL, ch));
	  state = ACCEPT;
	}
      break;
    case MACRO:
      if (ch == '\\')
	{
	  parser.push(TToken(TToken::CONTROL, buffer));
	  buffer.erase();
	  state = ESCAPE;
	}
      else if (ch == '#')
	{
	  parser.push(TToken(TToken::CONTROL, buffer));
	  buffer.erase();
	  state = PARAMETER;
	}
      else if (isalpha(ch))
	buffer.push_back(ch);
      else if (ch == -1)
        {
	  parser.push(TToken(TToken::CONTROL, buffer));
	  buffer.erase();
	  state = ACCEPT;
	}
      else
	{
	  parser.push(TToken(TToken::CONTROL, buffer));
	  buffer.erase();
	  if (isspace(ch)) state = IGNORE_SPACE;
	  else transaction(ch, ACCEPT);
	}
      break;
    case IGNORE_SPACE:
      if (ch == '\\') state = ESCAPE;
      else if (ch == '#') state = PARAMETER;
      else if (isspace(ch)) ;
      else if (ch == -1) state = ACCEPT;
      else transaction(ch, ACCEPT);
      break;
    case PARAMETER:
      if (ch == -1) error();
      else
        {
	  parser.push(TToken(TToken::PARAMETER, ch));
	  state = ACCEPT;
	}
      break;
    default:
      assert(0);
      break;
    }

  displayCursor();

}

void
TPushLexer::drop(bool alt)
{
  std::string restore = "";

  switch (state)
    {
    case ACCEPT:
    case IGNORE_SPACE:
      restore = parser.drop(alt);
      if (restore.length() > 0 && restore[0] == '\\')
	{
	  buffer = std::string(restore, 1, restore.length() - 1);
	  state = (buffer.length() > 0) ? MACRO : ESCAPE;
	}
      break;
    case ESCAPE:
      state = ACCEPT;
      break;
    case MACRO:
      if (alt) buffer.erase();
      else buffer.erase(buffer.length() - 1, 1);
      if (buffer.length() == 0) state = ESCAPE;
      break;
    case PARAMETER:
    default:
      assert(0);
      break;
    }

  displayCursor();

}

void
TPushLexer::displayCursor()
{
  switch (state)
    {
    case ESCAPE: parser.setCursorHint("\\"); break;
    case MACRO: parser.setCursorHint("\\" + buffer); break;
    case PARAMETER: parser.setCursorHint("#"); break;
    default: parser.setCursorHint(""); break;
    }
}

bool
TPushLexer::error() const
{
  return false;
}
