/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002-2004 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <config.h>

#include <gnome.h>
#include <string.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs-cancellation.h>
#include <curl/curl.h>

#include "drivel.h"
#include "drivel_request.h"
#include "msg_queue.h"
#include "login.h"
#include "journal.h"
#include "md5.h"
#include "network.h"

extern GMutex *net_mutex;
static GAsyncQueue *net_queue;

static void
send_error (DrivelClient *dc, const gchar *header, const gchar *msg)
{
	MsgInfo *info;
	
	info = msg_info_new ();
	info->type = MSG_TYPE_ERROR;
	info->msg = g_strdup (msg);
	info->header = g_strdup (header);
	info->dc = (gpointer)dc;
	g_async_queue_push (dc->net->msg_queue, info);
	
	return;
}

/* returns the next line of text in 'data' */

static size_t
write_cb (void *ptr, size_t size, size_t nmemb, void *data)
{
	gint realsize = size * nmemb;
	DrivelRequestData *memory = data;

	memory->data = (gchar *) g_realloc (memory->data, memory->len + realsize + 1);

	memcpy (&(memory->data [memory->len]), ptr, realsize);
	memory->len += realsize;
	memory->data [memory->len] = '\0';

	return realsize;
}

static gchar*
get_line (const gchar *data, gint *offset)
{
	gchar *line;
	guint i, len;

	len = strlen (data);
	
	for (i = 0; i < len; i++)
	{
		if (data [i] == '\r' && data [i + 1] == '\n')
		{
			*offset = 2;
			break;
		}
		else if (data [i] == '\r' || data [i] == '\n')
		{
			*offset = 1;
			break;
		}
	}
	
	line = g_new0 (gchar, i + 1);
	memcpy (line, data, i);
	line [i] = '\0'; /* trash the newline */

	return line;
}

/* store the key/value pairs in 'string' to a hash table for later processing */

static void
parse_to_hash_table (DrivelClient *dc, DrivelRequest *dr, const gchar *string, gint len)
{
	guint offset, line_offset;
	gchar *key, *value;

	offset = 0;
	
	if (!g_utf8_validate (string, -1, NULL))
		g_warning ("It looks like this string is not valid UTF8!");
    
	drivel_request_clear_values (dr);
	
	while (offset < len)
	{
		key = get_line (string + offset, &line_offset);
		offset += strlen (key) + line_offset;
		if (line_offset > 1)
		{
			/* this is the http header, discard it */
			g_free (key);
			continue;
		}

		value = get_line (string + offset, &line_offset);
		offset += strlen (value) + line_offset;

		drivel_request_value_insert (dr, key, value);
		
		g_free (key);
		g_free (value);
	}
	
	return;
}

/* use the msg-queue to update the progress dialog */

static gint
request_progress (gpointer *data, double dltotal, double dlnow, double ultotal, 
		double ulnow)
{
	gboolean retval;
	gdouble percent;
	MsgInfo *info;
	DrivelClient *dc = (DrivelClient *) data;
	
	if (dltotal)
		percent = dlnow / dltotal;
	else if (ultotal)
		percent = ulnow / ultotal;
	else
		percent = 0.0;
	
	info = msg_info_new ();
	info->type = MSG_TYPE_UPDATE_PROGRESS_PERCENT;
	info->progress = percent;
	g_async_queue_push (dc->net->msg_queue, info);
	
	/* cancel this request and anything which is already queued */
	retval = gnome_vfs_cancellation_check (dc->net->cancel);
	if (retval)
	{
		DrivelRequest *dr;
		
		do
		{
			dr = g_async_queue_try_pop (net_queue);
			if (dr)
				drivel_request_free (dr);
		} while (dr);
		
		gnome_vfs_cancellation_ack (dc->net->cancel);
	}
	
	return retval;
}

static void
setup_proxies (DrivelClient *dc, CURL *session, gchar **proxy_userpwd, gchar **proxy_url)
{
	gchar *user, *pass, *url, *userpwd;
	gboolean use_proxy;
	gint port;
	
	g_mutex_lock (net_mutex);
	use_proxy = dc->proxy;
	if (dc->proxy_user)
		user = g_strdup (dc->proxy_user);
	else
		user = NULL;
	if (dc->proxy_pass)
		pass = g_strdup (dc->proxy_pass);
	else
		pass = NULL;
	userpwd = NULL;
	url = g_strdup (dc->proxy_url);
	port = dc->proxy_port;
	g_mutex_unlock (net_mutex);

	/* Proxy stuff */
	if (use_proxy)
	{
		if (user && user [0] != '\0' && pass && pass [0] != '\0')
		{
			userpwd = g_strdup_printf ("%s:%s", user, pass);	
			curl_easy_setopt (session, CURLOPT_PROXYUSERPWD, userpwd);
		}
		/* we listen to GConf for changes to the proxy URL and port, so these
		   variables should always be up to date */
		curl_easy_setopt (session, CURLOPT_PROXY, url);
		curl_easy_setopt (session, CURLOPT_PROXYPORT, port);
	}
	
	g_free (user);
	g_free (pass);
	
	if (userpwd)
		*proxy_userpwd = userpwd;
	else
		*proxy_userpwd = NULL;
	if (url)
		*proxy_url = url;
	else
		*proxy_url = NULL;

	return;
}

/* build the authentication response token for a livejournal server */

static gchar*
get_response_lj (const gchar *pass, const gchar *challenge)
{
	gchar *response;
	guchar hash[16 * 2 + 1];
	gint i;
	md5_state_t state;
	md5_byte_t digest[16];
	
	response = g_strdup_printf ("%s%s", challenge, pass);
	
	/* hash the response */
	md5_init (&state);
	md5_append (&state, response, strlen (response));
	md5_finish (&state, digest);

	for (i = 0; i < 16; i++)
		g_snprintf (hash + i * 2, 3, "%02x", digest [i]);
	hash [16 * 2] = '\0';
	
	g_free (response);
	
	return g_strdup (hash);
}

/* get a authentication challenge token from a livejournal server */
static gint
get_challenge_lj (DrivelClient *dc, DrivelRequest *dr, const gchar *url, 
		gint fast_servers, gchar **challenge)
{
	CURL *session;
	CURLcode curl_code;
	DrivelRequestData *memory;
	gchar *cookie, *proxy_userpwd, *proxy_url;
	glong http_code;
	gint retval;
	
	memory = g_new (DrivelRequestData, 1);
	memory->len = 0;
	memory->data = NULL;
	proxy_userpwd = proxy_url = NULL;
	
	cookie = g_strdup_printf ("ljfastserver=%d", fast_servers);
	
	session = curl_easy_init ();
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, memory);
	curl_easy_setopt (session, CURLOPT_URL, url);
	curl_easy_setopt (session, CURLOPT_POSTFIELDS, "&mode=getchallenge");
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, FALSE);
	curl_easy_setopt (session, CURLOPT_PROGRESSFUNCTION, request_progress);
	curl_easy_setopt (session, CURLOPT_PROGRESSDATA, dc);
	curl_easy_setopt (session, CURLOPT_COOKIE, cookie);
	curl_easy_setopt (session, CURLOPT_SHARE, dc->net->curl_share);
	setup_proxies(dc, session, &proxy_userpwd, &proxy_url);
	
	curl_code = curl_easy_perform (session);
	
	http_code = 0;
	curl_easy_getinfo (session, CURLINFO_HTTP_CODE, &http_code);
	
	if (http_code == 200 && curl_code != CURLE_ABORTED_BY_CALLBACK)
	{
		parse_to_hash_table (dc, dr, memory->data, memory->len);
		*challenge = g_strdup (drivel_request_value_lookup (dr, "challenge"));
		retval = 0;
	}
	else
	{
		*challenge = NULL;
		if (curl_code == CURLE_ABORTED_BY_CALLBACK)
			retval = -2;
		else
		{
			g_warning ("Could not get a challenge token from the server.");
			retval = -1;
		}
	}
	
	g_free (proxy_userpwd);
	g_free (proxy_url);
	
	return retval;
}

/* concat the items of 'dr' together in HTTP/POST format */

static gchar*
build_post_request (DrivelRequest *dr)
{
	gchar *string;
	gboolean valid;
	
	string = g_strdup ("");
	valid = drivel_request_start (dr);
	while (valid)
	{
		DrivelRequestItem *item;
		gchar *temp;

		item = drivel_request_get_current_item (dr);
		temp = g_strdup (string);
		g_free (string);
		string = g_strdup_printf ("%s&%s=%s", temp, item->key, item->value);
				
		valid = drivel_request_next (dr);
	}
	
	return string;
}

/* perform the HTTP/POST operation */
static gint
perform_post (DrivelClient *dc, DrivelRequest *dr, const gchar *url, 
		const gchar *request_string, const gchar *cookies)
{
	CURL *session;
	CURLcode curl_code;
	DrivelRequestData *memory;
	gchar *proxy_userpwd, *proxy_url;
	glong http_code;
	gint retval;
	
	memory = g_new (DrivelRequestData, 1);
	memory->len = 0;
	memory->data = NULL;
	proxy_userpwd = proxy_url = NULL;
	
	session = curl_easy_init ();
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, memory);
	curl_easy_setopt (session, CURLOPT_URL, url);
	curl_easy_setopt (session, CURLOPT_POSTFIELDS, request_string);
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, FALSE);
	curl_easy_setopt (session, CURLOPT_PROGRESSFUNCTION, request_progress);
	curl_easy_setopt (session, CURLOPT_PROGRESSDATA, dc);
	curl_easy_setopt (session, CURLOPT_COOKIE, cookies);
	curl_easy_setopt (session, CURLOPT_SHARE, dc->net->curl_share);
	setup_proxies(dc, session, &proxy_userpwd, &proxy_url);
	
	curl_code = curl_easy_perform (session);
	
	http_code = 0;
	curl_easy_getinfo (session, CURLINFO_HTTP_CODE, &http_code);
	
	if (http_code == 200 && curl_code != CURLE_ABORTED_BY_CALLBACK)
	{
		parse_to_hash_table (dc, dr, memory->data, memory->len);
		retval = 0;
	}
	else
	{
		if (curl_code == CURLE_ABORTED_BY_CALLBACK)
			retval = -2;
		else
		{
			retval = -1;
			g_warning ("HTTP POST request failed.");
		}
	}
	
	g_free (memory->data);
	g_free (memory);
	g_free (proxy_userpwd);
	g_free (proxy_url);
	
	return retval;
}

/* perform the HTTP/POST operation */
static gint
perform_get (DrivelClient *dc, DrivelRequest *dr, const gchar *url, 
		const gchar *cookies)
{
	CURL *session;
	CURLcode curl_code;
	DrivelRequestData *memory;
	gchar *proxy_userpwd, *proxy_url;
	glong http_code;
	gint retval;
	
	memory = g_new (DrivelRequestData, 1);
	memory->len = 0;
	memory->data = NULL;
	proxy_userpwd = proxy_url = NULL;
	
	session = curl_easy_init ();
	curl_easy_setopt (session, CURLOPT_WRITEFUNCTION, write_cb);
	curl_easy_setopt (session, CURLOPT_WRITEDATA, memory);
	curl_easy_setopt (session, CURLOPT_URL, url);
	curl_easy_setopt (session, CURLOPT_NOPROGRESS, FALSE);
	curl_easy_setopt (session, CURLOPT_PROGRESSFUNCTION, request_progress);
	curl_easy_setopt (session, CURLOPT_PROGRESSDATA, dc);
	curl_easy_setopt (session, CURLOPT_COOKIE, cookies);
	curl_easy_setopt (session, CURLOPT_SHARE, dc->net->curl_share);
	setup_proxies(dc, session, &proxy_userpwd, &proxy_url);
	
	curl_code = curl_easy_perform (session);
	
	http_code = 0;
	curl_easy_getinfo (session, CURLINFO_HTTP_CODE, &http_code);
	
	if (http_code == 200 && curl_code != CURLE_ABORTED_BY_CALLBACK)
	{
		drivel_request_set_data (dr, memory);
		retval = 0;
	}
	else
	{
		g_free (memory->data);
		g_free (memory);
		
		if (curl_code == CURLE_ABORTED_BY_CALLBACK)
			retval = -2;
		else
		{
			retval = -1;
			g_warning ("HTTP GET request failed.");
		}
	}
	
	g_free (proxy_userpwd);
	g_free (proxy_url);
	
	return retval;
}

/* Update the network progress dialog to explain to the user what is happening */

static void
update_progress_msg (GAsyncQueue *queue, DrivelRequestType type)
{
	MsgInfo *info;
	gchar *msg;
	
	switch (type)
	{
		case REQUEST_TYPE_LOGIN:
		{
			msg = g_strdup ("Retrieving user information");
			break;
		}
		case REQUEST_TYPE_GETPICTURE:
		{
			msg = g_strdup ("Downloading user pictures");
			break;
		}
		case REQUEST_TYPE_POSTEVENT:
		{
			msg = g_strdup ("Posting journal entry");
			break;
		}
		case REQUEST_TYPE_EDITEVENT:
		{
			msg = g_strdup ("Updating journal entry");
			break;
		}
		case REQUEST_TYPE_GETEVENTS:
		{
			msg = g_strdup ("Retrieving journal entries");
			break;
		}
		case REQUEST_TYPE_GETDAYCOUNTS:
		{
			msg = g_strdup ("Retrieving journal history");
			break;
		}
		case REQUEST_TYPE_EDITFRIENDS:
		{
			msg = g_strdup ("Updating Friends list");
			break;
		}
		case REQUEST_TYPE_GETFRIENDS:
		{
			msg = g_strdup ("Retrieving Friends list");
			break;
		}
		default:
		{
			msg = g_strdup ("We're doing something, but I'm not sure what");
			break;
		}
	}
	
	info = msg_info_new ();
	info->type = MSG_TYPE_UPDATE_PROGRESS_LABEL;
	info->msg = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">"
			"Sending / Receiving</span>\n\n"
			"%s...", msg);
	g_async_queue_push (queue, info);
	
	g_free (msg);
	
	return;
}

/* this thread never dies, it just loops, waiting for input from 
   net_enqueue_request() */

static gpointer
net_thread (DrivelClient *dc)
{
	while (TRUE)
	{
		DrivelRequest *dr;
		DrivelRequestType type;
		MsgInfo *info;
		gint retval = 0;
		gchar *request_string = NULL, *cookies = NULL;
		
		dr = g_async_queue_pop (net_queue);
		
		type = drivel_request_get_type (dr);
		/* for everything except CHECKFRIENDS, we reparent the dialog to
		   ensure it is transient for the top window, and then update its
		   message. */
		if (type != REQUEST_TYPE_CHECKFRIENDS)
		{
			info = msg_info_new ();
			info->type = MSG_TYPE_REPARENT_DIALOG;
			info->dc = (gpointer) dc;
			g_async_queue_push (dc->net->msg_queue, info);
			
			update_progress_msg (dc->net->msg_queue, type);
		}
		
		switch (drivel_request_get_protocol (dr))
		{
			case REQUEST_PROTOCOL_POST:
			{
				gchar *url = NULL;
				
				switch (drivel_request_get_api (dr))
				{
					case BLOG_API_LJ:
					{
						gint fast_servers;
						gchar *challenge, *response, *password;
						
						g_mutex_lock (net_mutex);
						fast_servers = dc->net->fast_servers;
						url = g_strdup_printf ("http://%s/interface/flat", 
								dc->net->server);
						password = g_strdup (dc->password);
						g_mutex_unlock (net_mutex);
						
						cookies = g_strdup_printf ("ljfastserver=%d", fast_servers);
						retval = get_challenge_lj (dc, dr, url, fast_servers, &challenge);
						response = get_response_lj (password, challenge);
						drivel_request_add_items (dr,
								g_strdup ("auth_method"), g_strdup ("challenge"),
								g_strdup ("auth_challenge"), challenge,
								g_strdup ("auth_response"), response,
								NULL);
						
						g_free (password);
						
						break;
					}
					default:
					{
						g_warning ("net_thread: Unknown journal API.");
						break;
					}
				}
				
				request_string = build_post_request (dr);
				if (!retval)
				{
					retval = perform_post (dc, dr, url, request_string, cookies);
				}
				if (retval == -1)
					send_error (dc, _("Communication Error"),
							_("There was a problem sending information to the "
							"server.  Please try again later."));
				
				g_free (url);
				
				break;
			}
			case REQUEST_PROTOCOL_GET:
			{
				const gchar *url;
				
				switch (drivel_request_get_api (dr))
				{
					case BLOG_API_LJ:
					{
						gint fast_servers;
						
						if (drivel_request_item_lookup (dr, "fastservers"))
							fast_servers = 1;
						else
							fast_servers = 0;
						
						cookies = g_strdup_printf ("ljfastserver=%d", fast_servers);
						
						break;
					}
					default:
					{
						g_warning ("net_thread: Unknown journal API.");
						break;
					}
				}
				
				url = drivel_request_item_lookup (dr, "url");
				retval = perform_get (dc, dr, url, cookies);
				if (retval == -1)
					send_error (dc, _("Communication Error"),
							_("There was a problem receiving information from "
							"the server.  Please try again later."));
				
				break;
			}
			case REQUEST_PROTOCOL_XMLRPC:
			{
				g_warning ("XMLRPC hasn't been implemented yet.");
				retval = -1;
				break;
			}
			case REQUEST_PROTOCOL_NONE:
			{
				g_warning ("net_thread: No protocol was selected.");
				retval = -1;
				break;
			}
		}
		
		g_free (request_string);
		g_free (cookies);
		
		if (!retval)
		{
			info = msg_info_new ();
			info->type = MSG_TYPE_PROCESS_NET_REQUEST;
			info->dr = (gpointer) dr;
			info->dc = (gpointer) dc;
			g_async_queue_push (dc->net->msg_queue, info);
		}
		
		if (type != REQUEST_TYPE_CHECKFRIENDS)
		{
			g_usleep (500000);
			info = msg_info_new ();
			info->type = MSG_TYPE_DONE;
			g_async_queue_push (dc->net->msg_queue, info);
		}
	}
	
	return NULL;
}

/* add a network request to the queue for processing */
void
net_enqueue_request (DrivelClient *dc, DrivelRequest *dr)
{
	MsgInfo *info;
	DrivelRequestType type;
	
	/* first build the progress dialog */
	type = drivel_request_get_type (dr);
	if (type != REQUEST_TYPE_CHECKFRIENDS)
	{
		info = msg_info_new ();
		info->type = MSG_TYPE_BUILD_PROGRESS;
		info->msg = g_strdup ("<span weight=\"bold\" size=\"larger\">"
				"Sending / Receiving</span>\n\n"
				"Begining network transaction...");
		info->widget = drivel_get_current_window (dc->window_list);
		info->dc = (gpointer)dc;
		g_async_queue_push (dc->net->msg_queue, info);
	}
	
	/* then queue the network request */
	g_async_queue_push (net_queue, dr);
	
	return;
}

/* start the network thread */
gint
net_start_thread (DrivelClient *dc)
{
	static gboolean inited = FALSE;
	gint retval;
	
	/* we can only start one network thread */
	if (inited)
		return -1;
	
	net_queue = g_async_queue_new ();
	
	if (g_thread_create ((GThreadFunc) net_thread, dc, FALSE, NULL))
		retval = 0;
	else
		retval = -1;
	
	return retval;
}
