/*
 * png.c
 *
 * crafted - a pud editor for the freecraft project.
 * 
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <png.h>

#include "iolib.h"
#include "crafted.h"
#include "crafted_png.h"

typedef struct Palette
{
  guchar r, g, b;
} Palette;

/**
**	png read callback for CL-IO.
**
**	@param png_ptr	png struct pointer.
**	@param data	byte address to read to.
**	@param length	number of bytes to read.
*/
void CL_png_readfn(png_structp png_ptr,png_bytep data,png_size_t length)
{
  png_size_t check;

  check = (png_size_t)CLread((CLFile *)png_get_io_ptr(png_ptr), data, (size_t)length);
  if (check != length)
  {
    png_error(png_ptr, "Read Error");
  }
}

/**
**	Load a png graphic file.
**
**	FIXME: must support other formats than 8bit indexed
**
**	@param name	png filename to load.
**
**	@return		graphic object with loaded graphic, or NULL if failure.
*/
Graphic *LoadGraphicPNG(const gchar* name)
{
  Graphic      *graphic;
  Palette      *palette;
  CLFile       *fp;
  png_structp   png_ptr;
  png_infop     info_ptr;
  guchar      **lines;
  guchar       *data;
  guint         i;
  guchar       *rgb_buf;
  GdkGC        *gc;
  GdkColor      mask_pattern;

  /*
   * open + prepare
   */
  if (!(fp=CLopen(name)))
  {
    gchar *error_txt;

    error_txt = g_strdup_printf("Can't open file: %s", name);
    perror(error_txt);
    g_free(error_txt);
    return NULL;
  }

  png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (!png_ptr)
  {
    CLclose(fp);
    return NULL;
  }
  info_ptr = png_create_info_struct(png_ptr);
  if (!info_ptr)
  {
    png_destroy_read_struct(&png_ptr, NULL, NULL);
    CLclose(fp);
    return NULL;
  }
  if (setjmp(png_ptr->jmpbuf))
  {
    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
    CLclose(fp);
    return NULL;
  }
  /* png_init_io(png_ptr,fp); */
  png_set_read_fn(png_ptr, fp, CL_png_readfn);

  /*
   * Setup ready, read header info.
   */
  png_read_info(png_ptr, info_ptr);
  if (verbosity)
  {
    g_message("%s: width %ld, height %ld (%ld bytes)\n",
              name, info_ptr->width, info_ptr->height,
              info_ptr->width*info_ptr->height);
    g_message("%s: %s", name,
              png_get_valid(png_ptr, info_ptr, PNG_INFO_PLTE) ? "palette" : "");
    g_message(" %s", png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS) ? "transparent" : "");
    g_message(" depth %d\n", info_ptr->bit_depth);
  }

  /*
   * Setup translators:
   */
  palette = g_new(Palette, 256);

  if (info_ptr->color_type==PNG_COLOR_TYPE_PALETTE)
  {
    if (verbosity)
      g_message("Color palette\n");
    if (info_ptr->valid & PNG_INFO_PLTE)
    {
      if (verbosity)
        g_message(" palette %d\n", info_ptr->num_palette);
      if (info_ptr->num_palette>256)
      {
        abort();
      }
      for (i=0 ; i<info_ptr->num_palette ; ++i)
      {
        palette[i].r = info_ptr->palette[i].red;
        palette[i].g = info_ptr->palette[i].green;
        palette[i].b = info_ptr->palette[i].blue;
      }
      for ( ; i<256; ++i)
      {
        palette[i].r = palette[i].g = palette[i].b = 0;
      }
    }
  }

  if (info_ptr->bit_depth == 16)
  {
    png_set_strip_16(png_ptr);
  }
  if (info_ptr->bit_depth<8)
  {
    png_set_packing(png_ptr);
  }

  png_read_update_info(png_ptr,info_ptr);

  /*
   * Allocate and reserve memory.
   */
  graphic = g_new(Graphic, 1);
  graphic->width = info_ptr->width;
  graphic->height = info_ptr->height;
  if (info_ptr->width != info_ptr->rowbytes)
  {
    if (verbosity)
      g_message("width(%ld)!=rowbytes(%ld) in file:%s\n",
                info_ptr->width, info_ptr->rowbytes, name);
    abort();
  }

  lines = alloca(graphic->height * sizeof(*lines));
  if (!lines)
  {
    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
    CLclose(fp);
    return NULL;
  }
  data = g_malloc(graphic->height * graphic->width);
  if (!data)
  {
    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
    CLclose(fp);
    return NULL;
  }

  for (i=0; i < graphic->height ; ++i)
  { /* start of lines */
    lines[i] = data+i*graphic->width;
  }

  /*
   * Final read the image.
   */
  png_read_image(png_ptr, lines);
  png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
  CLclose(fp);
  /* Build the pixmap and mask */
  graphic->pixmap = gdk_pixmap_new(drawing_area->window,
                                   graphic->width,
                                   graphic->height, -1);
  graphic->mask = gdk_pixmap_new(drawing_area->window,
                                 graphic->width,
                                 graphic->height, 1);

  gc = gdk_gc_new (graphic->mask);
  mask_pattern.pixel = 0;
  gdk_gc_set_foreground(gc, &mask_pattern);
  gdk_draw_rectangle(graphic->mask, gc, TRUE, 0, 0, -1, -1);
  mask_pattern.pixel = 1;
  gdk_gc_set_foreground(gc, &mask_pattern);

  rgb_buf = g_new(guchar, 3 * graphic->width * graphic->height);
  for (i = 0 ; i<graphic->width * graphic->height ; i++)
  {
    rgb_buf[i*3+0] = palette[data[i]].r;
    rgb_buf[i*3+1] = palette[data[i]].g;
    rgb_buf[i*3+2] = palette[data[i]].b;
    if (data[i] != 255)
    {
      gdk_draw_point(graphic->mask, gc,
                     i % graphic->width,
                     i / graphic->width);
    }
  }
  gdk_draw_rgb_image(graphic->pixmap, drawing_area->style->white_gc,
                     0,0, graphic->width, graphic->height,
                     GDK_RGB_DITHER_NONE, rgb_buf,
                     3*graphic->width);
  gdk_gc_destroy(gc);
  g_free(rgb_buf);
  g_free(data);
  g_free(palette);
  return graphic;
}

Graphic **load_graphic_PNG_with_size(const gchar* name, guint width, guint height)
{
  Graphic     **graphic;
  Palette      *palette;
  CLFile       *fp;
  png_structp   png_ptr;
  png_infop     info_ptr;
  guchar      **lines;
  guchar       *data;
  guint         i, j, k;
  guchar       *rgb_buf;
  GdkGC        *gc;
  GdkColor      mask_pattern;
  Palette       player_palette[8][4] =
  {
    {
      { 164,   0,   0},  //  Player#0 Color 1
      { 124,   0,   0},  //  Player#0 Color 2
      {  92,   4,   0},  //  Player#0 Color 3
      {  68,   4,   0}   //  Player#0 Color 4
    },
    {
      {  12,  72, 204},  //  Player#1 Color 1
      {   4,  40, 160},  //  Player#1 Color 2
      {   0,  20, 116},  //  Player#1 Color 3
      {   0,   4,  76}   //  Player#1 Color 4
    },
    {
      {  44, 180, 148},  //  Player#2 Color 1
      {  20, 132,  92},  //  Player#2 Color 2
      {   4,  84,  44},  //  Player#2 Color 3
      {   0,  40,  12}   //  Player#2 Color 4
    },
    {
      { 152,  72, 176},  //  Player#3 Color 1
      { 116,  44, 132},  //  Player#3 Color 2
      {  80,  24,  88},  //  Player#3 Color 3
      {  44,   8,  44}   //  Player#3 Color 4
    },
    {
      { 248, 140,  20},  //  Player#4 Color 1
      { 200,  96,  16},  //  Player#4 Color 2
      { 152,  60,  16},  //  Player#4 Color 3
      { 108,  32,  12}   //  Player#4 Color 4
    },
    {
      {  40,  40,  60},  //  Player#5 Color 1
      {  28,  28,  44},  //  Player#5 Color 2
      {  20,  20,  32},  //  Player#5 Color 3
      {  12,  12,  20}   //  Player#5 Color 4
    },
    {
      { 224, 224, 224},  //  Player#6 Color 1
      { 152, 152, 180},  //  Player#6 Color 2
      {  84,  84, 128},  //  Player#6 Color 3
      {  36,  40,  76}   //  Player#6 Color 4
    },
    {
      { 104,  64,  20},  //  Player#7 Color 1
      { 124,  88,  32},  //  Player#7 Color 2
      { 144, 112,  48},  //  Player#7 Color 3
      {   0,   0,   0}   //  Player#7 Color 4
    }            
  };

  /*
   * open + prepare
   */
  if (!(fp=CLopen(name)))
  {
    gchar *error_txt;

    error_txt = g_strdup_printf("Can't open file: %s", name);
    perror(error_txt);
    g_free(error_txt);
    return NULL;
  }

  png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (!png_ptr)
  {
    CLclose(fp);
    return NULL;
  }
  info_ptr = png_create_info_struct(png_ptr);
  if (!info_ptr)
  {
    png_destroy_read_struct(&png_ptr, NULL, NULL);
    CLclose(fp);
    return NULL;
  }
  if (setjmp(png_ptr->jmpbuf))
  {
    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
    CLclose(fp);
    return NULL;
  }
  /* png_init_io(png_ptr,fp); */
  png_set_read_fn(png_ptr, fp, CL_png_readfn);

  /*
   * Setup ready, read header info.
   */
  png_read_info(png_ptr, info_ptr);
  if (verbosity)
  {
    g_message("%s: width %ld, height %ld (%ld bytes)\n",
              name, info_ptr->width, info_ptr->height,
              info_ptr->width*info_ptr->height);
    g_message("%s: %s", name,
              png_get_valid(png_ptr, info_ptr, PNG_INFO_PLTE) ? "palette" : "");
    g_message(" %s", png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS) ? "transparent" : "");
    g_message(" depth %d\n", info_ptr->bit_depth);
  }

  /*
   * Setup translators:
   */
  palette = g_new(Palette, 256);

  if (info_ptr->color_type==PNG_COLOR_TYPE_PALETTE)
  {
    if (verbosity)
      g_message("Color palette\n");
    if (info_ptr->valid & PNG_INFO_PLTE)
    {
      if (verbosity)
        g_message(" palette %d\n", info_ptr->num_palette);
      if (info_ptr->num_palette>256)
      {
        abort();
      }
      for (i=0 ; i<info_ptr->num_palette ; ++i)
      {
        palette[i].r = info_ptr->palette[i].red;
        palette[i].g = info_ptr->palette[i].green;
        palette[i].b = info_ptr->palette[i].blue;
      }
      for ( ; i<256; ++i)
      {
        palette[i].r = palette[i].g = palette[i].b = 0;
      }
    }
  }

  if (info_ptr->bit_depth == 16)
  {
    png_set_strip_16(png_ptr);
  }
  if (info_ptr->bit_depth<8)
  {
    png_set_packing(png_ptr);
  }

  png_read_update_info(png_ptr,info_ptr);

  /*
   * Allocate and reserve memory.
   */
  
  graphic = g_new(Graphic*, 8);
  for (k=0; k<8 ; k++)
  {
    graphic[k] = g_new(Graphic, 1);
    graphic[k]->width = info_ptr->width;
    graphic[k]->height = info_ptr->height;
  }
  if (info_ptr->width != info_ptr->rowbytes)
  {
    if (verbosity)
      g_message("width(%ld)!=rowbytes(%ld) in file:%s\n",
                info_ptr->width, info_ptr->rowbytes, name);
    abort();
  }

  lines = alloca(info_ptr->height * sizeof(*lines));
  if (!lines)
  {
    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
    CLclose(fp);
    return NULL;
  }
  data = g_malloc(info_ptr->height * info_ptr->width);
  if (!data)
  {
    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
    CLclose(fp);
    return NULL;
  }

  for (i=0; i < info_ptr->height ; ++i)
  { /* start of lines */
    lines[i] = data+i*info_ptr->width;
  }

  /*
   * Final read the image.
   */
  png_read_image(png_ptr, lines);
  png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
  CLclose(fp);
  /* Build the pixmap and mask */
  for (k=0 ; k<8 ; k++)
  {
    graphic[k]->pixmap = gdk_pixmap_new(drawing_area->window,
                                        width, height, -1);
    graphic[k]->mask = gdk_pixmap_new(drawing_area->window,
                                      width, height, 1);

    gc = gdk_gc_new(graphic[k]->mask);
    mask_pattern.pixel = 0;
    gdk_gc_set_foreground(gc, &mask_pattern);
    gdk_draw_rectangle(graphic[k]->mask, gc, TRUE, 0, 0, -1, -1);
    mask_pattern.pixel = 1;
    gdk_gc_set_foreground(gc, &mask_pattern);

    rgb_buf = g_new(guchar, 3 * width * height);
    for (j = 0 ; j < height ; j++)
    for (i = 0 ; i < width  ; i++)
    {
      switch (data[(j*graphic[k]->width+i)])
      {
        case 208:
          rgb_buf[(j*width+i)*3+0] = player_palette[k][0].r;
          rgb_buf[(j*width+i)*3+1] = player_palette[k][0].g;
          rgb_buf[(j*width+i)*3+2] = player_palette[k][0].b;
          break;

        case 209:
          rgb_buf[(j*width+i)*3+0] = player_palette[k][1].r;
          rgb_buf[(j*width+i)*3+1] = player_palette[k][1].g;
          rgb_buf[(j*width+i)*3+2] = player_palette[k][1].b;
          break;

        case 210:
          rgb_buf[(j*width+i)*3+0] = player_palette[k][2].r;
          rgb_buf[(j*width+i)*3+1] = player_palette[k][2].g;
          rgb_buf[(j*width+i)*3+2] = player_palette[k][2].b;
          break;

        case 211:
          rgb_buf[(j*width+i)*3+0] = player_palette[k][3].r;
          rgb_buf[(j*width+i)*3+1] = player_palette[k][3].g;
          rgb_buf[(j*width+i)*3+2] = player_palette[k][3].b;
          break;

        default:
          rgb_buf[(j*width+i)*3+0] = palette[data[(j*graphic[k]->width+i)]].r;
          rgb_buf[(j*width+i)*3+1] = palette[data[(j*graphic[k]->width+i)]].g;
          rgb_buf[(j*width+i)*3+2] = palette[data[(j*graphic[k]->width+i)]].b;
          break;
      }
      if (data[j*graphic[k]->width+i] != 255)
        gdk_draw_point(graphic[k]->mask, gc, i,j);
    }
    gdk_draw_rgb_image(graphic[k]->pixmap, drawing_area->style->white_gc,
                       0,0, width, height,
                       GDK_RGB_DITHER_NONE, rgb_buf, 3*width);
    gdk_gc_destroy(gc);
    g_free(rgb_buf);
  }
  g_free(data);
  g_free(palette);
  return graphic;
}
