(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

module M = Omom_messages
module T = Omom_templates
module C = Configwin
module V = Omom_variables
module GV = Omom_gui_variables

let rec rename_assoc name newname = function
  | (key, data) as hd :: tl ->
      if key = name then
	(newname, data) :: tl
      else
	hd :: rename_assoc name newname tl
  | [] -> []
      
class virtual box_template label (wnote : GPack.notebook) vars =
  let (vbox, boxes) = GV.build_var_list vars in
object(self)
  val vbox = vbox
  val mutable boxes = boxes
  val wnote = wnote
  val vars = vars
  val mutable w_label = (None : GMisc.label option)

  method set_label name =
    self#w_label#set_text name

  method set_w_label label =
    w_label <- Some label

  method w_label =
    match w_label with
      | Some w_label ->
	  w_label
      | None ->
	  failwith ("set_label: label uninitialized")
	  
  method add_to_boxes box =
    boxes <- boxes @ [box]

  method remove_to_boxes name =
    boxes <- List.remove_assoc name boxes

  method rename_box name newname =
    boxes <- rename_assoc name newname boxes

  method fields_name =
    List.map fst boxes

  method private value_list =
    List.map (fun (_, x) -> x#variable) boxes

  method template =
    T.new_template self#name self#filename self#value_list

  method virtual remove_template : bool
  method virtual add_field : unit
  method virtual remove_field : unit
  method virtual rename_field : unit
  method virtual edit_properties : unit

  method virtual name : string
  method virtual filename : string

  initializer
  let wl = GMisc.label ~text: label () in
    self#set_w_label wl;
    wnote#append_page
      ~tab_label: wl#coerce vbox#coerce
end

class ocaml_box_template label wnote vars =
object(self)
  inherit box_template label wnote vars as super

  method add_field =
    ()

  method remove_field =
    ()

  method remove_template =
    false

  method edit_properties =
    ()

  method rename_field =
    ()

  method name = "ocaml"
  method filename = "ocaml"
end

class virtual mutable_box_template label wnote vars =
object(self)
  inherit box_template label wnote vars as super

  method add_variable var =
    let ((name, box), expand) = GV.build_var var in
      vbox#pack ~expand box#coerce;
      self#add_to_boxes (name, box)
	
  method remove_variable name =
    let box = List.assoc name boxes in
      vbox#remove box#coerce;
      self#remove_to_boxes name

  method rename_variable name newname =
    let box = List.assoc name boxes in
      box#set_name newname;
      self#rename_box name newname
	
  method add_field =
    let s1 = ref None
    and s2 = ref (Some "string") in
    let p1 = C.string ~f:(fun s -> s1 := Some s) M.add_field_name  ""
    and p2 = C.combo ~f:(fun s -> s2 := Some s) ~editable:false
	       M.add_field_type ["string"; "list"] "string" in
      match C.simple_get M.add_field [p1; p2] with
	| Configwin.Return_ok ->
	    (match (!s1, !s2) with
	       | (Some name, Some typ) ->
		   self#add_variable
		   (V.new_empty_variable name (V.type_of_string typ)) 
	       | _ -> ())
	| _ -> ()

  method remove_field =
    let fields = self#fields_name in 
      if List.length fields = 0 then
	()
      else begin
	let first = List.nth fields 0 in
	let s1 = ref first in
	let p1 = C.combo ~f:(fun s -> s1 := s) ~editable:false
		   M.remove_field_name fields first in
	  match C.simple_get M.remove_field [p1] with
	    | Configwin.Return_ok ->
		self#remove_variable !s1
	    | _ -> prerr_endline "pouet"
      end

  method rename_field =
    let fields = self#fields_name in 
      if List.length fields = 0 then
	()
      else begin
	let first = List.nth fields 0 in
	let s1 = ref first
	and s2 = ref None in
	let p1 = C.combo ~f:(fun s -> s1 := s) ~editable:false
		   M.rename_field_name fields first
	and p2 = C.string ~f:(fun s -> s2 := Some s)
		   M.rename_field_newname  ""in
	  match C.simple_get M.rename_field [p1;p2] with
	    | Configwin.Return_ok ->
		(match (!s1, !s2) with
		   | (name, Some newname) ->
		      self#rename_variable name newname
		  | _ -> ())
	    | _ -> ()
      end
end

class common_box_template label wnote vars =
object(self)
  inherit mutable_box_template label wnote vars as super
    
  method remove_template =
    false

  method edit_properties =
    ()

  method name = "common"
  method filename = "common"
end
  
class named_box_template label wnote vars (file : string) =
object(self)
  inherit mutable_box_template label wnote vars as super

  val mutable name = label
  val mutable filename = file

  method name = name
  method filename = filename

  method set_name lb =
    name <- lb;
    self#set_label name

  method set_filename file =
    filename <- file

  method remove_template =
    wnote#remove_page (wnote#page_num vbox#coerce);
    true

  method edit_properties =
    let s_name = ref name
    and s_filename = ref filename in
    let p_name =
      C.string ~f:(fun s -> s_name := s) M.template_properties_name name
    and p_filename =
      C.string ~f:(fun s -> s_filename := s)
	M.template_properties_filename filename in
      match C.simple_get M.template_properties [p_name; p_filename] with
	| C.Return_ok ->
	    self#set_name !s_name;
	    self#set_filename !s_filename
	| _ ->
	    ()

end

class box_templates wnote data =
object(self)
  val mutable template_list = []
  val wnote = wnote

  method private add_to_template_list templates =
    template_list <- template_list @ templates

  method private remove_from_template_list name =  
    template_list <- 
    List.filter (fun x -> x#name <> name) template_list

  method private get_nth num =
    List.nth template_list num

  method remove_template () =
    let nth = self#get_nth (wnote#current_page) in
      if nth#remove_template then
	self#remove_from_template_list nth#name
	
  method add_template name filename =
    self#add_to_template_list
      [(new named_box_template name wnote [] filename :> box_template)]

  method add_field () =
    let nth = self#get_nth (wnote#current_page) in
      nth#add_field

  method remove_field () =
    let nth = self#get_nth (wnote#current_page) in
      nth#remove_field

  method rename_field () =
    let nth = self#get_nth (wnote#current_page) in
      nth#rename_field

  method edit_properties () =
    let nth = self#get_nth (wnote#current_page) in
      nth#edit_properties

(* bof *)
  method private update_data =
    let templates = ref [] in
    let update = function
      | ("ocaml", "ocaml", v) -> data#set_ocaml v
      |	("common", "common", v) -> data#set_common v
      | template -> templates := !templates @ [template] in
      List.iter (fun t -> update t#template) template_list;
      data#set_templates !templates

  method changed : bool =
    self#update_data;
    data#changed

  method save : unit =
    self#update_data;
    data#save

  initializer
  let ocaml =
    (new ocaml_box_template M.ocaml wnote data#get_ocaml :> box_template) in
  let common = 
    (new common_box_template M.common wnote data#get_common :> box_template) in
  let templates =
    List.map 
      (fun (name, filename, var_list) -> 
	 (new named_box_template name wnote var_list filename :> box_template))
      data#get_templates in 
    self#add_to_template_list (ocaml :: common :: templates)
end
