(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** The OCamlmake-o-matic plug-in. *)

open Cam_plug
open Cam_plug.Options

module D = Omom_data
module M = Omom_messages
module C = Configwin


let plugin_name = "omom-edit"
let omom_gen_bin = "omom-gen"
let omom_command = "omom"

(* editor *)

class omom_editor file =
  let w = GWindow.window 
	    ~title: Omom_messages.software 
	    ~width: 800
	    ~height: 400
	    () in
  let app = new Omom_app.app file.f_name in
  
  let _ = w#add app#box#coerce in
  let _ = app#init_window w in
  let _ = app#box#connect#destroy w#destroy in
    
  let _ = w#show () in

object (self)
  method close = app#box#destroy ()
  method reload = ()
  method save = app#save ()
  method changed = app#changed
end

(** Edit OCamlmake-o-matic files in the graphical editor. 
   @editor omom
*)
let new_omom_editor ?char f = 
  Some (new omom_editor f :> Cam_plug.editor_app)

let _ = Cam_plug.add_editor plugin_name new_omom_editor

(* generator *)
  
let get_command_string dir file template target =
  Printf.sprintf "cd %s; %s %s %s > %s" (Filename.quote dir) omom_gen_bin
    (Filename.basename file.f_name) template target

(* generic generator *)

let generic_generator_ask templates = 
  let first =  List.hd templates in
  let s_template = ref (Some first)
  and s_target = ref None in
  let p_template =
    C.combo ~editable:false ~f:(fun s -> s_template := Some s)
      M.template templates first 
  and p_target =
    C.string ~f:(fun s -> s_target := Some s)
      M.target "" in
    match C.simple_get M.software [p_template; p_target] with
    | C.Return_ok ->
	(
	 match (!s_template, !s_target) with
	 | (Some template, Some target) -> Some (template, target)
	 | _ -> None
	)
    | _ ->
	None

(** [omom template target] uses the selected OCamlmake-o-matic file
   to generate the given template in the given target file.
   [omom] does the same after having asked the template and the
   target file to the user.
   @command omom
*)
let generic_generator args =
    match (selected_dir (), selected_files ()) with
      | (Some dir, file :: _) ->
	  let data = new D.data file.f_name in
	    data#load;
	    (match List.map (fun (t, _, _) -> t) data#get_templates with
	       | [] -> ()
	       | templates ->
		   let generate_template templ target =
		     let com = get_command_string dir file templ target in
		     Cam_plug.eval ("#" ^ com) ();
		     Cam_plug.display_message
		       (Printf.sprintf "The file %s should have been generated"
			  (Filename.concat dir target))
		   in
		   match args with
		     [] | [_] ->
		       (
			match generic_generator_ask templates with
			| None -> ()
			| Some (template, target) ->
			    generate_template template target
		       )
		   | templ :: target :: _ ->
		       generate_template templ target
	    )
      | _ -> ()

let _ = add_command omom_command "" generic_generator
