
/*
 *  Diverse Bristol midi routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

//#define DEBUG

#include <string.h>

#include <bristolmidi.h>
#include "bristolmessages.h"

extern bristolMidiMain bmidi;

static void
buildOneMsg(unsigned char p1, unsigned char p2, int dev, bristolMidiMsg *msg)
{
#ifdef DEBUG
	printf("buildOneMsg(%x, %x, %i), comm %x\n", p1, p2, dev,
		bmidi.dev[dev].lastcommand);
#endif

	msg->timestamp = (time_t) 0;
	/*
	 * We have enough message capacity, put information into the message buffer
	 * and return.
	 */
	msg->command = bmidi.dev[dev].lastcommand;
	msg->channel = bmidi.dev[dev].lastchan;

	if (p1 != 0xff)
	{
		msg->params.key.key = p1;
		msg->params.key.velocity = p2;
	}
}

static int
parseCommand(unsigned char comm, int dev)
{
#ifdef int
	printf("parseCommand(%x, %i)\n", comm, dev);
#endif

	/*
	 * We have a new command, save any interesting device state infomation.
	 */
	bmidi.dev[dev].lastchan = comm & MIDI_CHAN_MASK;
	bmidi.dev[dev].lastcommand = comm & MIDI_COMMAND_MASK;

	/*
	 * Set up how many more bytes we need.
	 */
	switch (bmidi.dev[dev].lastcommand) {
		case MIDI_SYSTEM:
			if (comm == 0xf7) 
			{
				/*
				//printf("End Of Sysex\n");
				if (bmidi.dev[dev].sysex.count != sizeof(bristolMsg))
					printf("Was bad sysex message (wrong length)\n");
				else
					printf("Was right length message: %x\n",
						bmidi.dev[dev].sysex.count);
				*/
				bmidi.dev[dev].lastcommand = 0;
			}
			if (comm == 0xf8) 
			{
				//printf("SYSEX checks\n");
				bmidi.dev[dev].sysex.count = 0;
				return(1);
			}
			break;
		case MIDI_PROGRAM:
		case MIDI_CHAN_PRESS:
			bmidi.dev[dev].lastcommstate =
				BRISTOL_CHANSTATE_WAIT_1;
			break;
		case MIDI_NOTE_ON:
		case MIDI_NOTE_OFF:
		case MIDI_POLY_PRESS:
		case MIDI_PITCHWHEEL:
			bmidi.dev[dev].lastcommstate =
				BRISTOL_CHANSTATE_WAIT_2;
			break;
		default:
			break;
	}
	return(0);
}

bristolMidiRawToMsg(unsigned char *buff, int count, int index, int dev,
	bristolMidiMsg *msg)
{
	int parsed = 0;

#ifdef DEBUG
	printf("bristolMidiRawToMsg(%x, %i, %i, %i) [%x]\n",
		buff, count, index, dev, buff[index]);
#endif

	msg->command = -1;

	/*
	 * Attempt to parse a raw message buffer. If we can resolve a complete
	 * message then return the number we have parsed.
	 */
	if (count <= 0)
		return(0);

	/*
	 * Although we know that we have buffered data, we do not know if we are
	 * being given complete messages by the raw interface - it could be byte by
	 * byte, or chunks of a large sysex. Parse the data byte by byte, and see
	 * if we can put together complete messages.
	 *
	 * Check out our current command in operation on this device:
	 */
	while (parsed < count)
	{
		/*
		 * If this is a status byte, find out what we cn do with it. Otherwise
		 * look for data commands.
		if ((bmidi.dev[dev].lastcommand != MIDI_SYSTEM) &&
			(buff[index] & MIDI_STATUS_MASK))
		 */
		if (buff[index] & MIDI_STATUS_MASK)
		{
			parseCommand(buff[index], dev);
		} else {
			switch (bmidi.dev[dev].lastcommand)
			{
				/*
				 * Looking for one more byte. We have it, since we made it here.
				 */
				case MIDI_PROGRAM:
				case MIDI_CHAN_PRESS:
					buildOneMsg(buff[index], -1, dev, msg);
					return(parsed + 1);
				/*
				 * Looking for two more bytes, if there are there.
				 */
				case MIDI_NOTE_ON:
				case MIDI_CONTROL:
				case MIDI_NOTE_OFF:
				case MIDI_POLY_PRESS:
				case MIDI_PITCHWHEEL:
					/*
					 * if we do not have enough bytes, return.
					 */
					if ((count - parsed) < 2)
						return(0);

					/*
					 * Otherwise, go get the command, checking for buffer wrap.
					 * We also need to make sure that the next spare byte is not
					 * a status byte - ie, that we have not binned a byte in the
					 * midi cable.
					 */
					if ((index + 1) == BRISTOL_MIDI_BUFSIZE)
					{
						if (buff[0] & MIDI_STATUS_MASK) {
							break;
						} else
							buildOneMsg(buff[index], buff[0], dev, msg);
					} else {
						if (buff[index+1] & MIDI_STATUS_MASK) {
							break;
						} else
							buildOneMsg(buff[index], buff[index+1], dev, msg);
					}

					return(parsed + 2);
				case MIDI_SYSTEM:
					/*
					 * Sysex management requires we read bytes until we get
					 * a status byte, and it should minimally be bocking on 
					 * this device. Assume, since we have been sent this, that
					 * it is going to be a bristolMidiMsg, and look for the 
					 * first few bytes. If it is SLab we are OK, otherwise we
					 * have an issue, and should read to next status byte then
					 * dump the data.
					 *
					 * Looks bad? Turn the bm message into an array, and index
					 * it depending on which byte we have. Could do with some
					 * error handling!
					 */
					if (buff[index] == MIDI_EOS)
					{
						/*
						 * End of SYSEX, see what the message actually looks
						 * like.
						 */
						//printf("EOS\n");
					} else {
						((char *) &msg->params.bristol)[
							bmidi.dev[dev].sysex.count++] = buff[index];

						if (bmidi.dev[dev].sysex.count == 5)
						{
							/*
							 * If this is not a SLab message, then set the
							 * current state to unknown.
							 */
							if (strncmp(((char *) &msg->params.bristol), "SLab", 4) != 0)
							{
								//printf("unknown SYSEX type\n");
								bmidi.dev[dev].lastcommand = 0;
							}
							//else
							//	printf("SLab SYSEX type\n");
						}
						if (bmidi.dev[dev].sysex.count
							== sizeof(bristolMsg))
						{
							if (bmidi.dev[dev].sysex.count
								== msg->params.bristol.msgLen)
							{
								//printf("Received a SLab SYSEX message\n");
								buildOneMsg(0xff, 0xff, dev, msg);
								bmidi.dev[dev].lastcommand = 0;
							} else {
								//printf("unknown SYSEX message\n");
								bmidi.dev[dev].lastcommand = 0;
							}
						}
					}
					break;
				default:
					/*
					 * Looking for a recognised command:
					 */
					if (buff[index] & MIDI_STATUS_MASK)
						parseCommand(buff[index], dev);
					break;
			}
		}

		/*
		 * Look after our indices
		 */
		if ((index+=1) > BRISTOL_MIDI_BUFSIZE)
			index = 0;
		parsed++;
	}
	return(parsed);
}

