
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include "bristol.h"

extern audioMain audiomain;

bristolSound *
dropBristolOp(int op, bristolOP *palette[])
{
	bristolSound *newsound;

	if ((audiomain.debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG3)
		printf("dropBristolOp(%i, %x): %s\n",
			op, palette, palette[op]->specs->opname);

	/*
	 * Allocate the sound structure.
	 */
	newsound = (bristolSound *) bristolmalloc0(sizeof(bristolSound));

	if ((audiomain.debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG5)
		printf("allocate %i bytes to new soundop %x\n",
			sizeof(bristolSound), newsound);

	newsound->name = palette[op]->specs->opname;
	newsound->index = op;
	newsound->operate = palette[op]->operate;

	/*
	 * Give it a parameter table.
	 */
	newsound->param =
		(bristolOPParams *) bristolmalloc0(sizeof(bristolOPParams));

	/*
	 * Call a reset to clean/init any params.
	 */
	palette[op]->reset(palette[op], newsound->param);

	if ((audiomain.debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG5)
		printf("allocate %i bytes to params %x, %x\n", sizeof(bristolOPParams),
			newsound->param, newsound->operate);

	return(newsound);
}

/*
 * Initialising an algorithm requires minimally that the audio thread is 
 * dormant, and preferably also the midi thread. We build a voice template
 * configuration. It may be the responsiblity of the MIDI handler to init any
 * voices we use.
initSoundAlgo(int algo, int index, bristolSound *sound[], bristolOP *palette[])
 */
bristolSound *
initSoundAlgo(int algo, int index, Baudio *baudio, audioMain *audiomain,
bristolSound *sound[])
{
	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG3)
		printf("initSoundAlgo(%i, %i, %x, %x), %x\n",
			algo, index, baudio, audiomain, sound[index]);

	if (sound[index] != (bristolSound *) NULL)
	{
		if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG3)
			printf("freeing existing sound structure\n");

		freeSoundAlgo(baudio, index, sound[index]);
	}

	sound[index] = dropBristolOp(algo, audiomain->palette);
	sound[index]->next = -1;
	sound[index]->flags = BRISTOL_SOUND_START|BRISTOL_SOUND_END;

	printSoundList(baudio, sound);
}

freeSoundAlgo(Baudio *baudio, int algo, bristolSound *sound[])
{
	int i;

	if ((audiomain.debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG3)
		printf("freeSoundAlgo(%i, %x)\n", algo, sound);

	if (algo >= baudio->soundCount)
		return;

	if (sound == NULL)
		return;

	if (sound[algo] != (bristolSound *) NULL)
		freeSoundAlgo(baudio, algo + 1, sound);

	if (sound[algo] == NULL)
		return;

	/*
	 * See if we have assigned parameters, which is normal.
	 */
	if (sound[algo]->param != NULL)
	{
		int j;

		/*
		 * And then for each parameter, see if it has assigmed any local
		 * memory.
		 */
		for (j = 0; j < BRISTOL_PARAM_COUNT; j++)
		{
			if (sound[algo]->param->param[j].mem != NULL)
				bristolfree(sound[algo]->param->param[j].mem);
		}

		bristolfree(sound[algo]->param);
	}

	bristolfree(sound[algo]);
	sound[algo] = NULL;
}

printSoundList(Baudio *baudio, bristolSound *sound[])
{
	int i;
	bristolOPParams *params;

	for (i = 0; i < baudio->soundCount; i++)
	{
		if (sound[i] == NULL)
			return;

		if ((audiomain.debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG3)
		{
			printf("	name	%s\n", sound[i]->name);
			printf("	index	%i\n", sound[i]->index);
			printf("	op	%x\n", sound[i]->operate);
			printf("	param	%x\n", sound[i]->param);
			printf("	next	%x\n", sound[i]->next);
			printf("	flags	%x\n", sound[i]->flags);
		}
	}
}

