
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
//#define BRISTOL_DBG

/*
 * Need to have basic template for an operator. Will consist of
 *
 *	noiseinit()
 *	operate()
 *	reset()
 *	destroy()
 *
 *	destroy() is in the library.
 *
 * Operate will be called when all the inputs have been loaded, and the result
 * will be an output buffer written to the next operator.
 */

#include <stdlib.h>

#include "bristol.h"
#include "noise.h"

/*
 * The name of this operator, IO count, and IO names.
 */
#define OPNAME "Noise"
#define OPDESCRIPTION "Noise Generator"
#define PCOUNT 2
#define IOCOUNT 1

#define NOISESIZE 8192

#define NOISE_OUT_IND 0

float *whitenoise;
float *pinknoise;

/*
 * Reset any local memory information.
 */
static int destroy(bristolOP *operator)
{
#ifdef BRISTOL_DBG
	printf("reset(%x)\n", operator);
#endif

	/*
	 * Unmalloc anything we added to this structure
	 */
	bristolfree(whitenoise);
	bristolfree(pinknoise);
	bristolfree(operator->specs);

	/*
	 * Free any local memory. We should also free ourselves, since we did the
	 * initial allocation.
	 */
	cleanup(operator);
}

/*
 * Reset any local memory information.
 */
static int reset(bristolOP *operator, bristolOPParams *param)
{
#ifdef BRISTOL_DBG
	printf("noisereset(%x)\n", operator);
#endif
	param->param[0].float_val = 0.001;
	param->param[1].int_val = 0;
}

/*
 * Alter an internal parameter of an operator.
 */
static int param(bristolOP *operator, bristolOPParams *param,
	unsigned char index, float value)
{
#ifdef BRISTOL_DBG
	printf("noiseparam(%x, %x, %i, %i)\n", operator, param, index, value);
#endif

	if (index == 0)
		param->param[index].float_val = value * 0.0025;

	if (index == 1)
	{
		if (value == 0)
			param->param[index].int_val = 0;
		else
			param->param[index].int_val = 1;
	}

	return(0);
}

/*
 * Amplifier - takes input signal and mod signal, and mults them together.
 */
static int operate(register bristolOP *operator, bristolVoice *voice,
	bristolOPParams *param,
	void *lcl)
{
	bristolNOISElocal * local = lcl;
	register int count, nsi;
	register float *ob, *ns, gain;
	bristolNOISE *specs;

	specs = (bristolNOISE *) operator->specs;
	count = specs->spec.io[NOISE_OUT_IND].samplecount;

#ifdef BRISTOL_DBG
	printf("noise(%x, %x, %x)\n", operator, param, local);
#endif

	ob = specs->spec.io[NOISE_OUT_IND].buf;
	gain = param->param[0].float_val;

	if (param->param[1].int_val == 0)
		ns = whitenoise;
	else
		ns = pinknoise;
	nsi = local->coff;

	/*
	 * Go through each sample and amplify it, correcting gain back to zero.
	 */
	for (;count > 0; count-=8)
	{
		*ob++ += ns[nsi] * gain;
		if (++nsi >= NOISESIZE) nsi = 0;
		*ob++ += ns[nsi] * gain;
		if (++nsi >= NOISESIZE) nsi = 0;
		*ob++ += ns[nsi] * gain;
		if (++nsi >= NOISESIZE) nsi = 0;
		*ob++ += ns[nsi] * gain;
		if (++nsi >= NOISESIZE) nsi = 0;
		*ob++ += ns[nsi] * gain;
		if (++nsi >= NOISESIZE) nsi = 0;
		*ob++ += ns[nsi] * gain;
		if (++nsi >= NOISESIZE) nsi = 0;
		*ob++ += ns[nsi] * gain;
		if (++nsi >= NOISESIZE) nsi = 0;
		*ob++ += ns[nsi] * gain;
		if (++nsi >= NOISESIZE) nsi = 0;
	}

	local->coff = nsi;
}

/*
 * Setup any variables in our OP structure, in our IO structures, and malloc
 * any memory we need.
 */
bristolOP *
noiseinit(bristolOP **operator, int index, int samplerate, int samplecount)
{
	bristolNOISE *specs;
	int i;

#ifdef BRISTOL_DBG
	printf("noiseinit(%x(%x), %i, %i, %i)\n",
		operator, *operator, index, samplerate, samplecount);
#endif

	*operator = bristolOPinit(operator, index, samplecount);

	/*
	 * Then the local parameters specific to this operator. These will be
	 * the same for each operator, but must be inited in the local code.
	 */
	(*operator)->operate = operate;
	(*operator)->destroy = destroy;
	(*operator)->reset = reset;
	(*operator)->param = param;

	specs = (bristolNOISE *) bristolmalloc0(sizeof(bristolNOISE));
	(*operator)->specs = (bristolOPSpec *) specs;
	(*operator)->size = sizeof(bristolNOISE);

	whitenoise = (float *) bristolmalloc0(NOISESIZE * sizeof(float));
	pinknoise = (float *) bristolmalloc0(NOISESIZE * sizeof(float));

	for (i = 0; i < NOISESIZE; i++)
		pinknoise[i] = whitenoise[i] = (short) rand();
	filternoise(pinknoise, NOISESIZE);

	/*
	 * These are specific to this operator, and will need to be altered for
	 * each operator.
	 */
	specs->spec.opname = OPNAME;
	specs->spec.description = OPDESCRIPTION;
	specs->spec.pcount = PCOUNT;
	specs->spec.iocount = IOCOUNT;
	specs->spec.localsize = sizeof(bristolNOISElocal);

	specs->spec.param[0].pname = "gain";
	specs->spec.param[0].description = "output gain on signal";
	specs->spec.param[0].type = BRISTOL_FLOAT;
	specs->spec.param[0].low = 0;
	specs->spec.param[0].high = 1;
	specs->spec.param[0].flags = BRISTOL_ROTARY|BRISTOL_SLIDER;

	specs->spec.param[1].pname = "White/Pink";
	specs->spec.param[1].description = "white pink";
	specs->spec.param[1].type = BRISTOL_ENUM;
	specs->spec.param[1].low = 0;
	specs->spec.param[1].high = 1;
	specs->spec.param[1].flags = BRISTOL_BUTTON;

	/*
	 * Now fill in the dco IO specs.
	 */
	specs->spec.io[0].ioname = "output";
	specs->spec.io[0].description = "Noise Output Signal";
	specs->spec.io[0].samplerate = samplerate;
	specs->spec.io[0].samplecount = samplecount;
	specs->spec.io[0].flags = BRISTOL_AC|BRISTOL_OUTPUT;

	return(*operator);
}

filternoise(float *noise, int size)
{
	int i;
	float Bout = 0, BLim = 0.4;

	for (i = 0; i < size; i++)
		*noise++ = (Bout+=((*noise - Bout) * BLim));
}

