
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

//#define DEBUG

#include "bristol.h"

/*
 * It would make a lot of sense to do another search if this one fails - look
 * this time for KEYOFF, since if we have released a key then it is certainly
 * a preferable voice to use.
 */
bristolVoice *
findFreeVoice(audioMain *audiomain, bristolVoice *voice)
{
#ifdef DEBUG
	if ((audiomain->debuglevel & MIDI_DEBUG_MASK) >= MIDI_DEBUG2)
		printf("findFreeVoice(%x)\n", voice);
#endif

	if (voice == NULL)
		return(voice);

	if (voice->flags & BRISTOL_KEYDONE)
		return(voice);

	if (voice->next == NULL)
		return(voice);

	return(findFreeVoice(audiomain, voice->next));
}

/*
 * There will be a lot of processing involved with MIDI note on/off management,
 * so it has been pulled into a separate file.
 */

int midiNoteOff(audioMain *audiomain, bristolMidiMsg *msg)
{
#ifdef DEBUG
	if ((audiomain->debuglevel & MIDI_DEBUG_MASK) >= MIDI_DEBUG2)
		printf("midiNoteOff(%i, %i)\n",
			msg->params.key.key, msg->params.key.velocity);
#endif

	/*
	 * Will eventually support multitibral notes with multi note on/off
	 */
	domidiNoteOff(audiomain, msg);
}

domidiNoteOff(audioMain *audiomain, bristolMidiMsg *msg)
{
	bristolVoice *voice = audiomain->playlist;
	int key;

	/*
	 * If we get a note off, then find that note, and configure it for done.
	 */
	while (voice != NULL)
	{
		if ((voice->baudio == NULL)
			|| (voice->baudio->mixflags & BRISTOL_KEYHOLD))
		{
			voice = voice->next;
			continue;
		}

		if ((key = voice->key.key - voice->baudio->transpose) > 127)
			key = 127;
		else if ((key = voice->key.key - voice->baudio->transpose) < 0)
			key = 0;

		if ((key == msg->params.key.key)
			&& (voice->baudio->midichannel == msg->channel))
		{
			/*
			 * Do not put the note off velocity in the key velocity. It should
			 * be seen as a separate control parameter.
			 */
			voice->keyoff.velocity = msg->params.key.velocity;
			voice->flags |= BRISTOL_KEYOFF;
		}

		voice = voice->next;
	}
}

/*
 * Find a free voice, or take one off the current playlist.
 *
 * Need to converge in the correct locals list before we can assign a voice, 
 * and find an associated baudio structure. Consideration will have to be made
 * for the ability to do layering.
 */
int midiNoteOn(audioMain *audiomain, bristolMidiMsg *msg)
{
	Baudio *baudio = audiomain->audiolist;

#ifdef DEBUG
	if ((audiomain->debuglevel & MIDI_DEBUG_MASK) >= MIDI_DEBUG2)
		printf("midiNoteOn(%i, %i), %i\n",
			msg->params.key.key, msg->params.key.velocity, msg->channel);
#endif

	if (msg->params.key.velocity == 0)
		return(midiNoteOff(audiomain, msg));

	/*
	 * Find an associated baudio structure. We have to go through the
	 * baudio lists, and find the correct midi channel. Link up the locals
	 * list.
	 */
	while (baudio != (Baudio *) NULL)
	{
		if ((baudio->midichannel == msg->channel) &&
			(baudio->lowkey <= msg->params.key.key) &&
			(baudio->highkey >= msg->params.key.key))
			{
				int key;

				key = msg->params.key.key + baudio->transpose;
				if (key > 127)
					key = 127;
				else if (key < 0)
					key = 0;
				msg->params.key.key = key;
				/*
				 * We could avoid this, since we have the correct baudio 
				 * pointer.
				 */
				domidiNoteOn(audiomain, msg, baudio);
			}

		baudio = baudio->next;
	}
}

domidiNoteOn(audioMain *audiomain, bristolMidiMsg *msg, Baudio *baudio)
{
	bristolVoice *voice = audiomain->playlist;
	int seconded = 0, lastkey;

#ifdef DEBUG
	if ((audiomain->debuglevel & MIDI_DEBUG_MASK) >= MIDI_DEBUG2)
		printf("domidiNoteOn(%x, %x, %x)\n", audiomain, msg, baudio);
#endif
	/*
	 * We need two pointers, to the front and back of the playlist.
	 *
	 * Go through the playlist looking for voices which are flagged as
	 * BRISTOL_KEYDONE, and if none are found, take the tail of the list.
	 *
	 * findFreeVoice() should always return a voice pointer, although there is
	 * a possibility that the voice will be in use. To reduce re-use we are 
	 * going to first search ourselves, so see if this particular note is in
	 * use, and second it if we find one. This happens with long release times.
	 */
	while (voice != NULL)
	{
		/*
		 * See if this key is already active for the given bristolAudio. If so
		 * we are going to reactivate the key, not assign another voice.
		 * Alternatively, we could just turn this note off?
		 */
		if (((voice->key.key == msg->params.key.key)
			&& (voice->baudio != NULL)
			&& (voice->baudio->midichannel == msg->channel)
			&& (voice->baudio->sid == baudio->sid)
			&& ((voice->flags
				& (BRISTOL_KEYON|BRISTOL_KEYREON|BRISTOL_KEYDONE)) == 0))
			|| ((baudio->voicecount == 1) && (voice->baudio == baudio)))
		{
			seconded = 1;
			break;
		}
		voice = voice->next;
	}

	if ((voice == NULL)
		&& ((voice = findFreeVoice(audiomain, audiomain->playlist)) == NULL))
	{
#ifdef DEBUG
		if ((audiomain->debuglevel & MIDI_DEBUG_MASK) >= MIDI_DEBUG1)
			printf("No voices found. Is audiothread active?\n");
#endif
		return(0);
	}

	/*
	 * We need lastkey for some monophonic portamento
	 */
	lastkey = voice->key.key;

	if (!seconded) {
		voice->baudio = baudio;
		voice->locals = baudio->locals;

		/*
		 * Flag it for now as DONE, stops engine using it.
		 */
		if (voice->baudio->voicecount > 1)
		{
			voice->flags |= (BRISTOL_KEYDONE|BRISTOL_KEYOFF);
			/*
			 * Remove it from the list.
			 */
			if (voice->last != NULL)
				voice->last->next = voice->next;
			else
				audiomain->playlist = voice->next;

			if (voice->next != NULL)
				voice->next->last = voice->last;
			else
				audiomain->playlast = voice->last;
		}
	}

	voice->key.key = msg->params.key.key;

	/*
	 * Put some operational values into the voice structures.
	 */
	voice->key.velocity = msg->params.key.velocity;
	voice->velocity = ((float) voice->key.velocity) / 127;
	/*
	 * Stack up the last key structure for portamento. Glide starts at the
	 * current glide of the last played note. This will fail in multitimbral
	 * mode, since next key may be for a different voice!
	 * Also need to cater for single voice synths.
	 */
	if (voice->baudio->voicecount == 1)
		voice->lastkey = lastkey;
	else
		voice->lastkey = audiomain->playlist->key.key;
	voice->dFreq = voice->baudio->ctab[voice->key.key];
	if (voice->baudio->voicecount == 1)
		voice->cFreq = voice->baudio->ctab[lastkey];
	else
		voice->cFreq = audiomain->playlist->cFreq;
	if (voice->baudio->glide == 0)
		voice->cFreq = voice->dFreq;
	else
		voice->cFreqstep = (voice->dFreq - voice->cFreq)
			/ (voice->baudio->glide * audiomain->samplerate);

	if (!seconded) {
		/*
		 * Move the event to the head of the playlist.
		 */
		if (voice->baudio->voicecount > 1)
		{
			voice->next = audiomain->playlist;
			voice->last = NULL;
			if (audiomain->playlist != NULL)
				audiomain->playlist->last = voice;
			audiomain->playlist = voice;
			voice->flags |= BRISTOL_KEYON;
		} else if (voice->flags & (BRISTOL_KEYDONE|BRISTOL_KEYOFFING))
			voice->flags |= BRISTOL_KEYON;

		voice->flags &= ~(BRISTOL_KEYOFF|BRISTOL_KEYOFFING|BRISTOL_KEYDONE);
	} else {
		if (baudio->voicecount > 1)
			voice->flags |= BRISTOL_KEYREON;
		else {
			if (baudio->mixflags & BRISTOL_MULTITRIG)
			{
				voice->flags |= BRISTOL_KEYREON;
			} else if (voice->flags & (BRISTOL_KEYDONE|BRISTOL_KEYOFFING))
				voice->flags |= BRISTOL_KEYREON;
		}

		voice->flags &= ~(BRISTOL_KEYOFF|BRISTOL_KEYOFFING|BRISTOL_KEYDONE);
	}

#ifdef DEBUG
	if ((audiomain->debuglevel & MIDI_DEBUG_MASK) >= MIDI_DEBUG7)
		printPlayList(audiomain);
#endif
}

allNotesOff(audioMain *audiomain, int channel)
{
	bristolVoice *voice = audiomain->playlist;

	while (voice != NULL) 
	{
		if ((channel == -1)
			|| ((voice->baudio != NULL)
				&& (voice->baudio->midichannel == channel)))
		{
			voice->flags |= BRISTOL_KEYOFF;
		}
		voice = voice->next;
	}
}

printPlayList(audioMain *audiomain)
{
	bristolVoice *v = audiomain->playlist;

	printf("printPlayList(%x, %x)\n", audiomain->playlist, audiomain->playlast);

	while (v != NULL)
	{
		printf("%x, %i, %x (%x, %x)\n", v,
			v->key.key, v->flags, v->next, v->last);

		v = v->next;
	}
	printf("\n");
}

