
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef BRIGHTONINTERNALS_H
#define BRIGHTONINTERNALS_H

#include <stdio.h>

#include "brightondevflags.h"
#include "brightonevents.h"

/*
 * Some return codes.
 */
#define BRIGHTON_OK 0
#define BRIGHTON_XISSUE 1

#include "brighton.h"

/*
 * Shadow rendering
 */
#define BRIGHTON_STATIC 1
#define BRIGHTON_DYNAMIC 2
typedef struct BrightonShadow {
	unsigned int flags;
	brightonCoord *coords;
	int *mask;
	int minx, maxx, miny, maxy;
	int ccount;
} brightonShadow;

/*
 * palette flags
 */
#define BRIGHTON_CMAP_SIZE 4096
#define BRIGHTON_INACTIVE_COLOR 0x01
#define BRIGHTON_QR_QSIZE 4096
#define BRIGHTON_QR_COLORS 24
typedef struct BrightonPalette {
	unsigned int flags;
	int uses;
	unsigned short red, green, blue;
	unsigned long pixel;
	void *color;
	void *gc;
} brightonPalette;

/*
 * Flags are for preferred rendering.
 */
#define BRIGHTON_ANTIALIAS 0x01
typedef struct BrightonBitmap {
	unsigned int flags;
	struct BrightonBitmap *next, *last;
	int uses;
	char *name;
	int width, height, ncolors, ctabsize, istatic, ostatic;
	int *pixels;
	int *colormap;
} brightonBitmap;

#include "brightondev.h"

/*
 * Flags for brightonPanel.flags - LSW.
 */
#define BRIGHTON_ACTIVE 0x80000000
#define TOO_SMALL 0x02
#define BRIGHTON_GREYSCALE 0x04
#define BRIGHTON_DEV_ACTIVE 0x08
#define BRIGHTON_SET_SIZE 0x010
typedef struct BrightonPanel {
	unsigned int flags;
	struct BrightonPanel *next, *last;
	int x, y;
	unsigned int width, height, depth, border;
	brightonBitmap *image;
	brightonBitmap *surface;
	brightonBitmap *canvas; // surface overlaid with image
	brightonResource *resource;
	struct BrightonDevice **devices;
	brightonSize *locations;
	int activedev;
} brightonPanel;

#define BRIGHTON_DEFAULT_DISPLAY ":0.0"
#define BRIGHTON_NAMESIZE 64
typedef struct BrightonDisplay {
	unsigned int flags;
	struct BrightonDisplay *next, *last;
	brightonPalette *palette;
	void *display; // this type depends on underlying window system
	void *image; // this type depends on underlying window system
	struct brightonWindow *bwin;
	char name[BRIGHTON_NAMESIZE];
	int width, height, depth;
} brightonDisplay;

/*
 * Flags for brightonWindow.flags
 */
#define BRIGHTON_NO_DRAW 0x01
typedef struct BrightonWindow {
	unsigned int flags;
	struct BrightonWindow *next, *last;
	brightonDisplay *display;
	brightonBitmap *image;
	brightonBitmap *surface;
	brightonPanel *panels;
	brightonBitmap *bitmaps;
	brightonBitmap *canvas; // surface overlaid with image, scaled.
	brightonBitmap *dlayer; // device rendering layer.
	brightonBitmap *slayer; // shadow rendering layer.
	brightonBitmap *render; // output bitmap to screen
	void *win;
	void *gc;
	int x, y;
	unsigned int width, height, depth, border;
	float aspect;
	int cmap_size;
	int id;
	char *window_name;
	int lightX, lightY, lightH;
	float lightI;
	brightonRoutine callbacks[BLASTEvent];
	brightonIApp *app;
	brightonApp *template;
	brightonIResource *activepanel;
	brightonILocations *activedev;
} brightonWindow;

typedef int (*brightonDeviceAlgo)();

/*
 * Global structure for any given renderable object.
 */
#define BRIGHTON_DEV_INITED 0x01
typedef struct BrightonDevice {
	unsigned int flags;
	struct BrightonDevice *next, *last;
	int device; // device type. 0=rotary, etc.
	int index; // index of this instance
	int panel; // parent panel of this instance
	brightonWindow *bwin;
	brightonPanel *parent;
	int originx, originy;
	int x, y; // Location onto parent window.
	int width, height; // Target width and height
	brightonBitmap *imagebg; // background image, not rotated
	brightonBitmap *image; // Generic rotatatable
	brightonBitmap *image2; // Generic rotatatable
	brightonBitmap *imagec; // Not rotated. If null, render circle of fg
	int bg, fg; // background palette - usual not used. FG for top of pot.
	float value;
	float lastvalue;
	float position;
	float lastposition;
	float value2;
	float lastvalue2;
	float position2;
	float lastposition2;
	char text[64]; // displayable name for device.
	int varindex; // index into global variable table - device's tracking var.
	brightonDeviceAlgo destroy;
	brightonDeviceAlgo configure;
	brightonShadow shadow;
} brightonDevice;

#define BRIGHTON_DEFAULT_ICON 1

extern brightonDisplay *brightonOpenDisplay(char *);
extern brightonDisplay *brightonFindDisplay(brightonDisplay *, brightonDisplay *);
extern brightonWindow *brightonCreateWindow(brightonDisplay *, brightonApp *, int, int);
extern void brightonDestroyPanel(brightonPanel *);
extern void brightonEventLoop(brightonDisplay **);
extern void brightonInitDefHandlers(brightonWindow *);
extern int brightonFillBase(brightonWindow *, char *, char *);
extern void *brightonmalloc(int);
extern void brightonfree(void *);
extern brightonPalette *brightonInitColormap(brightonWindow *, int);
extern int brightonFindColor(brightonPalette *, int, unsigned short, unsigned short, unsigned short, float);
extern int brightonFindFreeColor(brightonPalette *, int);
extern int brightonGetGC(brightonWindow *, unsigned short, unsigned short, unsigned short);
extern int brightonGetGCByName(brightonWindow *, char *);
extern int brightonFreeGC(brightonWindow *, int);
extern brightonBitmap *brightonFreeBitmap(brightonWindow *, brightonBitmap *);
extern brightonBitmap *brightonCreateBitmap(brightonWindow *, int, int);
extern brightonBitmap *brightonReadImage(brightonWindow *, char *);
extern int brightonDestroyImage(brightonWindow *, brightonBitmap *);
extern brightonRender(brightonWindow *, brightonBitmap *, brightonBitmap *, int, int, int, int, int);
extern brightonTesselate(brightonWindow *, brightonBitmap *, brightonBitmap *, int, int, int, int, int);
extern brightonRotate(brightonWindow *, brightonBitmap *, brightonBitmap *, int, int, int, int, double);
extern brightonWorldChanged(brightonWindow *, int, int);
extern brightonDevice *brightonCreateDevice(brightonWindow *, int, int, int, char *);
extern int brightonDestroyDevice(brightonDevice *);
extern int brightonConfigureDevice(brightonDevice *, brightonEvent *);
extern int brightonDeviceEvent(brightonDevice *, brightonEvent *);
extern int brightonForwardEvent(brightonWindow *);
extern int brightonLocation(brightonWindow *, int, int, int, int, int);
extern int brightonXpmWrite(brightonWindow *, char *);

extern brightonILocations *brightonLocator(brightonWindow *, int, int);
extern brightonIResource *brightonPanelLocator(brightonWindow *, int, int);
extern brightonILocations *brightonDeviceLocator(brightonIResource *, int, int);

extern int brightonCreateInterface(brightonWindow *, brightonApp *);
extern int brightonDestroyInterface(brightonWindow *);

extern int brightonRenderShadow(brightonDevice *, int);
extern int brightonFinalRender(brightonWindow *, int, int, int, int);
extern int brightonDevUndraw(brightonWindow *, brightonBitmap *, int, int, int, int);

#endif /* BRIGHTONINTERNALS_H */

