/* packagetree.c -- store and manipulate the program's package tree structure
 * Written by Charles Briscoe-Smith; refer to the file LEGAL for details.
 */

/* Interface definition */

#include "parsetree.h"

/* Add a parsed compilation unit to the program */
extern void packagetree_add_compunit(struct typedecl *);

/* Call a function on each type in the program */
extern void packagetree_foralltypes(void (*)(struct typedecl *));

/* Ditto, but in topological sort order; supertypes before subtypes */
extern void packagetree_foralltypessuperfirst(void (*fun)(struct typedecl *));

/* Attempt to look up a type by fully-qualified name */
extern struct typedecl *packagetree_findtype(struct name *);

/* Decide whether SUB is a subclass of SUPER */
extern int issubclass(struct name *, struct name *);

/* Decide whether TYPE implements IFACE or, if IFACE is a class, is a
   subclass of IFACE */
extern int implements(struct name *, struct name *);


#ifndef SEEN_packagetree_h

/* Implementation */

#include <stdio.h>
#include <string.h>
#include "error.h"
#include "xmalloc.h"

enum package_member_type { Subpackage, Typedecl };

struct package_node {
	const char *name;
	enum package_member_type type;
	void *member;
	struct package_node *next;
};
typedef struct package_node *Node;

static struct package_node unnamed_package = { 0, Subpackage, 0, 0 };

static Node
lookup(struct name *n)
{
	Node p;

	if (n==0) return &unnamed_package;

	p=lookup(n->prefix);
	if (p==0 || p->type != Subpackage) return 0;
	p=p->member;

	while (p) {
		if (strcmp(n->id, p->name) == 0) {
			return p;
		} else {
			p=p->next;
		}
	}

	return 0;
}

static void
ensurepackage(struct name *n)
{
	Node p;

	p=lookup(n);
	if (p==0) {
		Node new;
		ensurepackage(n->prefix);
		new=(Node) xmalloc(sizeof(struct package_node));
		new->name=n->id;
		new->type=Subpackage;
		new->member=0;
		p=lookup(n->prefix);
		new->next=p->member;
		p->member=new;
	} else if (p->type == Subpackage) {
		return;
	} else {
		error("Type/package clash");
	}
}

void
packagetree_add_compunit(struct typedecl *t)
{
	Node p, new;

	while (t) {
		ensurepackage(t->name->prefix);
		p=lookup(t->name->prefix);
		new=(Node) xmalloc(sizeof(struct package_node));
		new->name=t->name->id;
		new->type=Typedecl;
		new->member=t;
		new->next=p->member;
		p->member=new;
		t=t->rest;
	}
}

static void
foralltypes(Node from, void (*fun)(struct typedecl *))
{
	from=from->member;
	while (from) {
		if (from->type==Subpackage) {
			foralltypes(from, fun);
		} else {
			fun(from->member);
		}
		from=from->next;
	}
}

void
packagetree_foralltypes(void (*fun)(struct typedecl *))
{
	foralltypes(&unnamed_package, fun);
}

/*
 * We often want to process types in topological sort order, supertypes
 * before subtypes.  Types cannot be 'protected', so we use that modifier
 * as a working flag while we do this.
 */

static void
clearprotectedbits(struct typedecl *t)
{
	t->modifiers &= ~ACC_protected;
}

static void (*curfun)(struct typedecl *);

static void
superfirsthelper(struct typedecl *t)
{
	struct name *iface;

	/* Only process each type once */
	if (t->modifiers & ACC_protected) {
		return;
	} else {
		t->modifiers |= ACC_protected;
	}

	/* First, make sure our supertype has been processed */
	if (t->super) {
		superfirsthelper(packagetree_findtype(t->super));
	}

	/* Then do all our superinterfaces */
	for (iface=t->implements; iface; iface=iface->rest) {
		superfirsthelper(packagetree_findtype(iface));
	}

	/* Then process this type */
	curfun(t);
}

void
packagetree_foralltypessuperfirst(void (*fun)(struct typedecl *))
{
	packagetree_foralltypes(clearprotectedbits);

	curfun=fun;
	packagetree_foralltypes(superfirsthelper);
}

struct typedecl *
packagetree_findtype(struct name *name)
{
	Node p;

	p=lookup(name);
	if (p==0 || p->type==Subpackage) {
		return 0;
	}
	return p->member;
}

int
issubclass(struct name *sub, struct name *super)
{
	struct typedecl *subt=packagetree_findtype(sub);

	if (subt==0)
		return 0;

	if (equalnames(sub, super))
		return 1;

	return issubclass(subt->super, super);
}

int
implements(struct name *sub, struct name *iface)
{
	struct typedecl *subt=packagetree_findtype(sub);
	struct name *iscan;

	if (subt==0)
		return 0;

	if (equalnames(sub, iface))
		return 1;

	for (iscan=subt->implements; iscan; iscan=iscan->rest) {
		if (implements(iscan, iface))
			return 1;
	}

	return implements(subt->super, iface);
}

#endif /* SEEN_packagetree_h */
