/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastien.laout@tuxfamily.org                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef ITEMEDIT_H
#define ITEMEDIT_H

#include <qobject.h>
#include <qwidget.h>
#include <kdialogbase.h>
#include <kaction.h>
#include <qiconset.h>

#include "focusedwidgets.h"
#include "variouswidgets.h"

class QRadioButton;
class KColorButton;
class KURLRequester;
class QLabel;
class QLineEdit;
class QCheckBox;
class KService;
class QKeyEvent;

class KToolBar;
class KIconButton;

class Item;
class IconSizeCombo;

class ColorIconFactory : public QIconFactory
{
  public:
	ColorIconFactory();
	~ColorIconFactory();
	QPixmap* createPixmap(const QIconSet &iconSet, QIconSet::Size size, QIconSet::Mode mode, QIconSet::State state);

	static ColorIconFactory *globalColorIconFactory;
};

class ColorIconSet : public QIconSet
{
  public:
	ColorIconSet(const QColor &color);
	~ColorIconSet();
//	QPixmap pixmap(Size size, Mode mode, State state = Off);
	QColor color() const;
	void setColor(const QColor &color);
  private:
	QColor m_color;
};

class ColorAction : public KAction
{
  Q_OBJECT
  public:
	ColorAction(const QColor &color, const QString &text, const KShortcut &cut,
	            KActionCollection *parent, const char *name);
	~ColorAction();
//	QIconSet iconSet(KIcon::Group, int size = 0) /*const*/;
  signals:
	void colorChanged(const QColor &color);
  private slots:
	void selectColor();
  private:
	QColor m_color;
};

/** A simple widget, that all Item content edit widget must inherit
  * for have various methods such as saveChanges() (must be reimplemented).
  * @author Sbastien Laot
  */
class ItemEditWidgetBase : public QWidget
{
  Q_OBJECT
  public:
	ItemEditWidgetBase(Item *item, QWidget *parent = 0, const char *name = 0, WFlags fl = 0);
	~ItemEditWidgetBase();
  public slots:
	virtual void saveChanges()  = 0;
	virtual void goFirstFocus() = 0;
  private:
	virtual void show();

  protected:
	Item *m_item;
	bool  m_isAlreadyFirstFocused;
};

/** The base class for content item editors.
  * It connect and manage the widgets and provide pointers to them.
  * Different editors then can be designed : a dialog that layout them
  * or an inline editor that place toolbar in a MainWindow, and editor in another place...
  * @author Sbastien Laot
  */
class ItemEditorBase : public QObject
{
  Q_OBJECT
  public:
	ItemEditorBase(Item *item);
	Item*     editedItem()   { return m_item;         }
	KToolBar* toolbar()      { return m_toolbar;      }
	QWidget*  editorWidget() { return m_editorWidget; }
	QWidget*  extraWidget()  { return m_extraWidget;  }
  public slots:
	virtual void saveChanges()  = 0;
	virtual void goFirstFocus() = 0;
  signals:
	void focusOut();
  protected:
	Item     *m_item;
	KToolBar *m_toolbar;
	QWidget  *m_editorWidget;
	QWidget  *m_extraWidget;
};

/** Editor to edit Text Item content.
  * @author Sbastien Laot
  */
class ItemTextEditor : public ItemEditorBase
{
  Q_OBJECT
  public:
	ItemTextEditor(Item *item, QWidget *toolbarParent, QWidget *widgetsParent, QKeyEvent *ke = 0L);
	~ItemTextEditor();
  public slots:
	void saveChanges();
	void goFirstFocus();
  protected slots:
	void changeFont(int fontType);
	void changeColor(const QColor &color);
	void slotFocusOut();
	void slotEscapePressed();
  private:
	FocusedTextEdit   *m_text;
	FocusedComboBox   *m_font;
	FocusedColorCombo *m_color;
};

/** The widget to edit Text Item content.
  * @author Sbastien Laot
  */
class ItemTextEditWidget : public ItemEditWidgetBase
{
  Q_OBJECT
  public:
	ItemTextEditWidget(Item *item, QWidget *parent = 0, const char *name = 0, WFlags fl = 0, QKeyEvent *ke = 0L);
	~ItemTextEditWidget();
  public slots:
	void saveChanges();
	void goFirstFocus();

  private:
	ItemTextEditor *m_editor;
};

/** Editor to edit Rich text Item content.
  * @author Sbastien Laot
  */
class ItemHtmlEditor : public ItemEditorBase
{
  Q_OBJECT
  public:
	ItemHtmlEditor(Item *item, QWidget *toolbarParent, QWidget *widgetsParent, QKeyEvent *ke = 0L);
	~ItemHtmlEditor();
  public slots:
	void saveChanges();
	void goFirstFocus();
  protected slots:
//	void slotShowSource(bool on);
	void cursorPositionChanged(int, int);
	void buttonToggled(int);
	void slotVerticalAlignmentChanged(QTextEdit::VerticalAlignment align);
	void slotFocusOut();
	void slotEscapePressed();
  private:
	FocusedTextEdit   *m_text;
	FocusedFontCombo  *m_font;
	FocusedColorCombo *m_color;
//	QCheckBox         *m_showHtmlSource;
	QWidget           *m_wid;

	// QTextView : The text edit may be able to provide some meta-information.
	//  For example, the documentTitle() function will return the text from within HTML <title> tags.
};

/** The widget to edit Html Item content.
  * FIXME: The same as ItemTextEditWidget : merge ?
  * @author Sbastien Laot
  */
class ItemHtmlEditWidget : public ItemEditWidgetBase
{
  Q_OBJECT
  public:
	ItemHtmlEditWidget(Item *item, QWidget *parent = 0, const char *name = 0, WFlags fl = 0, QKeyEvent *ke = 0L);
	~ItemHtmlEditWidget();
  public slots:
	void saveChanges();
	void goFirstFocus();

  private:
	ItemHtmlEditor *m_editor;
};

/** The widget to edit Link Item content.
  * @author Sbastien Laot
  */
class ItemLinkEditWidget : public ItemEditWidgetBase
{
  Q_OBJECT
  public:
	ItemLinkEditWidget(Item *item, QWidget *parent = 0, const char *name = 0, WFlags fl = 0, QKeyEvent *ke = 0L);
	~ItemLinkEditWidget();
  public slots:
	void saveChanges();
	void goFirstFocus();
  protected slots:
	void urlChanged(const QString&);
	void doNotAutoTitle(const QString&);
	void doNotAutoIcon(QString);
	void guessTitle();
	void guessIcon();
	void emptyIcon();

  private:
	bool           m_isAutoModified;
	KURLRequester *m_url;
	QLineEdit     *m_title;
	KIconButton   *m_icon;
	QPushButton   *m_autoTitle;
	QPushButton   *m_autoIcon;
};

/** The widget to edit Launcher Item content.
  * @author Sbastien Laot
  */
class ItemLauncherEditWidget : public ItemEditWidgetBase
{
  Q_OBJECT
  public:
	ItemLauncherEditWidget(Item *item, QWidget *parent = 0, const char *name = 0, WFlags fl = 0);
	~ItemLauncherEditWidget();
  public slots:
	void saveChanges();
	void goFirstFocus();
  protected slots:
	void guessIcon();
	void emptyIcon();

  private:
	bool                 m_isAutoModified;
	RunCommandRequester *m_command;
	QLineEdit           *m_name;
	KIconButton         *m_icon;
	KService            *m_service;
};

/** The widget to edit Image Item content.
  * @author Sbastien Laot
  */
class ItemImageEditWidget : public ItemEditWidgetBase
{
  Q_OBJECT
  public:
	ItemImageEditWidget(Item *item, QWidget *parent = 0, const char *name = 0, WFlags fl = 0);
	~ItemImageEditWidget();
  public slots:
	void saveChanges();
	void goFirstFocus();
  private:
	QPushButton *m_open;
};

/** The widget to edit Color Item content.
  * @author Sbastien Laot
  */
class ItemColorEditWidget : public ItemEditWidgetBase
{
  Q_OBJECT
  public:
	ItemColorEditWidget(Item *item, QWidget *parent = 0, const char *name = 0, WFlags fl = 0);
	~ItemColorEditWidget();
  public slots:
	void saveChanges();
	void goFirstFocus();

  private:
	KColorButton *m_color;
};

/** Widget to edit annotations of a Item, and check it.
  * @author Sbastien Laot
  */
class ItemEditWidget : public ItemEditWidgetBase
{
  Q_OBJECT
  public:
	ItemEditWidget(Item *item, QWidget *parent = 0, const char *name = 0, WFlags fl = 0);
	~ItemEditWidget();
  public slots:
	void saveChanges();
	void goFirstFocus();
	void slotUnmirror();

  private:
	KTextEdit   *m_annotations;
	QLabel      *m_fileLabel;
	QLineEdit   *m_fileName;
	QPushButton *m_unmirror;
};

/** Show a dialog with tabs to edit Item properties.
  * @author Sbastien Laot
  */
class ItemEditDialog : public KDialogBase//QTabDialog
{
  Q_OBJECT
  public:
	ItemEditDialog(Item *item, bool editAnnotations = false, QWidget *parent = 0L, QKeyEvent *ke = 0L);
	~ItemEditDialog();
  protected slots:
	void slotApply();
	void slotOk();

  private:
	Item               *m_item;
	ItemEditWidgetBase *m_editContent;
	ItemEditWidget     *m_itemEdit;
};

#endif // ITEMEDIT_H
