/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __PROGRESSDLG_H__
#define __PROGRESSDLG_H__

#include <qvaluelist.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qdialog.h>

#include <kapp.h>

#include "_progress.h"

/*========================================================*/

class ArsonProgress;
class KConfig;

class ArsonCtrlState
{
public:
	ArsonCtrlState (ArsonProgress *pd);

	ArsonCtrlState &operator>> (QCheckBox *pc);
	ArsonCtrlState &operator>> (QComboBox *pc);
	ArsonCtrlState &operator>> (QLineEdit *pe);

	ArsonCtrlState &operator<< (QCheckBox *pc);
	ArsonCtrlState &operator<< (QComboBox *pc);
	ArsonCtrlState &operator<< (QLineEdit *pe);

private:
	KConfig *m_pk;
};

/*========================================================*/

#define ARSON_CONNECT_PROGRESSCTRL(p,ctrl) \
	QObject::connect(ArsonProgress::findProgressWidget((p)), \
	SIGNAL(restoreState(ArsonCtrlState&)), \
	(ctrl), SLOT(onRestore(ArsonCtrlState&))); \
	QObject::connect(ArsonProgress::findProgressWidget((p)), \
	SIGNAL(saveState(ArsonCtrlState&)), \
	(ctrl), SLOT(onSave(ArsonCtrlState&)))

#define IMPLEMENT_PROGRESSCTRL_SLOTS(cls) \
	void cls::onRestore (ArsonCtrlState &state) { state >> this; } \
	void cls::onSave (ArsonCtrlState &state) { state << this; }

class ArsonProgressCheckbox : public QCheckBox
{
	Q_OBJECT
	
public:
	ArsonProgressCheckbox (const QString &label, QWidget *parent, const char *name = NULL);

private slots:
	void onRestore (ArsonCtrlState &state);
	void onSave (ArsonCtrlState &state);
};

class ArsonProgressCombobox : public QComboBox
{
	Q_OBJECT
	
public:
	ArsonProgressCombobox (QWidget *parent, const char *name = NULL);

private slots:
	void onRestore (ArsonCtrlState &state);
	void onSave (ArsonCtrlState &state);
};

class ArsonProgressLineedit : public QLineEdit
{
	Q_OBJECT

public:
	ArsonProgressLineedit (QWidget *parent, const char *name = NULL);

private slots:
	void onRestore (ArsonCtrlState &state);
	void onSave (ArsonCtrlState &state);
};

/*========================================================*/

class ArsonProcessOpts;
class ArsonProcessMgr;
class ArsonProcessUI;
class QBoxLayout;
class QGroupBox;
class pushButtonLogUI;
class QHBoxLayout;
class QLayoutItem;

class ArsonProgress : public ArsonProgressBase
{
	Q_OBJECT
	
	typedef QValueList<QWidget*> WIDGETLIST;
	
public:
	virtual ~ArsonProgress (void);

	void setWriter (ArsonProcessMgr *pWriter);
	ArsonProcessUI *ui (void);

	void output (const QString &str, bool error);

	struct WidgetMgr {
		WidgetMgr (ArsonProgress *pd);
		WidgetMgr &operator<< (QWidget *pw);
	private:
		ArsonProgress *m_pd;
	};

	struct LayoutRow {
		LayoutRow (ArsonProgress *pd, bool indent = false);
		LayoutRow &operator<< (QWidget *pw);
		LayoutRow &operator<< (QLayoutItem *pl);
		QHBoxLayout *layout (void) const { return m_row; }
	private:
		ArsonProgress *m_pd;
		QHBoxLayout *m_row;
	};

	QCheckBox *addCheckbox (const QString &text, const char *name, bool state = false);
	QHBoxLayout *makeLayoutRow (bool indent = false);
	bool isCheckboxChecked (const char *name);
	int addWidget (QWidget *widget, bool align = true);
	void setCaption (const QString &text);
	QBoxLayout *ctrlLayout (void);
	QWidget *ctrlParent (void);
	static QLayoutItem *spacer (int width = 6);

	LayoutRow layoutRow (bool indent = false) { return LayoutRow(this, indent); }
	WidgetMgr widgetMgr (void) { return WidgetMgr(this); }

	void setup (void);					///	Perform any post-construction setup.
	virtual bool cancel (void);			///	Returns true if parent can close.

	static ArsonProgress *findProgressWidget (QWidget *pw);

public slots:
	virtual void setCurrent (QWidget *was, QWidget *now) { }	/// for wizard use
	virtual void reconfigure (void) { }
	virtual void start_clicked (void);
	virtual void beginWrite (void);
	virtual void endWrite (void);
	void setText (const QString &text);
	void setMaxProgress (long maxp);
	void setProgress (long prog);
	void setProgressMode (int mode);

protected:
	ArsonProgress (QWidget *parent, const char *name);
	ArsonProgress (const ArsonProgress &) {}

	ArsonProcessMgr *writer (void) { return m_pWriter; }
	void enableScreenSaver (bool enable);
	void enableControls (bool enable);
	void _setCaption (void);

	virtual ArsonProcessMgr *createProcessMgr (void) = 0;
	virtual void processOpts (ArsonProcessOpts &opts) { }
	virtual void save_output_clicked (void);
	virtual bool cancelWarning (void);
	virtual void show_cmd_log (void);
	
	ArsonProcessMgr *m_pWriter;
	QGroupBox *m_pBtnGroup;
	QBoxLayout *m_pBtnLayout;
	WIDGETLIST m_widgets;
	QString m_baseCaption;
	pushButtonLogUI *m_pLogUI;
	ArsonProcessUI *m_pUI;
	bool m_bScreenSaver;

#ifdef ARSONDBG
	bool m_DEBUG_SETUP;
#endif

public:
	enum Status { Writing, Complete };
	
signals:
	void restoreState (ArsonCtrlState &state);
	void saveState (ArsonCtrlState &state);
	void enableUI (ArsonProgress::Status status);
	void setParentCaption (const QString &str);
};

/*========================================================*/
/**
 *	The parent of any ArsonProgress widget should create
 *	an instance of this class to handle the start/cancel/
 *	close logic.
 */

class QPushButton;

class ArsonProgressEventFilter : public QObject
{
	Q_OBJECT

public:
	ArsonProgressEventFilter(
		ArsonProgress *pp,
		QDialog *parent,
		QPushButton *start,
		QPushButton *cancel);

	virtual ~ArsonProgressEventFilter (void);

private:
	virtual bool eventFilter (QObject *po, QEvent *pe);

	ArsonProgress *m_progress;
	QPushButton *m_start;
	QPushButton *m_cancel;

protected slots:
	void slotEnableUI (ArsonProgress::Status status);
	void slotCancel (void);

signals:
	void complete (void);
};

/*========================================================*/
/**
 *	A standalone parent dialog for the ArsonProgress
 *	widget, provides Start, Configure, and Cancel
 *	buttons.
 */

class ArsonProgressDlg : public QDialog
{
	Q_OBJECT

public:
	ArsonProgressDlg (QWidget *parent, const char *name = NULL);

	void setProgressWidget (ArsonProgress *pd);

private:
	ArsonProgress *m_pProgressWidget;
	QVBoxLayout *m_pMainLayout;
	QPushButton *m_pConfig;
	QPushButton *m_pCancel;
	QPushButton *m_pStart;

private slots:
	void slotEnableUI (ArsonProgress::Status status);
	void config_clicked (void);
};

/*========================================================*/
/**
 *	A simple progress widget, message and caption
 *	can be set through the constructor.
 */

class ArsonSimpleProgress : public ArsonProgress
{
public:
	ArsonSimpleProgress(
		const QString &text,
		const QString &caption,
		QWidget *parent,
		const char *name);
};

/*========================================================*/
/**
 *	A control group to be inserted into ArsonProgress
 *	widgets for operations that use cdrdao to read a disk.
 */

class ArsonProcessOpts;

class ArsonCdrdaoCtrlGrp : public QWidget
{
public:
	ArsonCdrdaoCtrlGrp (QWidget *parent, const char *name = NULL);

	enum SubChan {
		None,
		RW,
		RW_Raw,
		SubChan_Count
	};

	void applyTo (ArsonProcessOpts &opts) const;
	SubChan subChan (void) const;
	bool readRaw (void) const;

private:
	QCheckBox *m_pRaw;
	QComboBox *m_pSubChan;
};

/*========================================================*/

namespace arson {
	template<class T> void showProgressDlg (void)
	{
		ArsonProgressDlg dlg (kapp->mainWidget());
		T *ptr = new T(&dlg);

		dlg.setProgressWidget(ptr);
		dlg.exec();
	}

	template<class T, class Args> void showProgressDlg (const Args &args)
	{
		ArsonProgressDlg dlg (kapp->mainWidget());
		T *ptr = new T(&dlg, args);

		dlg.setProgressWidget(ptr);
		dlg.exec();
	}
}

/*========================================================*/
#endif	/*	__PROGRESSDLG_H__	*/
