/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Various miscellaneous stuff
 *
 *	by Tony Sideris	(08:21PM Sep 08, 2001)
 *================================================*/
#include <qcheckbox.h>
#include <qlabel.h>

#include <kfiledialog.h>
#include <klocale.h>

#include "progressdlg.h"
#include "progressbar.h"
#include "tools.h"
#include "process.h"
#include "konfig.h"
#include "cdcopy.h"
#include "isofs.h"
#include "wizard.h"
#include "utils.h"

/*========================================================*/

bool isCueBin (const QString &file)
{
	const QString ext (QFileInfo(file).extension().lower());
	return (ext == "cue" || ext == "toc");
}

/*========================================================*/

ArsonReadCdMgr::ArsonReadCdMgr (ArsonProcessUI *pUI, const char *outfile)
	: ArsonProcessMgr(pUI),
	m_strOutFile(outfile)
{
}

/*========================================================*/

void ArsonReadCdMgr::begin (const ArsonProcessOpts &opts)
{
	ArsonProcessMgr::begin(opts);

	ui()->setText(
		i18n("Reading image from device %1 into file %2 ...")
		.arg(ACONFIG.device())
		.arg(m_strOutFile));

	try {
		if (isCueBin(m_strOutFile))
			setProcess(new ArsonCdrdaoReadCdProcess(this, m_strOutFile));
		else
			setProcess(new ArsonReadCdProcess(this, m_strOutFile));
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/

class isoFilePage : public ArsonWizardFilePage
{
public:
	isoFilePage (QWidget *parent, const char *name)
		: ArsonWizardFilePage(parent, name)
	{
		setMessage(
			i18n("Select image file to rip data to:"));
	}

private:
	virtual void fileNameFilter (ArsonFileFilter &filter) const
	{
		filter.addFilter(i18n("ISO Files"), "*.iso");
		filter.addFilter(i18n("CUE-BIN Files"), "*.cue *.toc");
	}

	virtual QString selectPath (void)
	{
		ArsonFileFilter filter;
		fileNameFilter(filter);
		return KFileDialog::getSaveFileName(QString::null, filter.toString());
	}

	virtual bool isValid (const QString &text) const
	{
		return !text.isEmpty();
	}
};

class readCdProgress : public ArsonSimpleProgress
{
public:
	readCdProgress (QWidget *parent)
		: ArsonSimpleProgress(
			i18n("Please insert the CD you wish to read and press Start..."),
			i18n("Readcd Progress"), parent, "readcd")
	{
		m_pCdrdao = new ArsonCdrdaoCtrlGrp(ctrlParent());
		widgetMgr() << m_pCdrdao;
	}

private:
	virtual void setCurrent (QWidget *was, QWidget *now)
	{
		isoFilePage *pi = (isoFilePage *) was;

		if (was && now == this)
		{
			m_outfile = pi->path().latin1();

			m_pCdrdao->setEnabled(
				isCueBin(pi->path()));
		}

		ArsonSimpleProgress::setCurrent(was, now);
	}

	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		if (!m_outfile.isEmpty())
			return new ArsonReadCdMgr(ui(), m_outfile);

		return NULL;
	}

	virtual void processOpts (ArsonProcessOpts &opts)
	{
		if (isCueBin(m_outfile))
			m_pCdrdao->applyTo(opts);

		ArsonSimpleProgress::processOpts(opts);
	}

	ArsonCdrdaoCtrlGrp *m_pCdrdao;
	QCString m_outfile;
};

/*========================================================*/

class readCdWizardFactory : public ArsonWizardFactory
{
public:
	readCdWizardFactory (void)
		: ArsonWizardFactory(ARSON_WIZARD_READCD,
			I18N_NOOP("Rip Data CD to Image File"),
			SecondaryAction) { }

	virtual void addPages (ArsonWizard *pw)
	{
		isoFilePage *pi = new isoFilePage(pw, "isopage");
		readCdProgress *pd = new readCdProgress(pw);

		QObject::connect(pw, SIGNAL(pageChange(QWidget*,QWidget*)),
			pd, SLOT(setCurrent(QWidget*,QWidget*)));

		pw->addPage(pi, i18n("Select Image File"));
		pw->addPage(pd, i18n("Read CD Progress"));
	}
};

readCdWizardFactory theReadCdFactory;

void arsonReadCd (void)
{
	ArsonWizard::showWizard(ARSON_WIZARD_READCD);
}

/*========================================================*/
/*	Unlock the drive
 *========================================================*/

class unlockMgr : public ArsonProcessMgr
{
public:
	unlockMgr (ArsonProcessUI *pUI)
		: ArsonProcessMgr(pUI)
	{
		//	Nothing...
	}

	virtual void begin (const ArsonProcessOpts &opts)
	{
		ArsonProcessMgr::begin(opts);

		try {
			setProcess(
				new ArsonCdrdaoUnlockProcess(this));
		}
		catch (ArsonError &err) {
			err.report();
		}
	}
};

class unlockDlg : public ArsonSimpleProgress
{
public:
	unlockDlg (QWidget *parent)
		: ArsonSimpleProgress(
			i18n("Press Start to unlock drive..."),
			i18n("Drive Unlock Progress"),
			parent, "unlockdlg")
	{
	}

private:
	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		setProgressMode(arsonProgressInfinite);

		return new unlockMgr(ui());
	}

	virtual void endWrite (void)
	{
		setProgressMode(arsonProgressBlank);
		ArsonSimpleProgress::endWrite();
	}
};

void arsonUnlockDrive (void)
{
	arson::showProgressDlg<unlockDlg>();
}

/*========================================================*/
/*	CDRW blanking
 *========================================================*/

class blankMgr : public ArsonProcessMgr
{
public:
	blankMgr (ArsonProcessUI *pUI)
		: ArsonProcessMgr(pUI) { }

	virtual void begin (const ArsonProcessOpts &opts)
	{
		ArsonProcessMgr::begin(opts);

		try {
			ArsonProcess *ptr;
			const long how = opts.getLong("how");

			if (ACONFIG.programPref(PROGGRP_BLANK) == "cdrdao")
				ptr = new ArsonCdrdaoBlankProcess(this, how);
			else
				ptr = new ArsonCdrecordBlankProcess(this, how);

			setProcess(ptr);
		}
		catch (ArsonError &err) {
			err.report();
		}
	}
};

class blankDlg : public ArsonSimpleProgress
{
public:
	blankDlg (QWidget *parent)
		: ArsonSimpleProgress(
			i18n("Insert a CDRW, and press Start to blank the disk..."),
			i18n("CDRW Blank Progress"),
			parent, "blankdlg")
	{
		const static QString modes[] = {
			i18n("Full"),
			i18n("Fast"),
		};
		QLabel *pl = new QLabel(i18n("Blank Mode: "), ctrlParent());
		m_pMode = new ArsonProgressCombobox(ctrlParent(), "blankmode");
		pl->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Fixed));

		for (int index = 0; index < ARRSIZE(modes); ++index)
			m_pMode->insertItem(modes[index]);

		m_pMode->setCurrentItem(0);


		layoutRow() << pl << m_pMode;
	}

private:
	virtual void processOpts (ArsonProcessOpts &opts)
	{
		opts.addLong("how", m_pMode->currentItem());
		ArsonSimpleProgress::processOpts(opts);
	}

	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		return new blankMgr(ui());
	}

	QComboBox *m_pMode;
};


void arsonBlankCdrw (void)
{
	arson::showProgressDlg<blankDlg>();
}

/*========================================================*/
/*	Fixate a disk
 *========================================================*/

class fixateMgr : public ArsonProcessMgr
{
public:
	fixateMgr (ArsonProcessUI *pUI)
		: ArsonProcessMgr(pUI) { }

	virtual void begin (const ArsonProcessOpts &opts)
	{
		ArsonProcessMgr::begin(opts);

		try {
			setProcess(
				new ArsonCdrecordFixateProcess(this)
				);
		}
		catch (ArsonError &err) {
			err.report();
		}
	}
};

class fixateDlg : public ArsonSimpleProgress
{
public:
	fixateDlg (QWidget *parent)
		: ArsonSimpleProgress(
			i18n("Press Start to fixate the disk currently in the drive..."),
			i18n("Fixate Progress"),
			parent, "fixatedlg")
	{
	}

private:
	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		return new fixateMgr(ui());
	}
};


void arsonFixate (void)
{
	arson::showProgressDlg<fixateDlg>();
}

/*========================================================*/
/*	Image writing
 *========================================================*/

class cueDlg : public ArsonCdWriterProgress
{
public:
	cueDlg (QWidget *parent, const char *cuefile)
		: ArsonCdWriterProgress(parent, "cuedlg"),
		m_cuefile(cuefile) {}

private:
	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		return new ArsonCueWriter(ui(), m_cuefile);
	}

	QCString m_cuefile;
};

class isoDlg : public ArsonCdWriterProgress
{
public:
	isoDlg (QWidget *parent, const char *isofile)
		: ArsonCdWriterProgress(parent, "isodlg"),
		m_isofile(isofile) {}

private:
	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		return new ArsonIsoWriter(ui(), m_isofile);
	}

	QCString m_isofile;
};

class imgWriterProgress : public ArsonCdWriterProgress
{
public:
	imgWriterProgress (ArsonWizardFilePage *pp, QWidget *parent, const char *name)
		: ArsonCdWriterProgress(parent, name), m_imgDlg(pp) { }

	imgWriterProgress (QWidget *parent, const char *filename)
		: ArsonCdWriterProgress(parent, "imgdlg"), m_imgDlg(NULL)
	{
		setImgFile(filename);
	}

	void setImgFile (const char *file) { m_imgfile = file; }

private:
	virtual void setCurrent (QWidget *was, QWidget *now)
	{
		ArsonCdWriterProgress::setCurrent(was, now);

		if (m_imgDlg)
			setImgFile(m_imgDlg->path());
	}

	virtual ArsonProcessMgr *createProcessMgr (void)
	{
		ArsonCdImageFile im (m_imgfile);
		const int type = im.imgFormat();

		Trace("creating processmgr w/ file %s\n", m_imgfile.data());
		switch (type)
		{
		case ArsonCdImageFile::Iso:
		case ArsonCdImageFile::Cif:
			return new ArsonIsoWriter(ui(), m_imgfile);

		case ArsonCdImageFile::Cue:
		case ArsonCdImageFile::Toc:
			return new ArsonCueWriter(ui(), m_imgfile);
		}

		return NULL;
	}

	ArsonWizardFilePage *m_imgDlg;
	QCString m_imgfile;
};

/*========================================================*/

class imgFilePage : public ArsonWizardFilePage
{
public:
	imgFilePage (QWidget *parent, const char *name)
		: ArsonWizardFilePage(parent, name) { }

	virtual void fileNameFilter (ArsonFileFilter &filter)
	{
		filter = ArsonFileFilter(
			ArsonCdImageFile::exts, ArsonCdImageFile::Max);
	}
};

/*========================================================*/

class imgWizardFactory : public ArsonWizardFactory
{
public:
	imgWizardFactory (void)
		: ArsonWizardFactory(ARSON_WIZARD_IMAGEFILE,
			I18N_NOOP("Burn an Image File"),
			SecondaryAction) { }

	virtual void addPages (ArsonWizard *pw)
	{
		imgFilePage *pi = new imgFilePage(pw, "imgpage");
		imgWriterProgress *pd = new imgWriterProgress(pi, pw, NULL);

		QObject::connect(pw, SIGNAL(pageChange(QWidget*,QWidget*)),
			pd, SLOT(setCurrent(QWidget*,QWidget*)));

		pw->addPage(pi, i18n("Select Image File"));
		pw->addPage(pd, i18n("Image Burn Progress"));
	}
};

imgWizardFactory theImgWizardFactory;

/*========================================================*/
/*	Worker image writer functions
 *========================================================*/

void arsonWriteCueFile (const char *cuefile)
{
	arson::showProgressDlg<imgWriterProgress>(cuefile);
}

void arsonWriteIsoFile (const char *isofile)
{
	arson::showProgressDlg<imgWriterProgress>(isofile);
}

/*========================================================*/
/*	Write a single existing directory tree
 *========================================================*/

class dirProgress : public ArsonIsoWriterProgress
{
public:
	dirProgress (QWidget *parent)
		: ArsonIsoWriterProgress(parent) { }

	virtual void setCurrent (QWidget *was, QWidget *now)
	{
		ArsonWizardDirPage *pd = (ArsonWizardDirPage *) was;

		if (pd && now == this)
			setDirs(QStringList(pd->path()));
	}
};

class dirWizardFactory : public ArsonWizardFactory
{
public:
	dirWizardFactory (void)
		: ArsonWizardFactory(
			ARSON_WIZARD_DIRECTORY,
			I18N_NOOP("Burn a Single Directory Tree"),
			ToolAction)
	{ }

	virtual void addPages (ArsonWizard *pw)
	{
		ArsonWizardDirPage *pd = new ArsonWizardDirPage(pw, "dir");
		ArsonIsoWriterProgress *pp = new dirProgress(pw);

		QObject::connect(pw, SIGNAL(pageChange(QWidget*,QWidget*)),
			pp, SLOT(setCurrent(QWidget*,QWidget*)));
		
		pw->addPage(pd, i18n("Select Directory"));
		pw->addPage(pp, i18n("Progress"));
	}
};

dirWizardFactory theDirWizardFactory;

void arsonWriteDir (const QString &indir)
{
	try
	{
		if (indir != QString::null)
			arson::showProgressDlg<ArsonIsoWriterProgress>(QStringList(indir));
		else
			ArsonWizard::showWizard(ARSON_WIZARD_DIRECTORY);
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/
/*	Class for handling image files
 *========================================================*/

const QString ArsonCdImageFile::exts[ArsonCdImageFile::Max] = {
	I18N_NOOP("*.iso|ISO Files"),
	I18N_NOOP("*.cif|CIF Files"),
	I18N_NOOP("*.cue|CUE Files"),
	I18N_NOOP("*.toc|TOC Files"),
};

ArsonCdImageFile::ArsonCdImageFile (const QString &filename)
	: m_file(filename), m_type(Unknown)
{ }

int ArsonCdImageFile::imgFormat (void)
{
	if (m_type == Unknown)
	{
		QFileInfo fi (QFile::encodeName(filename()));
		const QString ext = fi.extension(false).lower();

		if (fi.isDir())
			return (m_type = Dir);

		for (int index = 0; index < Max; ++index)
		{
			const int pipe = exts[index].find('|');

			if (exts[index].mid(2, pipe - 2) == ext)
				return (m_type = index);
		}
	}

	return m_type;
}

bool ArsonCdImageFile::write (void)
{
	switch (imgFormat())
	{
	case Iso:
	case Cif:
		arsonWriteIsoFile(filename());
		break;

	case Toc:
	case Cue:
		arsonWriteCueFile(filename());
		break;

	case Dir:
		arsonWriteDir(filename());
		break;

	default:
		return false;
	}

	return true;
}

/*========================================================*/
/*	Write any known image file format
 *========================================================*/

void arsonWriteImg (const char *fn)
{
	if (fn)
	{
		ArsonCdImageFile img (fn);

		if (img.imgFormat() == ArsonCdImageFile::Unknown)
			arsonErrorMsg(
				i18n("Unknown image file format."));
		else
			img.write();
	}
	else
		ArsonWizard::showWizard(ARSON_WIZARD_IMAGEFILE);
}

/*========================================================*/
/*	CD Copy function
 *========================================================*/

void arsonCdCopy (void)
{
	ArsonWizard::showWizard(ARSON_WIZARD_CDCOPY);
}

/*========================================================*/
