/* $Id: TerrainView.cpp,v 1.56 2003/03/20 19:39:51 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the ArkRPG Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef WIN32
#include <windows.h>
#endif

#include <sstream>

#include <math.h>

#include <Client/GLClient.h>
#include <Client/TerrainView.h>
#include <Client/MessageView.h>


namespace Client
{

	class CameraControl
	{
	 	Ark::Camera m_Camera;
	 	scalar m_RotX, m_RotXSpeed;
	 	scalar m_RotY, m_RotYSpeed;
	 	scalar m_Distance;

		Ark::Quaternion m_TargetRotation;
		bool m_LockCharacter;

	public:
		CameraControl()
		{
			m_Camera.m_PointOfView = Ark::Vector3();
			m_Camera.m_LookAt = Ark::Vector3(128.f, 5.f, 128.f);
			m_Camera.m_FOV = 60.0f;

			m_RotXSpeed = 0.f;
			m_RotYSpeed = 0.f;
			m_RotX = 20.f;
			m_RotY = -105.f;
			m_Distance = 8.f;

			m_LockCharacter = true;
		}

		// return true if handled
		bool HandleMouseButton(bool down, int b, int x, int y)
		{
			if (!down)
				return false;
			
			if (b == 4) 
			{
				m_Distance *= 1.1f;
				return true;
			}
			
			if (b == 5) 
			{
				m_Distance /= 1.1f;
				return true;
			}

			return false;
		}

		bool HandleMouseMotion (int state, int x, int y)
		{
			return false;
		}

		bool HandleKeyPress (bool down, int mods, int key)
		{
			if (down)
			{
				switch (key)
				{
					case SDLK_PAGEUP:
					{
						m_RotXSpeed =  60.f;
						return true;
					}
					case SDLK_PAGEDOWN:
					{
						m_RotXSpeed = -60.f;
						return true;
					}
				}  
			}
			else
			{
				switch (key)
				{
					// Lock camera on character
					case SDLK_BACKQUOTE:
					// case SDLK_BACKSPACE:
					case SDLK_TAB:
					{
						m_LockCharacter = !m_LockCharacter;
						return true;
					}
					case SDLK_PAGEUP:
					{
						m_RotXSpeed = 0.f;
						return true;
					}
					case SDLK_PAGEDOWN:
					{
						m_RotXSpeed = 0.f;
						return true;
					}
				}  
			}

			return false;
		}

		inline void SetRotationXSpeed(scalar s) { m_RotXSpeed = s; }
		inline void SetRotationYSpeed(scalar s) { m_RotYSpeed = s; }
		inline void SetDistance(scalar d) { m_Distance = d; }
		inline void SetTargetRotation(const Ark::Quaternion& q) 
		{ m_TargetRotation = q; }

		// Constant reference, take a copy if you want to modify state
		const Ark::Camera& Update(const Ark::ModelState* state, scalar deltaTime)
		{
			m_RotY += m_RotYSpeed * deltaTime;
			m_RotX += m_RotXSpeed * deltaTime;

			/// Clamp scale..
			m_Distance = std::min (32.0f,  std::max (m_Distance, 4.0f));

			/// Clamp X rotation
			m_RotX = std::min (89.0f, std::max(m_RotX, -10.0f));

			/// =============================================================
			/// Compute look at position...
			Ark::Vector3 lookAt;
			if (state)
			{
				const Ark::Vector3& position = state->m_Position;
				Ark::BBox bb = state->ExtractBbox();
				lookAt.X = position.X;
				lookAt.Y = position.Y + (bb.m_Max.Y - bb.m_Min.Y)/2.0f;
				lookAt.Z = position.Z;
			}
			else
			{
				lookAt = Ark::Vector3(128.0f, 5.0f, 128.0f);
			}

			/// Compute point of view..
			Ark::Vector3 pointofview;
			Ark::Matrix44 m;

			if (state && m_LockCharacter)
			{
				pointofview = Ark::Vector3(-1.0f, 0.0f, 0.0f);      
				m.MakeRotationZ(-m_RotX);
				pointofview = m.Transform (pointofview);
				m_TargetRotation = state->m_Rotation;
				pointofview = m_TargetRotation.Transform(pointofview);
			}
			else
			{
				pointofview = Ark::Vector3(0.0f, 0.0f, -1.0f);      
				m.MakeRotationX(m_RotX);
				pointofview = m.Transform (pointofview);
				m.MakeRotationY(m_RotY);
				pointofview = m.Transform (pointofview);
			}

			// Sets distance and move to look at
			// Note: this is only valid for a trackball camera
			pointofview.Scale(m_Distance);
			pointofview += lookAt;

			// Update camera
			Ark::CameraInstantChange l(lookAt);
			Ark::CameraInstantChange p(pointofview);
			m_Camera.Move(l, p);

			return m_Camera;
		}
	};
	
   TerrainView::TerrainView (UIRenderer *ui, Client *client)
   {
	   m_Camera = new CameraControl();

	   m_Client = client;
	   m_LastMousePositionCheck = false;
	   m_PlayerRotYSpeed = 0.0;
	   m_PlayerRotY = -105.0;

	   m_PlayerSpeed = 0.0;
	   m_FrameCounter = 0;
	   m_FPS = 0;
	   m_ScreenX = 0.5f;
	   m_ScreenY = 0.5f;

	   m_MessageView = new MessageView(ui);
   }

   TerrainView:: ~TerrainView ()
   {
      delete m_MessageView;
      delete m_Camera;
   }
      
   Ark::Vector3 
   UnProject (scalar sX, scalar sY, scalar mz,
	      Ark::Matrix44 invmat)
   {
      Ark::Vector3 R = invmat.Transform(Ark::Vector3 (sX, sY, mz));
      scalar w = (sX * invmat.M(0,3) +
		  sY * invmat.M(1,3) +
		  mz * invmat.M(2,3) +
		  invmat.M(3,3));

      R.Scale (1.0f/w);
      return R;
   }

   void
   TerrainView::ComputeScreenCoordinates()
   {
      GLClient *client = (GLClient*) m_Client;

      int mx, my;
      SDL_GetMouseState (&mx, &my);

      if (mx < 0 || my < 0 || mx > client->m_Width || my > client->m_Height)
		  return;
      
      // Invert y, because of OGL coord system
	  m_ScreenX = scalar(mx) / client->m_Width;
	  m_ScreenY = 1.f - scalar(my) / client->m_Height;
   }
   
   void
   TerrainView::TraceMousePosition(Ark::Renderer& renderer, const Ark::Vector3 &pov)
   {
	   m_LastMousePositionCheck = false;

	   //Ark::Matrix44 projmat, modelmat;
	   //glGetFloatv (GL_PROJECTION_MATRIX, &projmat.M(0,0));
	   //glGetFloatv (GL_MODELVIEW_MATRIX, &modelmat.M(0,0));

	   Ark::Matrix44 modelmat(renderer.GetViewMatrix());
	   modelmat.Multiply(renderer.GetProjectionMatrix());
	   modelmat.Invert();

	   const scalar sX = 2.0f * m_ScreenX - 1.0f;
	   const scalar sY = 2.0f * m_ScreenY - 1.0f;
	   Ark::Vector3 lookat = UnProject (sX, sY, 1.0f, modelmat);

	   Ark::Vector3 dir = lookat - pov;
	   dir.Scale (10000.0f);

	   lookat = pov + dir;
	   std::vector<Ark::Collision> collisions;
	   m_Client->GetWorld()->RayTrace (Ark::Ray(pov, lookat),
			   Ark::Collision::POSITION |
			   Ark::Collision::ENTITY |
			   Ark::Collision::WORLD,
			   collisions);

	   if (!collisions.empty())
	   {
		   m_LastHit = collisions[0];
		   m_LastMousePosition = m_LastHit.m_Pos;
		   m_LastMousePositionCheck = true;
	   }
   }
   
   void
   TerrainView::Render (UIRenderer *ui)
   {
      GLClient *client = (GLClient*) m_Client;
      
	  Ark::Renderer& renderer = *ui->Rdr();
      renderer.SetViewport (0, 0, client->m_Width, client->m_Height);

      const scalar dt = m_Timer.GetDelta();
      m_Timer.Update();
      
      m_PlayerRotY += m_PlayerRotYSpeed * dt;


	  ComputeScreenCoordinates();

	  // FIXME should be done in camera control
	  scalar rotationYspeed = 0.f;
	  const scalar xMinThreshold = 0.1f;
	  const scalar xMaxThreshold = 1.f - xMinThreshold;
      if (m_ScreenX < xMinThreshold) 
	  {
		  rotationYspeed = -100.f * (1.f - m_ScreenX/xMinThreshold);
	  }
      else if (xMaxThreshold < m_ScreenX)
	  {
		  rotationYspeed = 100.f * (m_ScreenX-xMaxThreshold)/(1.f-xMaxThreshold);
	  }
      m_Camera->SetRotationYSpeed(rotationYspeed);

	  // Move player 
	  Ark::ModelState* modelState = 0;
      if (m_Client->GetPlayer())
	  {
		  m_Client->GetPlayer()->m_Flags &= ~Ark::Entity::PATHFINDING; // TRO

		  modelState = &m_Client->GetPlayer()->m_MState;
		  Ark::Vector3& position = modelState->m_Position;

		  Ark::Quaternion q(0.0, m_PlayerRotY, 0.0);
		  modelState->m_Rotation = q;

		  position += q.Transform(Ark::Vector3(m_PlayerSpeed*dt, 0.0, 0.0));
	  }
      
	  // Update camera
	  Ark::Camera camera = m_Camera->Update(modelState, dt);

      if (m_Client->GetWorld())
	  {
		  //Ark::Timer timer;
		  m_Client->GetWorld()->Render(renderer, camera);
		  TraceMousePosition(renderer, camera.m_PointOfView);
	  }
	  else
	  {
		  m_LastMousePositionCheck = false;
	  }

      //-------------- FPS COUNTER
      renderer.SetIdentity();

      Ark::FontPtr fp = ui->GetFont ("arial1-1-ffffff");

      Ark::String str;
      if (m_FrameCounter == 30)
	  {
		  m_FrameCounter = 0;
		  m_FPS = 30.0f / m_FrameTimer.GetDelta();
		  m_FrameTimer.Update();
	  }
      else
		  m_FrameCounter++;

      {
	  std::ostringstream os;
	  os << (int)m_FPS << " fps";
	  str = os.str();
	  	  
      }

      int ypos = client->m_Height - ui->GetStringHeight (fp.Get(), str);
      int xpos = 5;
      ui->DrawString (fp.Get(), str, xpos, ypos);
      
      {
          std::ostringstream os;
          os << "x: "   << (int) m_Client->GetPlayer()->m_MState.m_Position.X;
	  os << "  y: " << (int) m_Client->GetPlayer()->m_MState.m_Position.Y;
	  os << "  z: " << (int) m_Client->GetPlayer()->m_MState.m_Position.Z;
          str = os.str();
      }
      xpos = client->m_Width - ui->GetStringWidth (fp.Get(), str) - 5;
      ui->DrawString (fp.Get(), str, xpos, ypos);
      
      if (m_LastHit.m_Flags & Ark::Collision::ENTITY)
      {
	 ypos -= ui->GetStringHeight (fp.Get(), m_LastHit.m_Entity->m_Name);
	 ui->DrawString (fp.Get(), m_LastHit.m_Entity->m_Name, 5, ypos);
      }

      if (m_Client->GetPlayer())
	 m_MessageView->Render (ui, m_Client->GetPlayer());	 
   }

   void
   TerrainView::HandleMButton (bool down, int b, int x, int y)
   {
      if (m_MessageView->HandleMButton(down, b, x, y))
		  return;

	  if (m_Camera->HandleMouseButton(down, b, x, y))
		  return;

      if (down && b == 1 && m_LastMousePositionCheck)
      {
	 if (m_LastHit.m_Flags & Ark::Collision::WORLD)
	 {
	    //m_Client->GetPlayer()->SetGoal(m_LastMousePosition);
	    
	    // FIXME: hack !!!
	    //m_Client->GetPlayer()->m_MState.Play
	    //   (Ark::ModelState::LOOP, "walk");
	 }
	 else if (m_LastHit.m_Flags & Ark::Collision::ENTITY)
	 {
	    if (m_LastHit.m_Entity != m_Client->GetPlayer())
	       m_Client->GetPlayer()->AddMessage
		  (m_LastHit.m_Entity, "MOUSEHIT");
	    else
	    {
	       // display an inventory, or something
	    }
	 }
      }
   }

   void
   TerrainView::HandleKey (bool down, int mods, int key)
   {
	   if (m_Camera->HandleKeyPress(down, mods, key))
		   return;

      if (down)
      {
	 switch (key)
	 {
	    case SDLK_LEFT:
	       m_PlayerRotYSpeed = 90.0;
	       if (m_PlayerSpeed == 0.0) // not walking
		  m_Client->GetPlayer()->m_MState.Play
		     (Ark::ModelState::LOOP, "rotate-left");
	       break;

	    case SDLK_RIGHT:
	       m_PlayerRotYSpeed = -90.0;
	       if (m_PlayerSpeed == 0.0) // not walking
		  m_Client->GetPlayer()->m_MState.Play
		     (Ark::ModelState::LOOP, "rotate-right");
	       break;

	    case SDLK_UP:
	       m_Client->GetPlayer()->m_MState.Play
		  (Ark::ModelState::LOOP, "walk");
	       m_PlayerSpeed = 1.5;
	       break;

	    case SDLK_F11:
	       m_Client->GetCache()->Dump();
	    	    
	 }  
      }
      else
      {
	 switch (key)
	 {
	    case SDLK_LEFT:
	    case SDLK_RIGHT:
	       if (m_PlayerSpeed == 0.0)
		  m_Client->GetPlayer()->m_MState.Play
		     (Ark::ModelState::LOOP, "idle");
	       m_PlayerRotYSpeed = 0.0;
	       break;

	    case SDLK_UP:
	       m_Client->GetPlayer()->m_MState.Play
		  (Ark::ModelState::LOOP, "idle");
	       m_PlayerSpeed = 0.0;
	       break;
	 }
      }
   }

   void
   TerrainView::HandleMotion (int state, int x, int y)
   {
	   // do not check return value because we do nothing here
	   m_Camera->HandleMouseMotion(state,x,y);
   }
}
